/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  main.cpp - MIC main file
 *
 *  Version: $Id: main.cpp 2593 2005-09-19 17:14:32Z orenk $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <list>
#include <map>

#include "CrSpace.h"
#include "ParamList.h"
#include "Param.h"
#include "Image.h"
#include "TImage.h"
#include "MultiFile.h"
#include "ConfFile.h"
#include "compatibility.h"

#define numb_el(x) (sizeof(x)/sizeof(x[0]))
namespace std {}; using namespace std;

#ifndef BLD_VER
#define BLD_VER Devel
#endif

#ifndef PROJ_VER
#define PROJ_VER MFT
#endif

#define __VFSTR(x) 			#x
#define _VFSTR(x) 			__VFSTR(x)

extern "C" const char *g_version   = "1.1.0";
const char *g_build_ver = _VFSTR(BLD_VER);
const char *g_proj_ver  = _VFSTR(PROJ_VER);
const char *g_svn_rev   = "$Revision: 2752 $";


// Supported FW-PREP Version 
const char *g_supported_prep_version = "1.0.0"; 

// Unset these environment variables!!!
const char *env[] = { "LANGUAGE", "LANG", "LC_ALL", "LC_COLLATE", "LC_CTYPE",
                      "LC_MESSAGES", "LC_MONETARY", "LC_NUMERIC", "LC_TIME"
};

////////////////////////////////////////////////////////////////////////
#define USAGE do { usage(av[0]); return 1; } while(0)
void usage(const char *sname)
{
    const char *descr =
        "\n"
        "          InfiniScale-III EEPROM management/burning tool.\n"
        "          -----------------------------------------------\n"
        "\n"
        "Usage:\n"
        "------\n"
        "\n"
        "        %s [SWITCHES]\n"
        "\n"
        "Short switches summary:\n"
        "        -help\n"
        "             Print this mesage\n"
        "        -nowarn\n"
        "             Do not print some warnings\n"
        "        -nofs\n"
        "             Resulting image will not be failsafe\n"
        "        -version\n"
        "             Print version number\n"
        "        -dev_type <Device type>\n"
	    "             The device for which the image is generated.\n"
	    "             Supported devices: MT47396 , MT23108, MT25208, MT25218, MT24204, MT25204\n"
        "        -fw          <FW image filename>\n"
        "             Specify input file with the FW\n"
        "        -conf        <input config file filename>\n"
        "             Read configuration (.INI) file\n"
        "        -wrconf      <output config file filename>\n"
        "             Write configuration (.INI) file\n"
        "        -wrimage     <output EEPROM image filename>\n"
        "             Write resulting image to file.\n"
        "        -format      <output EEPROM image format>\n"
        "             Output image format. May be IMAGE, DWORDS or BINARY.\n"
        "             Default is IMAGE.\n"
        "        -fwver\n"
        "             Print version of the given fw and exit\n"
        "        -exp_rom <expansion rom file>\n"
        "             Integrate the given expention file (in .img or .bin format) to the\n"
        "             firmware image (HCA's only).\n"
        "        -GROUP.PARAM <Value>\n"
        "             Set value to specified parameter.\n"
        "";
    printf(descr, sname);
}

////////////////////////////////////////////////////////////////////////
bool swcmp(const char *par, const char *sw)
{
    char *s = (char *)sw;
    char *p = (char *)par;

    while (*s)
        if (*p++ != *s++)
            return *(p-1) == 0 ? true : (*(p-1) == '=');
    return *p == 0 ? true : (*p == '=');
} // swcmp



#define SETERR(args) do { printf args; return 1; } while(0)

#define SWVAL(s, p) do {                                     \
    char *_eq_p = strchr(av[i], '=');                        \
    if (_eq_p)                                               \
    {                                                        \
        *_eq_p = 0;                                          \
        p = ++_eq_p;                                         \
    }                                                        \
    else if (++i >= ac)                                      \
        SETERR(("Missed parameter after \"%s\" switch\n", s)); \
    else                                                     \
        p = av[i];                                           \
    } while(0)
////////////////////////////////////////////////////////////////////////
int COMP_CDECL main(int ac, char *av[])
{
    // String-string pair type
    typedef pair<string,string>  ss_t;

    // Switch values and flags
    char         *perl_replace = 0;
    char         *fw_image = 0;
    char         *wr_conf = 0;
    char         *EEPROM_image = 0;
    char         *EEPROM_format = 0;
    char         *exp_rom_file  = 0;
    char         *user_data_file = 0;

    list<string> rd_conf;
    list<ss_t>   cl_params;
    bool         warn = true;
    bool         fw_ver = false;
    bool         fs_image = true;
    bool         add_conf_sect = true;
    bool         add_info_sect = true;
    bool         print_refnames = false;
    bool         exclude_conf_defaults = false;
    DeviceType   dev_type = MT47396;

    if (ac < 2)
        USAGE;

    /*
     * Drop all uneeded environment
     */
    for (unsigned i=0; i<numb_el(env); i++)
        unsetenv(env[i]);

    /*
     * Go thru command line options
     */
    for (int i=1; i < ac; i++)
    {
        /*
         * Switches
         * --------
         */
        if (*av[i] == '-')
        {
            char *swval;            // Switch value
            char *orig_sw = av[i];  // Original switch

            // Skip all "-" in prefix
            while (*av[i] == '-')
                ++av[i];

            // FW image
            if (swcmp(av[i], "fw"))
            {
                SWVAL("-fw", swval);
                if (fw_image)
                    SETERR(("FW image file may be specified onyl once.\n"
                           "New FW image file: \"%s\", "
                           "Old FW image file: \"%s\"\n",
                           swval, fw_image));
                fw_image = swval;
            }

            // read conf. file
            else if (swcmp(av[i], "conf"))
            {
                SWVAL("-conf", swval);
                rd_conf.push_back(swval);
            }

            // write image
            else if (swcmp(av[i], "wrimage"))
            {
                SWVAL("-wrimage", swval);
                if (EEPROM_image)
                    SETERR(("Image output file may be specified only once.\n"
                           "New image file: \"%s\", "
                           "Old image file: \"%s\"\n",
                           swval, EEPROM_image));
                EEPROM_image = swval;
            }

            // write image format
            else if (swcmp(av[i], "format"))
            {
                SWVAL("-format", swval);
                if (EEPROM_format)
                    SETERR(("Output image format may be specified only once.\n"
                           "New image format: \"%s\", "
                           "Old image format: \"%s\"\n",
                           swval, EEPROM_format));
                EEPROM_format = swval;

            }

            // expansion rom
            else if (swcmp(av[i], "exp_rom"))
            {
                SWVAL("-exp_rom", swval);
                if (exp_rom_file)
                    SETERR(("Expansion rom file may be specified only once.\n"
                           "New rom file: \"%s\", "
                           "Old rom file: \"%s\"\n",
                           swval, exp_rom_file));
                exp_rom_file = swval;

            }

            // expansion rom
            else if (swcmp(av[i], "user_data"))
            {
                SWVAL("-user_data", swval);
                if (user_data_file)
                    SETERR(("User data file may be specified only once.\n"
                           "New file: \"%s\", "
                           "Old file: \"%s\"\n",
                           swval, user_data_file));
                user_data_file = swval;

            }

            // write conf. file
            else if (swcmp(av[i], "wrconf"))
            {
                SWVAL("-wrconf", swval);
                if (wr_conf)
                    SETERR(("Output configuration file may be specified only once.\n"
                           "New conf.file name: \"%s\", "
                           "Old conf.file name: \"%s\"\n",
                           swval, wr_conf));
                wr_conf = swval;
            }

            // write conf. file without defaults
            else if (swcmp(av[i], "wrconf_no_def"))
            {
                SWVAL("-wrconf_no_def", swval);
                if (wr_conf)
                    SETERR(("Output configuration file may be specified only once.\n"
                           "New conf.file name: \"%s\", "
                           "Old conf.file name: \"%s\"\n",
                           swval, wr_conf));
                wr_conf = swval;
                exclude_conf_defaults = true;
            }


            // perl replacement
            else if (swcmp(av[i], "perl"))
            {
                SWVAL("-perl", swval);
                if (perl_replace)
                    SETERR(("Perl replacement file may be specified only once.\n"
                           "New perl replacement file name: \"%s\", "
                           "Old perl replacement file name: \"%s\"\n",
                           swval, perl_replace));
                perl_replace = swval;
            }

            // version
            else if (swcmp(av[i], "version"))
            {
                printf("mic %s, %s, BUILD %s\n", g_version, g_proj_ver, g_build_ver);
                return 0;
            }

            else if (swcmp(av[i], "vv"))
            {
                printf("mic %s, %s, BUILD %s, SVN %s\n", g_version, g_proj_ver, g_build_ver, g_svn_rev + 1);
                return 0;
            }

            // No FailSafe
            else if (swcmp(av[i], "nofs"))
            {
                fs_image = false;
            }

            // No configuration section
            else if (swcmp(av[i], "no_conf_sect"))
            {
                add_conf_sect = false;
            }

            // No configuration section
            else if (swcmp(av[i], "no_info_sect"))
            {
                add_info_sect = false;
            }

            // Print Refnames in ini file (Internal hack)
            else if (swcmp(av[i], "print_refnames"))
            {
                print_refnames = true;
            }

            // Print Fw Version
            else if (swcmp(av[i], "fwver"))
            {
                fw_ver = true;
            }
            // No warnings
            else if (swcmp(av[i], "nowarnings"))
            {
                warn = false;
            }

 
            // Device type
            else if (swcmp(av[i], "dev") || swcmp(av[i], "dev_type"))
            {
                SWVAL("-dev", swval);
                string sdev  = swval;
                if      (sdev == "MT47396")
                    dev_type    = MT47396;
                else if (sdev == "MT23108")
                    dev_type    = MT23108;
                else if (sdev == "MT25208")
                    dev_type    = MT25208;
                else if (sdev == "MT25218")
                    dev_type    = MT25218;
                else if (sdev == "MT24204")
                    dev_type    = MT24204;
                else if (sdev == "MT25204")
                    dev_type    = MT25204;
                else
                    SETERR(("Unsupported device type: %s\n", sdev.c_str()));
            }
                       
            // help
            else if (swcmp(av[i], "help"))
            {
                usage(av[0]);
                return 0;
            }

            // Parameters
            else if (strchr(av[i], '.'))
            {
                char *pname = av[i];
                SWVAL(av[i], swval);
                cl_params.push_back(ss_t(pname, swval));
            }

            /*
             * All other switches
             * ------------------
             */
            else
                SETERR(("Invalid switch \"%s\" is specified.\n", orig_sw));
        }

        else
            SETERR(("Unexpected parameter \"%s\" is specified.\n", av[i]));
    }

    /*
     * Now interpret all switches...
     * ----------------------------
     */
    ConfFile  cfile;
    MultiFile mfile(dev_type);
    ParamList plist;
    Image*    image;

    // Hack for printing refnames instead of names in ini file
    plist._print_refnames = print_refnames;

    // Set device for Param
    Param::dev_type = dev_type;

    // Warnings level
    Param::warn = warn;
    CrSpace::warn = warn;
    CrSpace::set_debugs();

    // Is image need be FailSafe
    Image::fs_image = fs_image;

    // Some sanity check
    if (wr_conf  &&
        find(rd_conf.begin(), rd_conf.end(), wr_conf) != rd_conf.end())
        SETERR(("It is impossible to read and to write parameters from/to "
               "same file (%s)\n", wr_conf));

    // Read FW image and parse parameter description
    if (!fw_image)
        SETERR(("Firmware image must be specified.\n"));
    if (!mfile.read(fw_image))
        SETERR(("\n-E- %s\n", mfile.err()));
    if (!plist.readFile(mfile, perl_replace))
        SETERR(("\n-E- %s\n", plist.err()));

    // Instantiate the correct image according to the device type.
    switch (mfile.DevType()) {
    case MT47396:
        if (exp_rom_file) {
            SETERR(("\n-E- Expansion rom file can be given to HCA's only.\n"));
        }
        if (user_data_file) {
            SETERR(("\n-E- User data file can be given to HCA's only.\n"));
        }

        image = new Image(&plist);
        break;
    case MT23108:
    case MT25208:
    case MT25218:
    case MT24204:
    case MT25204:
        image = new TImage(&plist, 
                           exp_rom_file, 
                           user_data_file, 
                           (rd_conf.empty() || !add_conf_sect) ? NULL : rd_conf.begin()->c_str(),
                           add_info_sect);
        break;
    default:
        SETERR(("\n-E- Unknown device type (%d)\n", mfile.DevType()));
    }

    if (image == NULL) {
        SETERR(("\n-E- Memory allocation failed\n"));
    }

    // Initial parameters check:
    if (!image->param_check(plist))
        SETERR(("\n-E- %s\n", image->err()));

    // If fwver flag is specified, print revision and exit
    if (fw_ver) {
        Image::FwVer v;
        if (! image->get_fw_ver(v))
            SETERR(("Can't read fw revision: %s",image->err()));

        printf("%d.%d.%d\n", v[0], v[1], v[2] );
        return 0;
    }

    // Read configuration file(s) if necessary
    for (list<string>::iterator it = rd_conf.begin(); it != rd_conf.end(); ++it)
        if (!cfile.read(it->c_str(), &plist))
            SETERR(("\n-E- %s\n", cfile.err()));

    // Process parameters from command line if necessary

    //TODO: move this code to plist class
    plist.clear_def_source();
    for (list<ss_t>::iterator it=cl_params.begin(); it != cl_params.end(); ++it)
    {
        string    name;
        u_int32_t geo;
        string    pname  = it->first;
        string    pvalue = it->second;
        vector<string> gr = splitv(pname, ",", true);

        switch (gr.size())
        {
        case 1:
            name = pname;
            geo = Param::GEO_DEF;
            break;
        case 2:
        {
            char *endp;

            name = gr[0];
            geo = strtol(gr[1].c_str(), &endp, 0);
            if (*endp)
                SETERR(("\nParameter \"%s\" - invalid GEO id (%s)\n",
                       pname.c_str(), gr[1].c_str()));
            break;
        }
        default:
            SETERR(("\nParameter \"%s\" - invalid syntax.\n", pname.c_str()));
            break;
        }

        map<string, Param*>::iterator pit = plist.params.find(name);
        if (pit == plist.params.end())
            SETERR(("Parameter \"%s\" not found.\n", name.c_str()));

        Param *par = pit->second;
        if (!par->assign(pvalue, "command line", -1, geo, true))
            SETERR(("Parameter \"%s\":  %s\n", pname.c_str(), par->err()));
    }

    // OK, all parameters are inside
    if (!plist.finalize_eval())
        SETERR(("\n-E- %s\n", plist.err()));

    // Check expansion rom enable parameter is set if an exp rom file is given
    bool exp_rom_en = false;
    const char* exp_rom_en_param = EXP_ROM_EN_PARAM_NAME;

    if (plist.exists(EXP_ROM_EN_PARAM_NAME)) {
        exp_rom_en = (plist.params[EXP_ROM_EN_PARAM_NAME]->get32() != 0);
    } else if (plist.exists(EXP_ROM_EN_PARAM_NAME_OLD)) {
        exp_rom_en = (plist.params[EXP_ROM_EN_PARAM_NAME_OLD]->get32() != 0);
        exp_rom_en_param = EXP_ROM_EN_PARAM_NAME_OLD;
    }

    if        ( exp_rom_file && !exp_rom_en) {
        SETERR(("\n-E- An expansion rom file is given, but parameter \"%s\" is not set.\n", exp_rom_en_param));
    } else if (!exp_rom_file &&  exp_rom_en) {
        SETERR(("\n-E- Parameter \"%s\" is set, but an expansion rom file not is given.\n", exp_rom_en_param));
    }

    // Write configuration file if necessary
    if (wr_conf  &&  !plist.writeConfFile(wr_conf, exclude_conf_defaults))
        SETERR(("\n-E- %s\n", plist.err()));
    
    // Preprocess parameter list
    if (EEPROM_image  &&  !image->preproc(plist, mfile))
        SETERR(("\n-E- %s\n", image->err()));

    // Compile image
    if (EEPROM_image  &&  !image->compile(plist, mfile))
        SETERR(("\n-E- %s\n", image->err()));

    // Write image
    if (EEPROM_image  &&  !image->write(EEPROM_image, EEPROM_format, mfile.DevType()))
        SETERR(("\n-E- %s\n", image->err()));

    delete image;

    return 0;
} // main

