#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
//#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <zlib.h>

#ifndef O_BINARY
#define  O_BINARY 0
#endif

#include <map>

#include "compatibility.h"

#include "TImage.h"
#include "MultiFile.h"
#include "Param.h"
#include "ParamList.h"

namespace std {}; using namespace std;

const char* TImage::VbhSect::_pll_names[NPLLS] = {"PLL.PLL0", "PLL.PLL1", "PLL.PLL2", "PLL.PLL3" };


////////////////////////////////////////////////////////////////////////
static int log2(const int val_)
{
    int          res = 0;
    unsigned int val = val_;
    if (val != 0)
        for (unsigned int tmp=1; tmp < val; tmp = tmp << 1)
            ++res;
    return res;
} // log2


TImage::~TImage() {
    // !!! Try to use auto_ptr for these lists !!!
    
    std::list<ImSection*>* plist[2] = {& _invariant_sects , &_image_sects};
    for (int i = 0 ; i < 2 ; i ++) 
    {
	for (std::list<ImSection*>::iterator it = plist[i]->begin(); it != plist[i]->end(); ++it)
	{
	    delete *it;
	}
    }
}


////////////////////////////////////
//
//  Image Section types
//
////////////////////////////////////

TImage::VbhSect::VbhSect (const string& name ) : ImSection(name) 
{
    clear();
}

bool TImage::VbhSect::Pack(vector<u_int8_t>& v) 
{
    int i;
    for (i=0;i<NPLLS;i++) 
	v << pll[i];
    for (i=0;i<4;i++)
	v << id[i];

    v << tlen;
    v << guid_ptr;

    return true;
}


void TImage::VbhSect::clear() {
    for (int i = 0; i < 4; i++)
    {
	pll[i] = 0;
	id[i]  = 0;
    }
    tlen       = 0;     
    guid_ptr   = 0;
}

bool TImage::VbhSect::fill_pll(ParamList* plist) 
{
    for (int i=0; i<NPLLS; i++)
	if (i == NPLLS-1)  // Last PLL is optional
	    pll[i] = plist->params[_pll_names[i]]->get32();
	else
	    pll[i] = plist->params[_pll_names[i]]->get32(); // ??? Check why is this  ??? 

    return true; 
}

bool TImage::VbhSect::fill_id(ParamList* plist)
{
    u_int32_t hwver;
    u_int32_t isver = 0;
    u_int32_t log2_sector_sz_ptr = 0;

    if (!get_fwid(plist, hwver))
        return false;
    if (plist->exists("FW._IS_RevisionID"))
	isver              = plist->params["FW._IS_RevisionID"]->get32();

    if (plist->exists("FW._log2_sector_sz_ptr"))
	log2_sector_sz_ptr = plist->params["FW._log2_sector_sz_ptr"]->get32();

    //printf("hwver:0x%x isver:0x%x log2_sector_sz_ptr:0x%x\n", hwver, isver, log2_sector_sz_ptr);
    id[0] = hwver | ((isver & 0xff) << 16);
    id[1] = log2_sector_sz_ptr;
    id[2] = 0;
    id[3] = 0;
    return true;
} // TImage::fill_vbh_id


////////////////////////////////////////////////////////////////////////
bool TImage::VbhSect::get_fwid(ParamList* plist, u_int32_t &fid)
{
    // FW ID
    u_int32_t     fw_rev;
    string        fw_rev_s;
    fw_rev_s = plist->params["FW._RevisionID"]->get();
    if (fw_rev_s == "")
        fw_rev_s = "0xa0";
    //--fw_rev_s = fw_rev_s.replace(QRegExp("0[Xx]"), "");
    //--fw_rev = fw_rev_s.simplifyWhiteSpace().toInt(&ok, 16);
    if (1 != sscanf(fw_rev_s.c_str() , " 0x%x ", &fw_rev  ))       // !!! Compare to infiniburn and see if works. Ask alexr how to use the Regexp class !!!
    {
        return errmsg("Invalid MLX revision ID (%s)",fw_rev_s.c_str());
    }
    // ??? !!! Endianess is reversed here - ask alexr ???
    fid = __cpu_to_be32(fw_rev);
    return true;
}

bool TImage::VbhSect::CreateVbhSect(ParamList* plist) 
{
    return fill_id(plist) && fill_pll(plist);
}


//////////////////////////////////////////////////////////////////////////


bool TImage::BootSect::Pack(vector<u_int8_t>& v) 
{
    v << code_addr;
    v << code_size;
    v << _data;
    v << strt_addr;
    v << (u_int32_t) _crc.get();

    return true;
}

bool TImage::BootSect::CreateBootSect(MSection* s)
{
    code_addr = s->addr;
    code_size = s->data.size() / 4;
    strt_addr = s->start;

    _data << s->data;
    return true;
}


TImage::PtrSect::PtrSect(const string& name)  : ImSection(name) 
{
    fi_addr = 0;
    fi_size = 0;
    memset(fw_reserved, 0, sizeof(fw_reserved));
    memset(vsd,         0, sizeof(vsd));
    memset(psid,        0, sizeof(psid));
}

bool TImage::PtrSect::CreatePtrSect(u_int32_t addr, u_int32_t size, ParamList* plist)
{
    // Fill PS
    fi_addr   = addr;
    fi_size   = size;
    signature = SIGNATURE;
    
    if (!plist->exists(BOOT0_LOAD_NAME))
	return errmsg("Param %s not found - Can't create section %s.\n", BOOT0_LOAD_NAME, _name.c_str()); 

    branch_to = plist->params[BOOT0_LOAD_NAME]->get32();

    if (plist->exists(VSD_PARAM_NAME))
    {
	strncpy(vsd , plist->params[VSD_PARAM_NAME]->get().c_str() , sizeof(vsd));
    }


    if (plist->refparams.find(PSID_PARAM_REFNAME) != plist->refparams.end())
    {
	strncpy(psid , plist->refparams[PSID_PARAM_REFNAME]->get().c_str() , sizeof(psid));
    } else {
	return errmsg("Param %s not found - Can't create section %s.\n", PSID_PARAM_REFNAME, _name.c_str());
    }

    // Calc CRC 
    CalcCrc();

    return true;
} 



bool TImage::PtrSect::Pack(vector<u_int8_t>& data) 
{
    data << fi_addr;       
    data << fi_size;       
    data << signature;     
    
    for (u_int32_t i = 0 ; i < sizeof(fw_reserved)/sizeof(fw_reserved[0]) ; i++) 
	data << fw_reserved[i];

    for (u_int32_t i = 0 ; i < sizeof(vsd)/sizeof(vsd[0]) ; i += 4) 
	data << *((u_int32_t*) &vsd[i]);

    for (u_int32_t i = 0 ; i < sizeof(psid)/sizeof(psid[0]) ; i += 4) 
	data << *((u_int32_t*) &psid[i]);

    data << branch_to;
    data << (u_int32_t) _crc.get();

    return true;
}

bool TImage::GpSect::Pack(vector<u_int8_t>& v) 
{
    v << type;
    v << size;
    v << param;
    v << next;
    v << _data;
    v << (u_int32_t) _crc.get();

    return true;
}

bool TImage::GpSect::SetNext(u_int32_t nptr) {
    next = nptr;
    return true;
}

bool TImage::GpSect::CreateExe(MSection *s) 
{
    type  = H_DDR;
    size  = s->data.size() / sizeof(u_int32_t);
    param = s->addr;
    _data << s->data;
    return true;
}


bool TImage::GpSect::CreateCnf (std::vector<u_int32_t>& cr_init) 
{
    type  = H_CNF;
    size  = cr_init.size(); 
    param = 0;

    _data << cr_init; 
    return true;
}

bool TImage::GpSect::CreateGuids (ParamList* plist, int gnum, u_int32_t guid_addr ) 
{
    
    // ??? Do I need to fill this data, or burner will replace it anyway ???

    // Get GUIDs and related info
    char *guid_names[NGUIDS]     = GUID_PARAM_NAMES;
    char *guid_names_old[NGUIDS] = GUID_PARAM_NAMES_OLD;


    u_int32_t           guids[NGUIDS];
    memset(guids, 0, sizeof(guids));

    for (int i=0; i<NGUIDS; i++)
    {
	if        (plist->exists(guid_names[i])) {
	    guids[i] = plist->params[guid_names[i]]->get32();
	} else if (plist->exists(guid_names_old[i])) {
	    guids[i] = plist->params[guid_names_old[i]]->get32();
	} else if (Param::warn) {
	    printf("-W- Missing GUID parameter : %s.\n", guid_names[i]);  // ??? is this an error, or put 0 ?
	}
    }

    if (!guids[6])             // SysImage GUIDs are same as Node GUIS
        guids[6] = guids[0];  // if not specified
    if (!guids[7])
        guids[7] = guids[1];


    // Put guids in data:

    type  = H_GUID;
    size  = gnum*2;
    param = guid_addr;  //??? What is this guid_addr - boot spec sais it must be ZERO
    
    // GUIDs
    for (int i=0; i<gnum*2; i++)
        _data << guids[i];

    return true;
}

    
bool TImage::GpSect::CreateImageInfo(Image* img, ParamList* plist) {

    #define TAG_HEADER(tagid, tagLen) ((u_int32_t)((tagid << 24) | (tagLen & 0xffffff)))

    // 1	8	FW Version
    // 2	8	FW Build Time
    // 3	4	Device Type
    // 4	16	PSID
    // 5	256	VSD
    // 6	Varying	Supported PSIDs (TBD)
    // 0xFF	0	END
    // All others	Reserved

    _data.clear();

    // Image Info Format Revision:
   
    _data << TAG_HEADER(II_IiFormatRevision, 4);
    _data << (u_int32_t) 1;

    // FwVersion:
    Image::FwVer fwVer;
    if (img->get_fw_ver(fwVer)) {
        _data <<  TAG_HEADER(II_FwVersion, 8);

        _data <<  (u_int32_t)(fwVer[0] << 16);
        _data <<  (u_int32_t) ((fwVer[1] << 16) | (fwVer[2] & 0xffff));
    }

    // DevId
    if (plist->exists("ADAPTER.adapter_dev_id")) {
        u_int32_t devId = plist->params["ADAPTER.adapter_dev_id"]->get32() & 0xffff;

        if (plist->exists("FW._RevisionID")) {
            devId |= ((plist->params["FW._RevisionID"]->get32() && 0xff) << 16);
        }

        _data << TAG_HEADER(II_DeviceType, 4);
        _data << devId;

    }

    // PSID
    if (plist->exists("ADAPTER.adapter_dev_id")) {
        char         psid_buf[17];
        const char*  psid = plist->params["ADAPTER.PSID"]->get().c_str();

        memset(psid_buf, 0, sizeof(psid_buf));
        memcpy(psid_buf, psid, min(strlen(psid), sizeof(psid_buf))); 

       _data << TAG_HEADER(II_PSID, 16);

       for (int i = 0; i < 16 ; i++) {
           _data << (u_int8_t) psid_buf[i];
       }
    }


    // End:
    _data << TAG_HEADER(II_End, 0);

    type  = H_IMG_INFO;
    size  = _data.size() / 4; 
    param = 0;


    return true;

}




bool TImage::GpSect::CreateJump(const MultiFile& mfile)
{
    std::list<string> ls = mfile.non_boot_sections();

    type  = H_JMP;
    size  = ls.size() * 2;
    param = 0;

    for (std::list<string>::iterator it = ls.begin(); it != ls.end() ; ++it) 
    {
	_data << mfile[it->c_str()]->crstart;
	_data << mfile[it->c_str()]->start;
    }
    return true;
}


//
// CreateFile() Creates 1 of 3 types of file sections:
// 1. Expansion rom 
// 2. User data.
// 3. FW configuration (brd/ini)
//
// USER_DATA section is treated exactly the same
// as exp rom, except that it does not require setting of any enabling parameter.
// This feature was added because exp rom size in Tavor is limited by 1MB. The user data
// section gives a mean for expanding the exp rom beyond that.
//
// FW configuration section contains a compressed ini/brd file used in the image preprocess stage.
//

bool TImage::GpSect::CreateFile (const string& file_name, TImage::SectionType file_type) {
    //
    // Get the file's extension. Allowed extensions: bin, img.
    //

    char* file_type_str;
    if        (file_type == H_ROM) {
	file_type_str = "expantion rom";
    } else if (file_type == H_USER_DATA) {
	file_type_str = "user data";
    } else if (file_type == H_FW_CONF) {
	file_type_str = "fw configuration";
    } else {
	return errmsg("Internal error: TImage::GpSect::CreateExpRom() expects types H_ROM, H_USER_DATA ot H_FW_CONF. Got %d.",
		      (u_int32_t) file_type);
    }


    string::size_type dot_pos = file_name.find_last_of(".");
    string extention;

    if (dot_pos != string::npos) {
	extention = file_name.substr(dot_pos + 1, dot_pos + 1 - file_name.size());
    }

    if        (extention == "img" || extention == "IMG") {

	if (!LoadImg(file_name))
	    return false; 

    } else if (extention == "bin" || extention == "BIN" || file_type == H_FW_CONF) {

	if (!LoadBin(file_name))
	    return false;

    } else {
	return errmsg("Bad extension for %s file: %s. Allowed extensions: .bin or .img  .", 
		      file_type_str,
		      file_name.c_str());
    }


    if (file_type !=  H_FW_CONF && _data.size() % 4) {
	return errmsg("Bad %s size (0x%x) read from file %s. Must a multiple of four.",
		      file_type_str,
		      (u_int32_t)_data.size(),
		      file_name.c_str());  
    }

    if (file_type == H_FW_CONF) {

	// gzip the file:
	// compress2() requirement: dest_buf.size >= src_buf.size * 1.001 + 12
	uLongf destLen = _data.size() + 12;
	destLen += destLen/100;
	vector<u_int8_t> tmp_buf(destLen);

	int rc = compress2(      (unsigned char *)((const char *)&(tmp_buf[0])), 
			   &destLen,
			   (const unsigned char *)((const char *)&(_data[0])),
			   _data.size(), 9);
	if (rc != Z_OK)
	{ 
	    return errmsg("Failed compressing file %s. compress2() returned %d.", file_name.c_str(), rc);
	}

	// Resize to fit compressed size. Keep DW alligned. (uncompress() ignores the tail)
	tmp_buf.resize((((u_int32_t)destLen + 3)/4) * 4);

	_data = tmp_buf; 
    }

    //
    // Set header:
    //

    size  = _data.size() / 4;
    type  = file_type;
    param = 0;

    return true;
}


bool TImage::GpSect::LoadImg(const string& file_name) {

    // Read and parse image file
    FILE *fp;  // !!! use autofile or c++ ifstream
    if ((fp = fopen(file_name.c_str(), "r")) == NULL)
    {
 	return errmsg("Can't open file %s: %s", file_name.c_str(), strerror(errno));
    }

    const int   MAX_STR = 1024;
    char        str[MAX_STR];
    char        *endp;

    bool        in_sect = false;

    u_int32_t   curr_line = 0;
    
    while(fgets(str, MAX_STR, fp))
    {
        curr_line++;

        // Skip comments and empty lines
        if (str[strspn(str, " \t")] == '#')
            continue;
        if (str[strspn(str, " \t")] == '\n')
            continue;
            
        vector<string> words = splitv(str, " \n");

        if (!in_sect )
        {
            //
            // Outside section
            //

            if (words[0] == "START") {
                in_sect = true;
            } else {
		return errmsg("%s:%d: Unexpected token - \"%s\". Expexted \"START\".",
			      file_name.c_str(),
			      curr_line,
			      words[0].c_str());
	    }
        }
        else
        {
            //
            // Inside section
            //

            if (words[0] == "END")
            {
		in_sect = false;
            }
            else
            {
                // Load the actual data
                for (vector<string>::iterator it = words.begin() ; it != words.end() ; ++it) {
                    u_int8_t data_byte = strtoul(it->c_str(), &endp, 16);
                    if (*endp != '\0') {
			return errmsg("%s:%d: Expecting hexadecimal bytes %s. Read \"%s\".",
				      file_name.c_str(),
				      curr_line,
				      it == words.begin() ? "or \"END\"" : "" , 
				      it->c_str());
                    }

                    _data.push_back(data_byte);
                }
            }
        }
    }
    fclose(fp);

    return true;
}

bool TImage::GpSect::LoadBin(const string& file_name) {

    Stat               stat_buf;
    int                r_cnt;

    int fd = COMP_OPEN(file_name.c_str(), O_RDONLY | O_BINARY);
    if (fd < 0)
    {
        return errmsg("Can't open file \"%s\" - %s\n", 
		      file_name.c_str(),
		      strerror(errno));
    }

    if (COMP_FSTAT(fd, &stat_buf) < 0)
    {
        return errmsg("Can't stat file \"%s\" - %s\n", 
		      file_name.c_str(),
		      strerror(errno));

    }

    //printf("%ld / 0x%lx\n", stat_buf.st_size, stat_buf.st_size);

    u_int8_t* buff = new u_int8_t[stat_buf.st_size];

    if ((r_cnt = COMP_READ(fd, buff, stat_buf.st_size)) != stat_buf.st_size)
    {
        if (r_cnt < 0)
            return errmsg("Read error on file \"%s\" - %s\n",
			  file_name.c_str(), 
			  strerror(errno));
        else
            return errmsg("Read error on file \"%s\" - read only %d bytes (from %ld)\n",
			  file_name.c_str(),
			  r_cnt, 
			  (unsigned long)stat_buf.st_size);
    }

    COMP_CLOSE(fd);

    _data.insert(_data.begin(), buff, buff + stat_buf.st_size);

    delete [] buff;

    return true;
} 


// Tavor params check
bool  TImage::param_check (ParamList& plist)
{
    // Add PSID parameter if it doesn't exist. Reference to 
    // this parameter is through it's refname - the name doesn't 
    // really matter.

    _plist = &plist;
    if (_plist->refparams.find(PSID_PARAM_REFNAME) == _plist->refparams.end()) {
	_plist->add_psid(PSID_PARAM_NAME, "dummy", "0x0");
    }
    return true;
}

////////////////////////////////////
//
// Compile
//
////////////////////////////////////


bool TImage::compile(ParamList &plist, const MultiFile &mfile)
{
    _plist = &plist;

    // Try find BOOT0
    _boot0_found = (NULL != mfile[BOOT0_EXE_NAME]); // ??? orenk is it OK to use this hardcoded name for BOOT0 ???

    if (fs_image && !_boot0_found)
	return errmsg("Failsafe image requested, but boot0 section (%s) not found in file %s.\n", BOOT0_EXE_NAME, mfile.name());

    if (!CreateImage(mfile))
	return false; 

    if (!_boot0_found) 
    {
        _total             = _im_total;
	_first_image_start = 0;
    }
    else
    {
	

	/*
	 * Invariant sector 0
	 * ------------------
	 */

	// Header
	VbhSect* vbh;
	vbh = new VbhSect("InvariantSector0");
	
	if (!vbh->CreateVbhSect(_plist))
	    return errmsg("BOOT0 Init failed");

	// GUID ptr
	vbh->guid_ptr = PtrSect::SIGNATURE;
	_invariant_sects.push_back(vbh);

	// Boot0 exe:

	BootSect* boot0 = new BootSect(BOOT0_EXE_NAME);
	boot0->CreateBootSect(mfile[BOOT0_EXE_NAME]); 

        // Patch Boot0 executable
/*
  How to patch boot0 executable:

  ._________________________________________________________________________.
  ! Offset   ! Offset    !                        !                         !
  ! from     ! from      ! bits 31-16             ! bits 15-0               !
  ! IS beg.  ! BOOT0 exe !                        !                         !
  !__________!___________!________________________!_________________________!
  !          !           !                        !                         !
  ! 0x30     ! 0x0       ! Do not touch           ! log2(sector size)       !
  ! 0x34     ! 0x4       ! Do not touch           ! PPS offset (in sectors) !
  ! 0x38     ! 0x8       ! Do not touch           ! SPS offset (in sectors) !
  ! 0x3c     ! 0xc       ! Do not touch           ! log2(bank size)         !
  !__________!___________!________________________!_________________________!

 */
        // Try to get log2_sector_sz_ptr from MLX file
        u_int32_t log2_sector_sz_ptr;
        if (_plist->exists("FW._log2_sector_sz_ptr")) // ??? Ask tsirkin about the name ???
	    log2_sector_sz_ptr = _plist->params["FW._log2_sector_sz_ptr"]->get32();
	else
            log2_sector_sz_ptr = 0; // ??? what does it mean when this param is not found ???

        if (log2_sector_sz_ptr & 0x03)
            return errmsg("Error: FW._log2_sector_sz_ptr is %d. Must by multiple of 4!!!\n", log2_sector_sz_ptr);
        

	// Patch boot0
	boot0->SetWord(log2_sector_sz_ptr       , (u_int16_t)log2(_nvram._sect_size));  // Log2 Sector size
	boot0->SetWord(log2_sector_sz_ptr + 0x4 , (u_int16_t)1);                        // PPS offset (in sectors)
	boot0->SetWord(log2_sector_sz_ptr + 0x8 , (u_int16_t)2);                        // SPS offset (in sectors)
	boot0->SetWord(log2_sector_sz_ptr + 0xc , (u_int16_t)log2(_nvram._bank_size));  // Log2 Bank size

	boot0->CalcCrc();

	_invariant_sects.push_back(boot0);

        /*
         * Primary/Seconady pointer sectors
         * --------------------------------
         */

        //!!! Think I can remove: if (_image_sects.front()->GetName() != BOOT2_EXE_NAME)
        //!!!                         return errmsg("%s section must be first in image, But %s is first.", BOOT2_EXE_NAME ,_image_sects.front()->GetName().c_str());

	//
	// Pointer sections
	//

	PtrSect* ps;

	ps = new PtrSect("PrimaryPointerSector0");
	if (!ps->CreatePtrSect(_nvram._sect_size * 3, _im_total, _plist))
            return errmsg(ps->err()); 
	
	_invariant_sects.push_back(ps);


	ps = new PtrSect("SecondaryPointerSector0");
	if (!ps->CreatePtrSect(_nvram._sect_size * (3 + _im_total_s), _im_total, _plist))
            return errmsg(ps->err()); 
	
	_invariant_sects.push_back(ps);


        /*
         * Calculate image length
         * ----------------------
         */

	SectList::iterator it;

        // Total length in bytes
        _total = _im_total * 2;
	for (it = _invariant_sects.begin() ; it != _invariant_sects.end() ; it++ ) {
	    _total += (*it)->Size();	    
	}

	// inv sector addresses:
	it = _invariant_sects.begin();
	(*it)->SetRomStartAddr(0);                                    // Prolog - start = 0
	++it;
	(*it)->SetRomStartAddr((*_invariant_sects.begin())->Size());  // Boot0  - starts at end of Prolog section.
	++it;
	(*it)->SetRomStartAddr(_nvram._sect_size);                    // PPS    - starts at second sector
	++it;
	(*it)->SetRomStartAddr(2 * _nvram._sect_size);                 // SPS    - starts at secondthird sector

	_first_image_start  = _nvram._sect_size * 3;
	_second_image_start = _first_image_start + _im_total_s * _nvram._sect_size;
    }

    SetReplaces();

    return true; 
} // TImage::compile

bool TImage::SetReplaces()
{
    if (fs_image) {
	_guids_repl.push_back( EE_loc(_first_image_start  + _guid_sect->GetRomStartAddr() + _guid_sect->HdrSize() , 0));
	_guids_repl.push_back( EE_loc(_second_image_start + _guid_sect->GetRomStartAddr() + _guid_sect->HdrSize() , 0));
	
	set_crc_rep(_guid_sect , _first_image_start);
	set_crc_rep(_guid_sect , _second_image_start);

	SectList::iterator it = _invariant_sects.begin();
	++it;  // Skip PLL
	++it;  // Skip VBH

	// Go through both PtrSects
	for (;it != _invariant_sects.end(); ++it) { 
	    _psid_repl.push_back( EE_loc((*it)->GetRomStartAddr() + PtrSect::PSID_OFFSET , 0));	    
	    _vsd_repl.push_back( EE_loc((*it)->GetRomStartAddr() + PtrSect::VSD_OFFSET , 0));
	    set_crc_rep(*it, 0);
	}

    } else {
	_guid_repl.push_back( EE_loc(_guid_sect->GetRomStartAddr() + _guid_sect->HdrSize() , 0));
	set_crc_rep(_guid_sect , 0);
    }
    return true;
}

bool TImage::set_crc_rep(ImSection* s, u_int32_t offs) 
{
    _crc_repl.push_back(CRC_Replacement(
	EE_loc(offs + s->GetRomStartAddr()                 , 0),
	EE_loc(offs + s->GetRomStartAddr() + s->Size() - 8 , 0),
	EE_loc(offs + s->GetRomStartAddr() + s->Size() - 4 , 0)));

    return true;
}



bool TImage::add_sects(SectList& sl, u_int32_t sect_size)
{
    SectList::iterator it;
    int i;
    for (i = 0, it = sl.begin(); it != sl.end() ; ++it, ++i)
    {
	if (sect_size)
	    _nvram.pad_to_addr(sect_size * i);

	_nvram.start((*it)->GetName().c_str());
	if (!(*it)->Pack(_nvram.data()))
	    return false;
    }
    return true;
}

////////////////////////////////////////////////////////////////////////


bool TImage::CreateImage(const MultiFile& mfile)
{
    VbhSect*              vbh;
    BootSect*             boot;
    GpSect*               gp;
    GpSect*               guidp;
    GpSect*               image_info = NULL;

    // Initial value
    //--    _cnf_counter = 0;

    if (NULL == mfile[BOOT2_EXE_NAME])
	return errmsg ("BOOT2 exe \"%s\" not found.\n", BOOT2_EXE_NAME);


    // Very beggining headers (PLL, FW ID)
    vbh = new VbhSect("PrologSection_PLL_EEPROMID");
    vbh->CreateVbhSect(_plist);
    _image_sects.push_back(vbh);


    // Boot 2
    boot = new BootSect(BOOT2_EXE_NAME);
    boot->CreateBootSect(mfile[BOOT2_EXE_NAME]);
    _image_sects.push_back(boot);

    // Boot3 (if exists)
    if (NULL != mfile[BOOT3_EXE_NAME]) {
	boot = new BootSect(BOOT3_EXE_NAME);
	boot->CreateBootSect(mfile[BOOT3_EXE_NAME]);
	_image_sects.push_back(boot);
    }


    // insert boot2 static configuration
    gp = new GpSect("StaticConfigurationCection0");
    gp->CreateCnf(_cr_boot2[Param::GEO_DEF].rmw_init);
    _image_sects.push_back(gp);

    // insert Expansion ROM section
    if (NULL != _exp_rom_fn) 
    {
	gp = new GpSect("ExpansionROMSection");
	if (!gp->CreateFile(_exp_rom_fn, H_ROM))
	    return errmsg(gp->err());
	_image_sects.push_back(gp);

    }

    // insert user data section
    if (NULL != _user_data_fn) 
    {
	gp = new GpSect("UserDataSection");
	if (!gp->CreateFile(_user_data_fn, H_USER_DATA))
	    return errmsg(gp->err());
	_image_sects.push_back(gp);
    }

    // Insert GUIDs section
    guidp = new GpSect("GUIDsSection");
    guidp->CreateGuids(_plist, _boot0_found ? 4 : 3, 0); // ???!!! What is this guid addr - ask alexr ???!!!
    _image_sects.push_back(guidp);

    _guid_sect = guidp;

    // Image info:
    if (_add_info_sect) {
        image_info = new GpSect("ImageInfo");
        if (!image_info->CreateImageInfo(this, _plist)) {
            return errmsg(image_info->err());
        }

        _image_sects.push_back(image_info);
    }

    //InsertBoardID();                              // Insert Board ID section !!! Add later !!!

    // All other executables but first (except boot2/boot3/boot0 that are already here)
    #ifndef TAVOR_DEBUG
    list<string>           sl = mfile.non_boot_sections();
    #else
    //-- !!! This is for diffing with infiniburn - REMOVE WHEN DONE !!!
    list<string>           sl;
    sl.push_back("exus.exe");
    sl.push_back("exur.exe");
    sl.push_back("tcu.exe" );
    sl.push_back("tpt.exe" );
    sl.push_back("qpc.exe" );
    if (mfile.DevType() == MT23108)
	sl.push_back("ntu.exe");
    else
	sl.push_back("sysdis.exe");
    #endif
    
    list<string>::iterator it;

    for (it = sl.begin() ; it != sl.end(); ++it)
    {
	gp = new GpSect(*it);
	gp->CreateExe(mfile[it->c_str()]);
	_image_sects.push_back(gp);
    }

    // Second static configuration partition
    gp = new GpSect("StaticConfigurationCection1");
    gp->CreateCnf(_cr[Param::GEO_DEF].rmw_init);
    _image_sects.push_back(gp);

    // Jump address
    gp = new GpSect("IriscsJumpAddressesSection");
    gp->CreateJump(mfile);
    _image_sects.push_back(gp);


    // InsertServiceSect();
    if (_fw_conf_fn != NULL) {
	gp = new GpSect("ConfFileSection");
	if (!gp->CreateFile(_fw_conf_fn, H_FW_CONF))
	    return errmsg(gp->err());
	_image_sects.push_back(gp);
    }

    // arrange addresses
    ArrangeAddresses();

    // set pointer to GUID section
    vbh->guid_ptr = guidp->GetRomStartAddr() + 
	            guidp->HdrSize();

    if (_add_info_sect) {
        u_int32_t   info_ptr = image_info->GetRomStartAddr() +
                               image_info->HdrSize();
        
        u_int32_t info_ptr_max = 0xffffff;
        if (info_ptr >= info_ptr_max) {
            return errmsg("Info section pointer is 0x%x. Must be less than 0x%x",
                          info_ptr,
                          info_ptr_max);
        }


        u_int8_t info_ptr_cs = ( info_ptr &     0xff) +
                               ((info_ptr &   0xff00) >>  8) +
                               ((info_ptr & 0xff0000) >> 16);

        info_ptr = info_ptr | ((-info_ptr_cs) << 24);

        vbh->id[3] = info_ptr;
    }
    
    // Calc total length
    _im_total = 0;
    for (SectList::iterator sit = _image_sects.begin() ; sit != _image_sects.end() ; ++sit) 
	_im_total += (*sit)->Size();

    _im_total   = (_im_total + 3) & ~0x3;
    _im_total_s = (_im_total + _nvram._sect_size - 1) / _nvram._sect_size;

    vbh->tlen = _im_total;

    // Calc CRC for all sects
    for (SectList::iterator sit = _image_sects.begin() ; sit != _image_sects.end() ; ++sit) 
	(*sit)->CalcCrc();

    return true;

} // TImage::CreateImage()


////////////////////////////////////////////////////////////////////////
bool TImage::ArrangeAddresses()
{

    // Addresses are arranged from start of image in flash.
    // !!! NOTE below code doesn't support multi bank !!!

    u_int32_t     current_addr = 0;
    //-- int           current_bank = 0;
    //-- int           current_len  = 0;

    SectList::iterator    prev , curr;

    _image_sects.front()->SetRomStartAddr(0);

    prev = _image_sects.begin();
    curr = prev;
    ++curr;

    for (;curr != _image_sects.end() ; ++curr, ++prev) 
    {
	current_addr += (*prev)->Size();

	(*curr)->SetRomStartAddr(current_addr);
	if (!(*prev)->SetNext(current_addr))
	    return false; 
    }

    (*prev)->SetNext(_boot0_found ? 0 : 0xff000000 );  // ??? Ask alexr why is that ???

    return true;

} // Image::arrangeAddresses

       
bool    TImage::write_image(FILE* fp, const char* fname, FORMATS frm)
{
    bool ret = true;
     
    // I'm using pointer to a member function. 
    // That's disgusting !!!

    typedef bool (TImage::*DUMP_SECTIONS)(FILE* fp, u_int32_t rom_offset, SectList& sects);
    
    DUMP_SECTIONS dump_sections;

    if        (frm == OUT_BINARY) {
	fp = fopen(fname, "wb");
	if (!fp)  
	    return errmsg("Can't open file \"%s\": %s", fname, strerror(errno));

	// !!! for anafa2 files open for binary is done localy - maybe unify it same way as IMAGE format !!!
	dump_sections = &TImage::binary_dump;

    } else if (frm == OUT_IMAGE) {
	dump_sections = &TImage::mlx_dump;
    } else {
	return errmsg("Error: HCA image compiler currently supports only IMAGE and BINARY formats\n");
    }


    if (fs_image) {
	
	ret &= (this->*dump_sections)    (fp , 0                   , _invariant_sects );
	ret &= (this->*dump_sections)    (fp , _first_image_start  , _image_sects );
	ret &= (this->*dump_sections)    (fp , _second_image_start , _image_sects );
    }
    else 
    {
	ret &= (this->*dump_sections)    (fp , 0                   , _image_sects );
    } 
    
    if (frm == OUT_BINARY) 
	fclose (fp);

    return ret;
}

bool TImage::mlx_dump(FILE* fp, u_int32_t rom_offset, SectList& sects)
{
    if (sects.empty())
	return true;

    for (SectList::iterator it = sects.begin() ; it != sects.end() ; ++it) 
    {
	u_int32_t i;
	vector<u_int8_t> data;
	
	fprintf(fp, "START %s %08x %02x\n", (*it)->GetName().c_str(), (*it)->GetRomStartAddr() + rom_offset ,  0 );
	
	(*it)->Pack(data); // !!! check err

	for (i = 0; i < data.size(); i++)
	{ 	    
	    // Dump data
	    fprintf(fp, " %02x", data[i] & 0xff);
	     
	    if (((i+1) % IN_LINE) == 0)
		fprintf(fp, "\n");
	}

	fprintf(fp, "%sEND %s\n\n", i % IN_LINE ? "\n" : "", (*it)->GetName().c_str());
    }
    return true; 
}


bool TImage::binary_dump(FILE* fp, u_int32_t rom_offset, SectList& sects)
{
    if (sects.empty())
	return true;


    for (SectList::iterator it = sects.begin() ; it != sects.end() ; ++it) 
    {
	u_int32_t sect_start = (*it)->GetRomStartAddr() + rom_offset;

//	printf("-D- Dumping sect %-30s STRT=%8x SIZE=%8x CURR=%8x\n", 
//	       (*it)->GetName().c_str(), 
//	       sect_start,
//	       (*it)->Size(),
//	       _current_written_addr);

	// Pad to section start
	char b_ff = (char)0xff;
	for (;_current_written_addr < sect_start  ; _current_written_addr++) { 
	    fwrite(&b_ff, 1, 1, fp);
	}

	vector<u_int8_t> data;
	     
	(*it)->Pack(data); // !!! check err
	
	for (u_int32_t i = 0 ; i < data.size() ; i++) { 
	    fwrite(&(data[i]) ,1 , 1 , fp);
	    _current_written_addr++;
	}

    }

    return true; 

}

bool TImage::get_fw_ver (FwVer v) {

    char* ver_params[] = {
	"FW.Major Rev ID",
	"FW.Minor Rev ID",
	"FW.SubMinor rev ID"
    };
    map<string, Param*>::iterator pit;

    for (u_int32_t i = 0; i < 3 ; i++) {
        if ((pit = _plist->params.find(ver_params[i])) != _plist->params.end())
            v[i] = pit->second->get32();
        else
            return errmsg("Can't get version info: Param %s not found\n", ver_params[i]);
    }

    return true;
}

bool TImage::fs_info_dump(FILE* fp) 
{

    SectList::iterator it = _invariant_sects.begin();
    ++it;  // Skip PLL
    ++it;  // Skip VBH
    
    // Go through both PtrSects
    for (u_int32_t i = 0; i < 2; ++it, ++i) { 
	PtrSect* ps = (PtrSect*)(*it);  // !!! Dangerous assumption - should I keep ptr to PSs ? !!!

	fprintf(fp, "PS        %d %08x %02x\n", i, ps->GetRomStartAddr(), 0);
	fprintf(fp, "SIGNATURE %d %08x %02x\n", i, ps->GetRomStartAddr() + PtrSect::SIGNATURE_OFFSET, 0);
	fprintf(fp, "IMAGE     %d %08x %02x\n", i, ps->fi_addr, 0);     
	fprintf(fp, "\n");
    }
    
    fprintf(fp, "\n");
    return true;
}


