/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  ParamList.cpp - Paramater definition class
 *
 *  Version: $Id: ParamList.cpp 2752 2006-01-19 14:40:17Z mst $
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include <iostream>
#include <fstream>
#include <vector>

#include "MultiFile.h"
#include "PerlClass.h"
#include "Param.h"
#include "ParamList.h"
#include "utils.h"
#include "compatibility.h"
#include "MicExpr.h"

#ifndef XML_STATUS_OK
#define XML_STATUS_OK 1
#endif

extern char *g_version;

// Common global macros/types definitons
#define numb_el(x) (sizeof(x)/sizeof(x[0]))
namespace std {}; using namespace std;
typedef pair<string,string>        ss_t;
typedef pair<string,u_int32_t>     su_t;
typedef pair<string,vector<string> > sl_t;
typedef void (ParamList::*start_el_hnd)(map<string,string>& atts);
typedef void (ParamList::*end_el_hnd)();
typedef pair<string,start_el_hnd>  s1_t;
typedef pair<string,end_el_hnd>    s2_t;

// Version of prepA2FW
static const char *_auto_vers = "Auto.prepVerion";

// Tag surrounds parameters definition. The same
// name used as an section name in MultiFile for
// FW parameters
const char *params_tag = "FWParameters";

// Mandatory attributes for <source> element
static const char *_s_name = "source";
static const char *_s_mand[] = { "file", "line" };

// Mandatory attributes for <enum>
static const char *_e_name = "enum";
static const char *_e_mand[] = { "name" };

// Mandatory attributes for <group> element
static const char *_g_name = "group";
static const char *_g_mand[] = { "name" };

// Mandatory attributes for <param> element
static const char *_p_name = "param";
static const char *_p_mand[] = { "name", "def", "type" };

// Extra tags (allowed although unhandled);
static const char *_ex_tags[] = { params_tag ,
                                  "br",         // Leftovers from infiniburn (GUI related tags) - ignore.
                                  "center",
                                  "clospan",
                                  "colspan",
                                  "custColor",
                                  "frame",
                                  "hr",
                                  "label",
                                  "sp",
                                  "style",
                                  "TEMTParameters"
};

// Default values for <param> non-madatory parameters
static const ss_t _p_def[] = { ss_t("repr",      "uns"),
                               ss_t("hidden",    "0"),
                               ss_t("ro",        "0"),
                               ss_t("tag",       "0"),
                               ss_t("allow_geo", "0"),
                               ss_t("offset",     "0"),
                               ss_t("boot2",     "0")
};

// Possible representations
static const su_t _r_def[] = { su_t("uns",   Param::UNS),
                               su_t("int",   Param::INT),
                               su_t("uns64", Param::UNS64),
                               su_t("int64", Param::INT64),
                               su_t("enum",  Param::ENUM),
                               su_t("ascii", Param::ASCII),
                               su_t("ASCII", Param::ASCII),
                               su_t("bool",  Param::BOOL)
};

// Possible types
static const su_t _t_def[] = { su_t("c",       Param::C),
                               su_t("crspace", Param::CRSPACE),
                               su_t("dummy",   Param::DUMMY)
};

// Start element handlers per element names
static const s1_t _start_h[] = { s1_t("source", &ParamList::start_source),
                                 s1_t("PERL",   &ParamList::start_perl),
                                 s1_t("enum",   &ParamList::start_enum),
                                 s1_t("group",  &ParamList::start_group),
                                 s1_t("param",  &ParamList::start_param),
                                 s1_t("inipref",&ParamList::start_inipref),
                                 s1_t("set_default_prop", &ParamList::start_set_default_prop)
};

// Start element handlers per element names
static const s2_t _end_h[] = { s2_t("source", &ParamList::end_source),
                               s2_t("PERL",   &ParamList::end_perl),
                               s2_t("enum",   &ParamList::end_enum),
                               s2_t("group",  &ParamList::end_group),
                               s2_t("param",  &ParamList::end_param),
                               s2_t("inipref",&ParamList::end_inipref),
                               s2_t("set_default_prop", &ParamList::end_set_default_prop)
              
};


////////////////////////////////////////////////////////////////////////
// Static containers initialization for mandatory attribures
static vector<string>       group_mand; //(&_g_mand[0], &_g_mand[numb_el(_g_mand)]);
static vector<string>       param_mand; //(&_p_mand[0], &_p_mand[numb_el(_p_mand)]);
static vector<string>       enum_mand;  //(&_e_mand[0], &_e_mand[numb_el(_e_mand)]);
static vector<string>       src_mand;   //(&_s_mand[0], &_s_mand[numb_el(_s_mand)]);

static const sl_t _sl_def[] = { sl_t(_s_name, src_mand),
                                sl_t(_e_name, enum_mand),
                                sl_t(_g_name, group_mand),
                                sl_t(_p_name, param_mand)
};
static map<string,vector<string> >
                      mand_atts;      //(&_sl_def[0], &_sl_def[numb_el(_sl_def)]);

////////////////////////////////////////////////////////////////////////
// Static container initialization for allowed extra tags
static vector<string>       extra_tags; //(&_ex_tags[0], &_ex_tags[numb_el(_ex_tags)]);

////////////////////////////////////////////////////////////////////////
// Static containers initialization for default attribute values
static map<string,string> param_def; //(&_p_def[0], &_p_def[numb_el(_p_def)]);

////////////////////////////////////////////////////////////////////////
// Possible representations static initialization
static map<string,u_int32_t> repr_def; //(&_r_def[0], &_r_def[numb_el(_r_def)]);

////////////////////////////////////////////////////////////////////////
// Possible types static initialization
static map<string,u_int32_t> types_def; //(&_t_def[0], &_t_def[numb_el(_t_def)]);

////////////////////////////////////////////////////////////////////////
// start/end handler maps static initialization
static map<string,start_el_hnd>  start_h; //(&_start_h[0],&_start_h[numb_el(_start_h)]);
static map<string,end_el_hnd>    end_h;   //(&_end_h[0], &_end_h[numb_el(_end_h)]);

static void init_vec_from_array(vector<string>& v , const char* arr[], u_int32_t size) {
    v.clear();
    v.resize(size);
    for (u_int32_t i=0 ; i < size; i++) {
        v[i] = arr[i];
    }
}

template <class K, class V>
static void init_map_from_array(map<K,V>& m, const pair<K,V>* arr , u_int32_t size) {
    m.clear(); 
    for (u_int32_t i=0 ; i < size; i++) {
        m.insert(arr[i]);
    }
}


static void init_static_db() {

    //group_mand.assign(&_g_mand[0]), &_g_mand[numb_el(_g_mand)]);
    //param_mand.assign(&_p_mand[0], &_p_mand[numb_el(_p_mand)]);   
    //enum_mand.assign (&_e_mand[0], &_e_mand[numb_el(_e_mand)]);   
    //src_mand.assign  (&_s_mand[0], &_s_mand[numb_el(_s_mand)]);

    // extra_tags.assign(&_ex_tags[0],&_ex_tags[numb_el(_ex_tags)]);

    init_vec_from_array(group_mand, _g_mand, numb_el(_g_mand));
    init_vec_from_array(param_mand, _p_mand, numb_el(_p_mand));
    init_vec_from_array(enum_mand , _e_mand, numb_el(_e_mand));
    init_vec_from_array(src_mand,   _s_mand, numb_el(_s_mand));

    init_vec_from_array(extra_tags, _ex_tags,numb_el(_ex_tags));

    init_map_from_array(mand_atts, _sl_def,  numb_el(_sl_def));
    init_map_from_array(param_def, _p_def,   numb_el(_p_def));
    init_map_from_array(repr_def,  _r_def,   numb_el(_r_def));
    init_map_from_array(types_def, _t_def,   numb_el(_t_def));
    init_map_from_array(start_h,   _start_h, numb_el(_start_h));
    init_map_from_array(end_h,     _end_h,   numb_el(_end_h));

    // mand_atts.insert (&_sl_def[0], &_sl_def[numb_el(_sl_def)]);
    // param_def.insert (&_p_def[0],  &_p_def[numb_el(_p_def)]);
    // repr_def.insert  (&_r_def[0],  &_r_def[numb_el(_r_def)]);
    // types_def.insert (&_t_def[0],  &_t_def[numb_el(_t_def)]);
    // start_h.insert   (&_start_h[0],&_start_h[numb_el(_start_h)]);  
    // end_h.insert     (&_end_h[0],  &_end_h[numb_el(_end_h)]);         

}


////////////////////////////////////////////////////////////////////////
ParamList::ParamList() : _mfile(0), _line(0)
{
    _parser_debug = getenv("MIC_PARSER_DEBUG") ? true : false;

    init_static_db();

    _Perl = new Perl();
} // ParamList::ParamList


////////////////////////////////////////////////////////////////////////
ParamList::~ParamList()
{
    // Delete params
    for (map<string, Param*>::iterator mit = params.begin();
         mit != params.end(); ++mit)
        delete mit->second;

    // Delete groups
    for (map<string, Group*>::iterator mit = grmap.begin();
         mit != grmap.end(); ++mit)
        delete mit->second;

    // Delete enums
    for (map<string, Enum*>::iterator mit = enums.begin();
         mit != enums.end(); ++mit)
        delete mit->second;

    delete _Perl;
} // ParamList::~ParamList

////////////////////////////////////////////////////////////////////////
bool ParamList::exists(const char* pname) 
{
    map<string, Param*>::iterator it = params.find(pname);
    if (it != params.end())
        return true;
    else 
        return errmsg("Parameter %s does not exist.", pname);
}

bool ParamList::add_psid(string name ,string type , string addr) {
    //orenk !!! Add PSID parameter if not found, so it would be accessible from conf file.
    
    if (!exists(name.c_str())) {
        std::map<std::string,std::string> gr_atts; 
        std::map<std::string,std::string> par_atts;
        
        // printf("-D- Adding %s param\n", name.c_str());  
        vector<string> vname = splitv(name, ".");
          
        gr_atts["name"]     = vname[0];
        
        par_atts["name"]    = vname[1];
        par_atts["refname"] = "PSID";
        par_atts["def"]     = "";
        par_atts["hidden"]  = "0"; 
        par_atts["ro"]      = "0";
        par_atts["type"]    = type;
        par_atts["repr"]    = "ASCII";
        par_atts["addr"]    = addr;
        par_atts["length"]  = "16";
        
        start_group(gr_atts);
        start_param(par_atts);
        end_param();
        end_group(); 
    }
    
    return true ;
}


bool ParamList::enableExpRom() {

    map<string, Param*>::iterator it = params.find(EXP_ROM_EN_PARAM_NAME);
    if (it == params.end()) 
        return errmsg("Parameter \"%s\" not found.", EXP_ROM_EN_PARAM_NAME);

    Param *par = it->second;
    if (!par->assign("1"))
        return errmsg("Parameter \"%s\":  %s\n", EXP_ROM_EN_PARAM_NAME, par->err());  

    return true;
}


////////////////////////////////////////////////////////////////////////
void ParamList::setPerl(const string& expr)
{
    _perl = _perl_replacement.empty() ? expr : _perl_replacement;
    if (_perl.empty())
        return;

    _Perl->eval(_perl);
} // ParamList::setPerl


////////////////////////////////////////////////////////////////////////
bool ParamList::readFile(const MultiFile& mfile, const char *pfname)
{
    // Perl replacement
    if (pfname)
    {
        string   rbuf;
        ifstream fp(pfname);

        if (!fp)
            return errmsg("Can't open file \"%s\": %s", pfname, strerror(errno));
        while (getline(fp, rbuf))
        {
            _perl_replacement += rbuf;
            _perl_replacement += "\n";
        }
    }

    _mfile = (MultiFile*)&mfile;
    _fname = mfile.name();
    MSection *sect = mfile[params_tag];

    if (!sect)
        return errmsg("Can't find section \"%s\" in file \"%s\"",
                      params_tag, mfile.name());
    if (sect->bin)
        return errmsg("Section \"%s\" in file \"%s\" should be ASCII.",
                      params_tag, mfile.name());

    XML_Parser     parser = XML_ParserCreate(0);
    XML_SetUserData(parser, this);
    XML_SetElementHandler(parser, &ParamList::s_start_el, &ParamList::s_end_el);
    XML_SetCharacterDataHandler(parser, &ParamList::s_chars);

    h_start_doc();

    _line=sect->start_line;
    for (vector<string>::iterator vit = sect->lines.begin();
         vit != sect->lines.end(); _line++, ++vit)
    {
        string rbuf = *vit;
        if (XML_Parse(parser, rbuf.c_str(), rbuf.length(), 0) != XML_STATUS_OK)
            return errmsg("File: %s, Line: %d - %s", _fname.c_str(), _line,
                          XML_ErrorString(XML_GetErrorCode(parser)));
        if (err())
            return false;
    }

    if (XML_Parse(parser, 0, 0, 1) != XML_STATUS_OK)
        return errmsg("File: %s, Line: %d (End of file) - %s", _fname.c_str(),
                      _line, XML_ErrorString(XML_GetErrorCode(parser)));

    if (err())
        return false;

    h_end_doc();
    XML_ParserFree(parser);

    return finalize_init();
} // ParamList::readFile

#if 0
// Now all reads - only via mfile interface
// The code is here for debug purposes only

////////////////////////////////////////////////////////////////////////
bool ParamList::readFile(const char *fname)
{
    _fname = fname;

    ifstream fp(fname);
    if (!fp)
        return errmsg("Can't open file \"%s\": %s",
                      _fname.c_str(), strerror(errno));

    XML_Parser     parser = XML_ParserCreate(0);
    XML_SetUserData(parser, this);
    XML_SetElementHandler(parser, &ParamList::s_start_el, &ParamList::s_end_el);
    XML_SetCharacterDataHandler(parser, &ParamList::s_chars);

    h_start_doc();

    string rbuf;
    for (_line=1; getline(fp, rbuf); _line++)
    {
        rbuf.append("\n");
        if (XML_Parse(parser, rbuf.c_str(), rbuf.length(), 0) != XML_STATUS_OK)
            return errmsg("File: %s, Line: %d - %s", _fname.c_str(), _line,
                          XML_ErrorString(XML_GetErrorCode(parser)));
        if (err())
            return false;
    }

    if (XML_Parse(parser, 0, 0, 1) != XML_STATUS_OK)
        return errmsg("File: %s, Line: %d (End of file) - %s", _fname.c_str(),
                      _line, XML_ErrorString(XML_GetErrorCode(parser)));

    if (err())
        return false;

    h_end_doc();
    XML_ParserFree(parser);

    return finalize_init();
} // ParamList::readFile
#endif



////////////////////////////////////////////////////////////////////////

bool ParamList::readBrdFile(const char * fname)
{
    ifstream fp(fname);
    if (!fp)
        return errmsg("Can't open board file \"%s\": %s",
                      fname, strerror(errno));
    _Perl->SetSource(fname);

    string perl;
    string rbuf;
    for (int line=1; getline(fp, rbuf); line++)
    { 
        perl += rbuf; 
        perl += "\n";
    };

    if (!_Perl->eval(perl) ) 
        return errmsg ("Failed processing file %s.\n", fname);

    return true;
}


////////////////////////////////////////////////////////////////////////
bool ParamList::writeConfFile(const char *fname, bool delta)
{
    bool ret;
    FILE *fp = fopen(fname, "w");
    if (!fp)
        return errmsg("Can't open file \"%s\": %s", fname, strerror(errno));

    _ini_pref = 
        ";; Mellanox Technologies LTD\n"
        ";; File generated automatically by MIC v-VERSION()\n"
        ";; Date: strftime(%d-%b-%y %H:%M:%S)\n\n";

    if      (! strcasecmp(fname + strlen(fname) - 4 , ".ini")) 
        ret = writeIniFile(fp, delta);
    else if (! strcasecmp(fname + strlen(fname) - 4 , ".brd")) 
        ret = writeBrdFile(fp);
    else
        ret = errmsg("Unsupported suffix for conf file %s. Allowed: .ini , .brd\n", fname);

    fclose(fp);

    return ret;
}




bool ParamList::writeIniFile(FILE* fp, bool delta) 
{

    if (!_ini_pref.empty())
    {
        replaceVersion(_ini_pref);
        replaceDate(_ini_pref);
        fprintf(fp, "%s", _ini_pref.c_str());
    }

    for (list<Group*>::iterator git = grlist.begin(); git != grlist.end(); ++git)
    {
        int params_to_print = 0;
        for (list<Param*>::iterator pit = (*git)->prlist.begin();
             pit != (*git)->prlist.end(); ++pit) {
                
            if (!(*pit)->hidden && !(*pit)->ro) {
                if (delta) {
                    if ((*pit)->get() != (*pit)->def) {
                        params_to_print++;
                    }  
                } else {
                    params_to_print++;
                }
            }
        } 

        if (params_to_print)
        {
            fprintf(fp, "\n[%s]%s", (*git)->name.c_str(),
                    ((*git)->descr.empty() || delta) ? "\n" : (*git)->descr.c_str());

            string assignment_descr_prev;               
            for (list<Param*>::iterator pit = (*git)->prlist.begin();
                 pit != (*git)->prlist.end(); ++pit)
            {
                if (!(*pit)->hidden && !(*pit)->ro) {  
                    string assignment_descr;
		    string group_descr;
		    
                    assignment_descr = (*pit)->GetAssignmentDescription(";;;;; ");
		    
                    if ((*pit)->descr.empty()) {
			// If the parameter does not have a specific description, print 
			// assignment description only if it differs from last printed descr.
			if (assignment_descr == assignment_descr_prev) {
			    assignment_descr = "";
			}   
                    } else {
			group_descr = group_descr + ";;;;; Under [" + (*git)->name + "] section\n";
		    }

		    if (!assignment_descr.empty()) {
			assignment_descr_prev = assignment_descr;
		    }

                    if (delta) {
                        if ((*pit)->get() != (*pit)->def) {
                            //printf("-D- %-20s %10s %10s\n", (*pit)->name.c_str(), (*pit)->get().c_str(), (*pit)->def.c_str());
                            fprintf(fp, "%s = %s\n",     
                                    _print_refnames ? (*pit)->refname.c_str() : (*pit)->name.c_str(),
                                    (*pit)->get().c_str());     
                        }
                    } else {
                    
                        fprintf(fp, "%s%s%s%s%s%s = %s\n", 
                                (*pit)->descr.empty() ? "" : "\n",
                                (*pit)->descr.c_str(),
				group_descr.c_str(),
                                assignment_descr.c_str(),    
                                (*pit)->descr.empty() ? "" : "\n",
                                _print_refnames ? (*pit)->refname.c_str() : (*pit)->name.c_str(),
                                (*pit)->get().c_str());

                    }
                      
                }       
            }
            if (!delta) {           
                fprintf(fp, "\n\n\n");
            }
        }
    }
    return true;
}



bool ParamList::writeBrdFile(FILE* fp) 
{
    fprintf(fp, "my %%REF;\n"
                "tie %%REF, 'MIC';\n\n");

    if (!_ini_pref.empty())
    {
        replaceVersion(_ini_pref);
        replaceDate(_ini_pref);
        fprintf(fp, "# %s", _ini_pref.c_str());  // ???  Check what it prints ???
    }

    for (list<Group*>::iterator git = grlist.begin(); git != grlist.end(); ++git)
    {
        int params_to_print = 0;
        for (list<Param*>::iterator pit = (*git)->prlist.begin();
             pit != (*git)->prlist.end(); ++pit)
            if (!(*pit)->hidden)
                params_to_print++;

        if (params_to_print)
        {
            fprintf(fp, "\n#\n# Group: %s%s", (*git)->name.c_str(),
                    (*git)->descr.empty() ? "\n#\n" : convIni2PerlComments((*git)->descr).c_str());

            for (list<Param*>::iterator pit = (*git)->prlist.begin();
                 pit != (*git)->prlist.end(); ++pit)
            {
                if (!(*pit)->hidden) {
                    char q;
                    if ((*pit)->repr == Param::ASCII || (*pit)->repr == Param::ENUM ) 
                        q = '"';
                    else
                        q = ' ';
                    
                    fprintf(fp, "%s$REF{'%s.%s'} = %c%s%c;\n", convIni2PerlComments((*pit)->descr).c_str(),
                            (*git)->name.c_str(), (*pit)->name.c_str(), q,(*pit)->get().c_str(),q);
                }
            }
            fprintf(fp, "\n\n\n");
        }
    }
    return true;
}


string ParamList::convIni2PerlComments(const string& descr) 
{
    string s = descr;
    int pos = 0;
    while ((pos = s.find('\n', pos)) != -1 && 
           (u_int32_t)pos < s.length() - 1)
    {
        if (s[pos+1] == ';' ) 
            s[pos+1] = '#';
        pos++;
    }
    return s;
}


//-------------------------------------------------------
//------------------------------ Parser utility functions
//-------------------------------------------------------
//
////////////////////////////////////////////////////////////////////////
void ParamList::replaceVersion(string& txt)
{
    const char  *PR = "VERSION(";
    const char  *SF = ")";

    int pos1 = txt.find(PR);
    int pos2;
    if (pos1 != -1  &&  (pos2 = txt.find(SF, pos1)) != -1)
    {
        string    rc;

        rc = txt.substr(0, pos1);
        #ifndef __WIN__
        rc += g_version;
        #endif
        rc += txt.substr(pos2+strlen(SF));
        txt = rc;
    }
} // ParamList::replaceVersion


////////////////////////////////////////////////////////////////////////
void ParamList::replaceDate(string& txt)
{
    const char *PR = "strftime(";
    const char *SF = ")";

    int pos1 = txt.find(PR);
    int pos2;
    if (pos1 != -1  &&  (pos2 = txt.find(SF, pos1)) != -1)
    {
        string    rc;
        const int strftime_buflen = 512;
        char      strftime_buf[strftime_buflen];
        time_t    t = time(0);
        string    frmt = txt.substr(pos1 + strlen(PR),
                                    pos2 - pos1 - strlen(PR));

        strftime(strftime_buf, strftime_buflen, frmt.c_str(), localtime(&t));
        rc = txt.substr(0, pos1);
        rc += strftime_buf;
        rc += txt.substr(pos2+strlen(SF));
        txt = rc;
    }
} // ParamList::replaceDate


////////////////////////////////////////////////////////////////////////
bool ParamList::extractBool(map<string,string>& atts, const char *aname,
                            bool& value)
{
    if (atts.find(aname) == atts.end())
        return errmsg("Attribute \"%s\" not found in attributes map!", aname);

    char *endp;
    int v = strtol(atts[aname].c_str(), &endp, 0);
    if (*endp)
        return errmsg("Invalid value (%s) for attribute \"%s\"",
                      atts[aname].c_str(), aname);
    if (v != 0 && v != 1)
        return errmsg("Invalid value (%s) for attribute \"%s\"",
                      atts[aname].c_str(), aname);
    value = (v == 1);
    return true;
} // ParamList::extractBool

////////////////////////////////////////////////////////////////////////
bool ParamList::extractInt(map<string,string>& atts, const char *aname,
                           int32_t& value)
{
    if (atts.find(aname) == atts.end())
        return errmsg("File: %s, Line: %d - Attribute \"%s\" not found in attributes map!",
                      _fname.c_str(), _line, aname);

    char *endp;
    value = strtol(atts[aname].c_str(), &endp, 0);
    if (*endp)
        return errmsg("File: %s, Line: %d - Invalid value (%s) for attribute \"%s\"",
                      _fname.c_str(), _line, atts[aname].c_str(), aname);
    return true;
} // ParamList::extractInt


////////////////////////////////////////////////////////////////////////
bool ParamList::extractUns(map<string,string>& atts, const char *aname,
                           u_int32_t& value)
{
    if (atts.find(aname) == atts.end())
        return errmsg("File: %s, Line: %d - Attribute \"%s\" not found in attributes map!",
                      _fname.c_str(), _line, aname);

    char *endp;
    value = strtoul(atts[aname].c_str(), &endp, 0);
    if (*endp)
        return errmsg("File: %s, Line: %d - Invalid value (%s) for attribute \"%s\"",
                      _fname.c_str(), _line, atts[aname].c_str(), aname);
    return true;
} // ParamList::extractUns


////////////////////////////////////////////////////////////////////////
bool ParamList::extractInt64(map<string,string>& atts, const char *aname,
                             int64_t& value)
{
    if (atts.find(aname) == atts.end())
        return errmsg("File: %s, Line: %d - Attribute \"%s\" not found in attributes map!",
                      _fname.c_str(), _line, aname);

    char *endp;
    value = strtoll(atts[aname].c_str(), &endp, 0);
    if (*endp)
        return errmsg("File: %s, Line: %d - Invalid value (%s) for attribute \"%s\"",
                      _fname.c_str(), _line, atts[aname].c_str(), aname);
    return true;
} // ParamList::extractInt


////////////////////////////////////////////////////////////////////////
bool ParamList::extractUns64(map<string,string>& atts, const char *aname,
                             u_int64_t& value)
{
    if (atts.find(aname) == atts.end())
        return errmsg("File: %s, Line: %d - Attribute \"%s\" not found in attributes map!",
                      _fname.c_str(), _line, aname);

    char *endp;
    value = strtoull(atts[aname].c_str(), &endp, 0);
    if (*endp)
        return errmsg("File: %s, Line: %d - Invalid value (%s) for attribute \"%s\"",
                      _fname.c_str(), _line, atts[aname].c_str(), aname);
    return true;
} // ParamList::extractUns


////////////////////////////////////////////////////////////////////////
bool ParamList::extractVers(const char *pname, const string& sver, int& ver)
{
    char   *endp;

    int    n = sver.find_first_of('.');
    if (n == -1)
        return errmsg("Parameter \"%s\" has invalid format (no \".\")", pname);

    string svers1 = sver.substr(0, n);
    ver = strtol(svers1.c_str(), &endp, 0);
    if (*endp)
        return errmsg("Parameter \"%s\" has invalid format (major version)", pname);
    return true;
} // ParamList::extractVers


//#define CONVDESCRIPTION_DEBUG
#ifdef CONVDESCRIPTION_DEBUG
#define EMPTY_CHECK(t) do { if (t.empty()) { printf("EMPTY\n"); return; }} while(0)
#else
#define EMPTY_CHECK(t) do { if (t.empty()) return; } while(0)
#endif
////////////////////////////////////////////////////////////////////////
void ParamList::convDescription(string &txt, const bool rem_all_traling_newlines)
{
    int              pos;

#ifdef CONVDESCRIPTION_DEBUG
    printf("Before convDescription :%s:\n", txt.c_str());
#endif

    EMPTY_CHECK(txt);

    // Delete all '\r' symbols (windows to unix conversion)
    while((pos = txt.find('\r')) != -1)
        txt.erase(pos, 1);
    EMPTY_CHECK(txt);

    // Delete trailing whitespaces
    while((pos = txt.find(" \n")) != -1)
        txt.erase(pos, 1);
    EMPTY_CHECK(txt);

    // Delete first '\n'
    if (txt[0] == '\n')
        txt.erase(0, 1);
    EMPTY_CHECK(txt);

    // Delete last '\n' and trailing whitespaces
    if (rem_all_traling_newlines)
    {
        unsigned old_len;
        do
        {
            old_len = txt.length();
            while (txt[txt.length()-1] == '\n')
            {
                txt.resize(txt.length()-1);
                EMPTY_CHECK(txt);
            }
            while (txt[txt.length()-1] == ' ')
            {
                txt.resize(txt.length()-1);
                EMPTY_CHECK(txt);
            }
            while (txt[txt.length()-1] == '\n')
            {
                txt.resize(txt.length()-1);
                EMPTY_CHECK(txt);
            }
        } while (old_len != txt.length());
    }
    else
    {
        if (txt[txt.length()-1] == '\n')
            txt.resize(txt.length()-1);
        EMPTY_CHECK(txt);
        while (txt[txt.length()-1] == ' ')
        {
            txt.resize(txt.length()-1);
            EMPTY_CHECK(txt);
        }
        if (txt[txt.length()-1] == '\n')
            txt.resize(txt.length()-1);
        EMPTY_CHECK(txt);
    }
    EMPTY_CHECK(txt);

    // Convert to comment form (for ini file)
    txt.replace(txt.begin(), txt.begin(), "\n;;;;; ");
    pos = 2;
    while ((pos = txt.find('\n', pos)) != -1)
    {
        // \n can't be last char here (we already ate it), so txt[pos+1] is safe
        if (txt[pos+1] == ';')
        {
            if ((int)txt.length() == pos+3  &&  txt[pos+2] == ';')
            {
                // Text ends with ;;
                txt.resize(txt.length()-2);
                EMPTY_CHECK(txt);
            }
            pos += 1;
        }
        else
        {
            txt.replace(pos, 1, "\n;;;;; ");
            pos += 2;
        }
    }
    txt.append("\n");

#ifdef CONVDESCRIPTION_DEBUG
    printf(" After convDescription :%s:\n", txt.c_str());
#endif
} // ParamList::convDescription


////////////////////////////////////////////////////////////////////////
void ParamList::_debug(const char *tag)
{
    if (!_parser_debug)
        return;
    printf("%s\n", tag);
} // ParamList::_debug


////////////////////////////////////////////////////////////////////////
void ParamList::_debuga(const char *tag, map<string,string>& atts)
{
    if (!_parser_debug)
        return;
    printf("%s - ", tag);
    for (map<string,string>::iterator ait=atts.begin(); ait!=atts.end(); ++ait)
        printf("%s=\"%s\" ", ait->first.c_str(), ait->second.c_str());
    printf("\n");
} // ParamList::_debuga

////////////////////////////////////////////////////////////////////////
bool ParamList::name_is_valid(const string& name)
{
    if ((int)name.find_first_of(NOT_ALLOWED_IN_PARAM_NAME) != -1)
        return errmsg("File: %s, Line: %d - name \"%s\" contains some unallowed symbols",
                      _fname.c_str(), _line, name.c_str());
    return true;
} // ParamList::name_is_valid


//--------------------------------------------------------
//------------------------------ Element specific handlers
//--------------------------------------------------------
//
#define SETERR(args) do { errmsg args ; return; } while(0)
#define MUST_BE_PRESENT(a, m) do {               \
    if (atts.find(a) == atts.end())              \
        SETERR(("File: %s, Line: %d - \"%s\" %s", \
               _fname.c_str(), _line, a, m));     \
    } while(0)

////////////////////////////////////////////////////////////////////////
void ParamList::start_source(std::map<std::string,std::string>& atts)
{
    _fname = atts["file"];
    if (!extractInt(atts, "line", _line))
        return;
    _line--;
} // ParamList::start_source

////////////////////////////////////////////////////////////////////////
void ParamList::end_source()
{
} // ParamList::end_source

void ParamList::start_set_default_prop(std::map<std::string,std::string>& atts)
{
    _debuga("Start set_default_prop", atts); 
    for (map<std::string,std::string>::iterator ait = atts.begin();  ait != atts.end() ; ++ait) 
    {
        bool found = false;
        for (u_int32_t i = 0 ; i < numb_el(_p_def) ; i++) 
        {
            if (_p_def[i].first == ait->first) {
                _p_def[i].second == ait->second;
                found = true;
                break;
            }
        }
        
        if (!found)
            SETERR(("File: %s, Line: %d - Tried to set default to a non existing attribute (%s)",
               _fname.c_str(), _line, ait->first.c_str()));
    }
}

void ParamList::end_set_default_prop() {
    _debug("End set_default_prop");
}

////////////////////////////////////////////////////////////////////////
void ParamList::start_inipref(std::map<std::string,std::string>& atts)
{
    _debuga("Start inipref", atts);

    if (!_ini_pref.empty())
        SETERR(("File: %s, Line: %d - inipref tag may be define only once",
               _fname.c_str(), _line));

    _cur_txt = "";
} // ParamList::start_inipref

////////////////////////////////////////////////////////////////////////
void ParamList::end_inipref()
{
    _debug("End inipref");

    convDescription(_cur_txt);
    _ini_pref = _cur_txt;
    if (_ini_pref[0] == '\n')
        _ini_pref.erase(0, 1);
} // ParamList::end_inipref

////////////////////////////////////////////////////////////////////////
void ParamList::start_perl(std::map<std::string,std::string>& atts)
{
    _debuga("Start Perl", atts);

    _cur_txt = "";
} // ParamList::start_perl

////////////////////////////////////////////////////////////////////////
void ParamList::end_perl()
{
    _debug("End Perl");

    setPerl(_cur_txt);
} // ParamList::end_perl

////////////////////////////////////////////////////////////////////////
void ParamList::start_enum(std::map<std::string,std::string>& atts)
{
    _debuga("Start enum", atts);

    // Enum definition name
    string enum_name = atts["name"];

    // Check for enum duplications
    if (enums.find(enum_name) != enums.end())
        SETERR(("File: %s, Line: %d - \"%s\" enum already defined in "
               "file: %s, line: %d", _fname.c_str(), _line,
               enum_name.c_str(),
               enums[enum_name]->fname.c_str(),
               enums[enum_name]->line));

    // Check for enum tag nesting
    if (_cur_enum)
        SETERR(("File: %s, Line: %d - \"enum\" tag can't be nested.",
               _fname.c_str(), _line));

    // Construct new enum definition
    _cur_enum = new Enum(enum_name);
    _cur_enum->fname = _fname;
    _cur_enum->line = _line;

    // Fill new enum definition
    for (int ai = 0; _char_atts[ai];  ai+=2)
    {
        // Ignore name attribute
        if (!strcmp(_char_atts[ai], "name"))
            continue;

        char      *endp;
        u_int32_t enum_val = strtoul(_char_atts[ai+1], &endp, 0);
        if (*endp)
            SETERR(("File: %s, Line: %d - \"%s\" is invalid \"enum\" "
                   "value specification (for the \"%s\" key)",
                   _fname.c_str(), _line, _char_atts[ai+1], _char_atts[ai]));

        if (!_cur_enum->append(_char_atts[ai], enum_val))
            SETERR(("File: %s, Line: %d - enum error: %s",
                   _fname.c_str(), _line, _cur_enum->err()));
    }
} // ParamList::start_enum

////////////////////////////////////////////////////////////////////////
void ParamList::end_enum()
{
    _debug("End enum");

    enums[_cur_enum->name] = _cur_enum;
    _cur_enum = 0;
} // ParamList::end_enum

////////////////////////////////////////////////////////////////////////
void ParamList::start_group(std::map<std::string,std::string>& atts)
{
    _debuga("Start group", atts);

    // Check for group tag nesting
    if (_cur_group)
        SETERR(("File: %s, Line: %d - \"group\" tag can't be nested.",
               _fname.c_str(), _line));

    // Check for group duplications
    if (grmap.find(atts["name"]) != grmap.end())
        SETERR(("File: %s, Line: %d - \"%s\" group already defined in "
               "file: %s, line: %d", _fname.c_str(), _line,
               atts["name"].c_str(),
               grmap[atts["name"]]->fname.c_str(),
               grmap[atts["name"]]->line));

    // Construct new Group
    _cur_txt = "";
    _first_par = true;
    _cur_group = new Group(atts["name"]);
    _cur_group->fname = _fname;
    _cur_group->line = _line;

} // ParamList::start_group

////////////////////////////////////////////////////////////////////////
void ParamList::end_group()
{
    _debug("End group");

    grmap[_cur_group->name] = _cur_group;
    grlist.push_back(_cur_group);

    _cur_group = 0;
} // ParamList::end_group

////////////////////////////////////////////////////////////////////////

void ParamList::start_param(std::map<std::string,std::string>& atts)
{
    _debuga("Start param", atts);

    // Check for current group
    if (!_cur_group)
        SETERR(("File: %s, Line: %d - \"param\" tag can't be specified "
               "outside of \"group\" tag.", _fname.c_str(), _line));

    // Check for par tag nesting
    if (_cur_param)
        SETERR(("File: %s, Line: %d - \"param\" tag can't be nested.",
               _fname.c_str(), _line));

    // Tavor change: if refname exists, use it instead of name,
    // and ssign refname.
    if (atts["refname"] != "") {
        string msg = "Replaced name \"" + atts["name"] + "\" with refname \"" + atts["refname"] + "\".\n";
        _debug(msg.c_str());
        // atts["name"] = atts["refname"];
    }
    
    
    // Check for parameters duplications in same group
    if (_cur_group->prmap.find(atts["name"]) != _cur_group->prmap.end())
    {
        // Change for Tavor: Rename duplicated params with their index.
        // !!! This is a hack - maybe change it in t2a (add refname ?)

        if (_cur_group->dup_params.find(atts["name"]) == _cur_group->dup_params.end())
            _cur_group->dup_params[atts["name"]] = 0;

        _cur_group->dup_params[atts["name"]] += 1;

        string org_name = atts["name"];
        
        char num_buff[64];
        sprintf(num_buff, "%d", _cur_group->dup_params[atts["name"]]);

        atts["name"] = org_name + ".AutoRename." + num_buff;
        
        string msg = "Param " + org_name + " re-declared. Renamed to " + atts["name"] + "\n";

        bool rename_good = true;
        if (!rename_good)
            SETERR(("File: %s, Line: %d - \"%s\" parameter already defined in "
                   "file: %s, line: %d", _fname.c_str(), _line,
                   atts["name"].c_str(),
                   _cur_group->prmap[atts["name"]]->fname.c_str(),
                   _cur_group->prmap[atts["name"]]->line));
        else
            _debug(msg.c_str());

    }

    // Check refname: if an identical refname or GROUP.Paramname exists - error
    if (! atts["refname"].empty() ) {
        if (refparams.find(atts["refname"]) != refparams.end() ) {
            SETERR(("File: %s, Line: %d - \"%s\" parameter with same refname already defined in "
                   "file: %s, line: %d", _fname.c_str(), _line,
                   atts["name"].c_str(),
                   refparams[atts["refname"]]->fname.c_str(),
                   refparams[atts["refname"]]->line));
        }

        if (params.find(atts["refname"]) != params.end() ) {
            SETERR(("File: %s, Line: %d - \"%s\" parameter with same name as current parameter refname already defined in "
                   "file: %s, line: %d", _fname.c_str(), _line,
                   atts["name"].c_str(),
                   refparams[atts["refname"]]->fname.c_str(),
                   refparams[atts["refname"]]->line));
        }
    }

    // Check name validity
    if (!name_is_valid(atts["name"]))
        return;


    // Fill default attributes
    for (map<string,string>::iterator mit = param_def.begin();
         mit != param_def.end(); ++mit)
    {
        if (atts.find(mit->first) == atts.end())
            atts[mit->first] = mit->second;
    }

    // If it is first parameter in group - add group comment
    if (_first_par)
    {
        convDescription(_cur_txt, true);
        _cur_group->descr = _cur_txt;
        _first_par = false;
    }

    // Construct new Param
    _cur_txt = "";
    _cur_param = new Param(atts["name"]);
    _cur_param->fname = _fname;
    _cur_param->line = _line;
    _cur_param->group = _cur_group;

    // Fill it
    // -------

    // Type
    if (types_def.find(atts["type"]) == types_def.end())
        SETERR(("File: %s, Line: %d - type \"%s\" is invalid",
               _fname.c_str(), _line, atts["type"].c_str()));
    _cur_param->type = Param::Type(types_def[atts["type"]]);

    //refname
    if ( atts["refname"] != "" )
        _cur_param->refname = atts["refname"];


    // Repr
    string rname = atts["repr"];
    string subrname;
    int    cpos = rname.find(':');
    if (cpos != -1)
    {
        subrname = rname.substr(cpos+1);
        rname.erase(cpos);
    }

    if (repr_def.find(rname) == repr_def.end())
        SETERR(("File: %s, Line: %d - type \"%s\" is invalid",
               _fname.c_str(), _line, rname.c_str()));
    _cur_param->repr = Param::Repr(repr_def[rname]);

    // Hidden
    if (!extractBool(atts, "hidden", _cur_param->hidden))
        return;

    // ro
    if (!extractBool(atts, "ro", _cur_param->ro))
        return;

    // Description will be filled later (in end_param handler)

    // min and max may be specifed only togehter!
    if ((atts.find("min") != atts.end() && atts.find("max") == atts.end()) ||
        (atts.find("min") == atts.end() && atts.find("max") != atts.end()))
        SETERR(("File: %s, Line: %d - \"min\" and \"max\" attributes may be"
               " specifed only together", _fname.c_str(), _line));

    // Expr:
    _cur_param->expr = atts["expr"];
    
    // check_r
    _cur_param->check_r = (atts.find("min") != atts.end());

    if (_cur_param->check_r &&
        (_cur_param->repr != Param::INT && _cur_param->repr != Param::UNS &&
         _cur_param->repr != Param::INT64 && _cur_param->repr != Param::UNS64))
        SETERR(("File: %s, Line: %d - \"min\" and \"max\" attributes may be"
               " specifed only if type is \"uns\" or \"int\"",
               _fname.c_str(), _line));

    // Follow parametrs are filled according to representation
    switch(_cur_param->repr)
    {
    case Param::INT:
    {
        int32_t smin,smax;
        if (_cur_param->check_r)
        {
            if (!extractInt(atts, "min", smin))
                return;
            if (!extractInt(atts, "max", smax))
                return;
            _cur_param->min = (u_int32_t)smin;
            _cur_param->max = (u_int32_t)smax;
        }
        break;
    }
    case Param::UNS:
        if (_cur_param->check_r)
        {
            if (!extractUns(atts, "min", _cur_param->min))
                return;
            if (!extractUns(atts, "max", _cur_param->max))
                return;
        }
        break;
    case Param::INT64:
    {
        int64_t smin,smax;
        if (_cur_param->check_r)
        {
            if (!extractInt64(atts, "min", smin))
                return;
            if (!extractInt64(atts, "max", smax))
                return;
            _cur_param->lmin = (u_int64_t)smin;
            _cur_param->lmax = (u_int64_t)smax;
        }
        break;
    }
    case Param::UNS64:
        if (_cur_param->check_r)
        {
            if (!extractUns64(atts, "min", _cur_param->lmin))
                return;
            if (!extractUns64(atts, "max", _cur_param->lmax))
                return;
        }
        break;
    case Param::ENUM:
        if (enums.find(subrname) == enums.end())
            SETERR(("File: %s, Line: %d - enum type \"%s\" not found",
                   _fname.c_str(), _line, subrname.c_str()));
        _cur_param->enump = enums[subrname];
        break;
    case Param::ASCII:
        MUST_BE_PRESENT("length", "attribute is mandatory for ASCII parameters");
        if (!extractUns(atts, "length", _cur_param->slength))
            return;
        break;
    case Param::BOOL:
        break;
    }

    // Follow parametrs are filled according to type
    u_int32_t offs, addr;

    switch(_cur_param->type)
    {
    case Param::C:
        //--if (_cur_param->repr == Param::ASCII)
        //--    SETERR(("File: %s, Line: %d - ASCII parameters of C type doesn't supported yet.",
        //--           _fname.c_str(), _line);
        MUST_BE_PRESENT("exe", "attribute is mandatory for C parameters");
        MUST_BE_PRESENT("addr", "attribute is mandatory for C parameters");
        
        if (!extractUns(atts, "addr", addr) || !extractUns(atts, "offset", offs))
            return;

        _cur_param->offset = addr + offs;

        if (_cur_param->offset & 0x03)
            SETERR(("File: %s, Line: %d - addr should be 4-bytes aligned.",
                   _fname.c_str(), _line));
        _cur_param->sect = (*_mfile)[atts["exe"].c_str()];
        if (!_cur_param->sect)
            SETERR(("File: %s, Line: %d - Executable \"%s\" not found.",
                   _fname.c_str(), _line, atts["exe"].c_str()));
        if (_cur_param->sect->addr + _cur_param->sect->data.size() <=
            _cur_param->offset)
            SETERR(("File: %s, Line: %d - Executable \"%s\" too short for this addr.",
                   _fname.c_str(), _line, atts["exe"].c_str()));
        if (_cur_param->offset < _cur_param->sect->addr)
            SETERR(("File: %s, Line: %d - Executable \"%s\" - offset less than start address",
                   _fname.c_str(), _line, atts["exe"].c_str()));
        break;
    case Param::CRSPACE:
        MUST_BE_PRESENT("addr", "attribute is mandatory for CRSPACE parameters");
        MUST_BE_PRESENT("bitoffs", "attribute is mandatory for CRSPACE parameters");
        MUST_BE_PRESENT("size", "attribute is mandatory for CRSPACE parameters");
        MUST_BE_PRESENT("boot2", "attribute is mandatory for CRSPACE parameters");
        
        if (!extractUns(atts, "addr", addr) || !extractUns(atts, "offset", offs))
            return;

        _cur_param->offset = addr + offs;
        
                
        if (_cur_param->offset & 0x03)
            SETERR(("File: %s, Line: %d - addr should be 4-bytes aligned.",
                   _fname.c_str(), _line));
        if (!extractUns(atts, "bitoffs", _cur_param->bitoffs))
            return;
        if (!extractUns(atts, "size", _cur_param->bitsize))
            return;
        if (!extractBool(atts, "boot2", _cur_param->boot2))
            return;
        if (_cur_param->repr == Param::ASCII)
        {
            if (_cur_param->bitoffs & 0x07)
                SETERR(("File: %s, Line: %d - ASCII parameters should be byte aligned.",
                       _fname.c_str(), _line));
        }
        else
        {
            if (_cur_param->bitoffs + _cur_param->bitsize > 32)
                SETERR(("File: %s, Line: %d - invalid bitoffs/size combination.",
                       _fname.c_str(), _line));
        }
        break;
    case Param::DUMMY:
        break;
    }

    // Special parameters (tag and offs attributes)
    if (!extractInt(atts, "tag", _cur_param->tag_id))
        return;
    if (_cur_param->tag_id)
    {
        MUST_BE_PRESENT("offs", "attribute is mandatory if \"tag\" is specified.");
        if (!extractUns(atts, "offs", _cur_param->tag_offs))
            return;
        if (_cur_param->type != Param::DUMMY)
            SETERR(("File: %s, Line: %d - Type of special parameter \"%s\" must "
                   "be dummy.", _fname.c_str(), _line, _cur_param->name.c_str()));
    }

    // Allow GEO id attribute
    if (!extractBool(atts, "allow_geo", _cur_param->allow_geo))
        return;
    if (_cur_param->allow_geo  &&  _cur_param->type == Param::C)
        SETERR(("File: %s, Line: %d - GEO id can't be allowed for parameters "
               "of type \"C\".", _fname.c_str(), _line));

    // And finally assign default value
    string def_value = atts["def"];
    if (def_value != "NONE") {
        if (!_cur_param->assign(def_value))
            SETERR(("File: %s, Line: %d - %s", _fname.c_str(), _line, _cur_param->err()));

        // Don't use te default value itself to prevent formatting issues (0xF != 0xf) 
        _cur_param->def = _cur_param->get(); 
    }   
} // ParamList::start_param

////////////////////////////////////////////////////////////////////////
void ParamList::end_param()
{
    _debug("End param");

    _cur_group->prmap[_cur_param->name] = _cur_param;
    _cur_group->prlist.push_back(_cur_param);
    params[_cur_group->name + "." + _cur_param->name] = _cur_param;

    if (_cur_param->refname != "") {
        refparams[_cur_param->refname] = _cur_param;
    }

    // Description
    convDescription(_cur_txt);
    _cur_param->descr = _cur_txt;

    _cur_param = 0;
} // ParamList::end_param


//--------------------------------------------------------------
//------------------------------ Real handlers (expat callbacks)
//--------------------------------------------------------------
//
////////////////////////////////////////////////////////////////////////
void ParamList::h_start_el(const XML_Char *name, const XML_Char **char_atts)
{
    // Something failed before - do nothing here
    if (err())
        return;

    // Store original attributes in original order - some element needs that
    _char_atts = (XML_Char **)char_atts;

    // Prepare attributes map
    map<string,string> atts;
    for (int ai = 0; char_atts[ai];  ai+=2)
        atts[char_atts[ai]] = char_atts[ai+1];

    // Check mandatory attributes
    map<string,vector<string> >::iterator mit = mand_atts.find(name);
    if (mit != mand_atts.end())
    {
        vector<string> lst = mit->second;
        for (vector<string>::iterator it=lst.begin(); it != lst.end(); ++it)
            if (atts.find(*it) == atts.end())
                SETERR(("File: %s, Line: %d - for \"%s\" element attribute "
                       "\"%s\" is mandatory",
                       _fname.c_str(), _line, name, it->c_str()));

    }

    // Now call handler per element name
    map<string,start_el_hnd>::iterator hit = start_h.find(name);
    if (hit != start_h.end())
    {
        (this->*(hit->second))(atts);
        return;
    }

    // All other (unhandled) tags
    if (find(extra_tags.begin(), extra_tags.end(), name) == extra_tags.end())
        //SETERR(("File: %s, Line: %d - tag \"%s\" is invalid.",
               //_fname.c_str(), _line, name);
        printf("%s:%d: Unhandeled tag start - ignored: %s\n",_fname.c_str(), _line, name);

} // ParamList::h_start_el

////////////////////////////////////////////////////////////////////////
void ParamList::h_end_el(const XML_Char *name)
{
    if (err())
    {
        // Something failed before - cleanup up
        delete _cur_enum;
        _cur_enum = 0;
        delete _cur_group;
        _cur_group = 0;
        delete _cur_param;
        _cur_param = 0;
        return;
    }

    // Now call handler per element name
    map<string,end_el_hnd>::iterator hit = end_h.find(name);
    if (hit != end_h.end())
    {
        (this->*(hit->second))();
        return;
    }

    // All other (unhandled) tags
    if (find(extra_tags.begin(), extra_tags.end(), name) == extra_tags.end())
        //SETERR(("File: %s, Line: %d - tag \"%s\" is invalid.", _fname.c_str(), _line, name);
        printf("%s:%d: Unhandeled tag end   - ignored: %s\n",_fname.c_str(), _line, name);
} // ParamList::h_end_el

////////////////////////////////////////////////////////////////////////
void ParamList::h_chars(const XML_Char *s, int len)
{
    // Something failed before - do nothing here
    if (err())
        return;

    _cur_txt += string(s, len);
} // ParamList::h_chars

////////////////////////////////////////////////////////////////////////
void ParamList::h_start_doc()
{
    err_clear();
    _cur_group = 0;
    _cur_param = 0;
    _cur_enum = 0;
    _cur_txt = "";
} // ParamList::h_start_doc


////////////////////////////////////////////////////////////////////////
void ParamList::h_end_doc()
{
} // ParamList::h_end_doc


////////////////////////////////////////////////////////////////////////
bool ParamList::finalize_init()
{
    /*
     * All parameters are parsed here.
     * Now we check their consistency
     * and do some finalizing
     */


    // Check MIC's PREP supported version vs. prepA2FW versions
    #ifdef __WIN__
    char* g_supported_prep_version = "1.0.0"; // TODO - FIX EXTERN PROBLOMS !!!
    #else
    extern char* g_supported_prep_version;
    #endif
    map<string, Param*>::iterator vers_it = params.find(_auto_vers);
    if (vers_it == params.end())
        return errmsg("\"%s\" parameter must be specified.", _auto_vers);
    Param *ap = vers_it->second;
    if (ap->repr != Param::ASCII)
        return errmsg("Parameter \"%s\" should be ASCII.", _auto_vers);
    int   prep_vers, supported_vers;
    if (!extractVers(_auto_vers, ap->get(), prep_vers))
        return false;
    if (!extractVers("Supported prep version", g_supported_prep_version , supported_vers))
        return false;

    if (prep_vers > supported_vers)
        return errmsg("The FW was prepared with FW prepation tool v%s which has major\n"
                      "       version (%d) that is greater than MIC's "
                      "       supported FW prepation tool version (%d)\n"
                      "       Probably you need to upgrade the MIC tool.",
                      ap->get().c_str(), prep_vers, supported_vers);

    // Backward compatible expr evaluator:
    MicExpr mic_expr(this);
    
    if (!mic_expr.eval_all()) {
        return errmsg(mic_expr.err());
    }
        
    _Perl->pinit(this);

    // Now clean all definitions (where_f field). Because its OK
    // do assign something to parameter in init and then reassign
    // it later.
    clear_def_source();
    return true;
} // ParamList::finalize_init


////////////////////////////////////////////////////////////////////////
void ParamList::clear_def_source()
{
    for (map<string, Param*>::iterator pit = params.begin(); pit != params.end(); ++pit)
        for (int geo_idx=0; geo_idx < pit->second->values.size(); geo_idx++)
        {
            Param::Value v = pit->second->values.value(geo_idx);
            v.where_f = "";
            pit->second->values.set_value(geo_idx, v);
        }
} // ParamList::clear_def_source


////////////////////////////////////////////////////////////////////////
bool ParamList::finalize_eval()
{
    // Backward compatible expr evaluator:
    MicExpr mic_expr(this);
    
    if (!mic_expr.eval_all()) {
        return errmsg(mic_expr.err());
    }

    _Perl->peval(this);
    return true;
} // ParamList::finalize_eval
