/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  Param.cpp - Paramater/Parameters group definition class
 *
 *  Version: $Id: Param.cpp 2752 2006-01-19 14:40:17Z mst $
 *
 */
#define __EXTENSIONS__
#include <stdio.h>
#include <sstream>

#include "Param.h"
#include "utils.h"

namespace std {}; using namespace std;

//------------------------------------------
//------------------------------ Param class
//------------------------------------------
//
DeviceType Param::dev_type;
bool       Param::warn;

////////////////////////////////////////////////////////////////////////
bool Param::assign(const string& sval, const string& w_f, const int w_l,
                   const u_int32_t geo, const bool ext_assignment)
{
    char      *endp;
    string    s_old = values[geo].s;
    u_int32_t i_old = values[geo].i;
    u_int64_t l_old = values[geo].l;

    string      int_s;
    const char* int_str = sval.c_str();

    if (!allow_geo  &&  geo != GEO_DEF)
        return errmsg("Parameter %s.%s doesn't accept non-default GEO ID (%08x)",
                      group->name.c_str(), name.c_str(), geo);

    if ((repr == INT || repr == UNS || repr == INT64 || repr == UNS64) && 
	 sval.find_first_of(".") != string::npos) 
    {
	// !!! If given val for int is a float, truncate the part after the dot.
        if (warn) {
            printf("-W- Trying to assign float value (%s) to an integer parameter (%s). Fraction truncated.\n",
                   sval.c_str(),
                   name.c_str());
        }

	int_s = sval; 
	int_s.erase(int_s.find_first_of(".")); 
	int_str = int_s.c_str();
    }

    switch(repr)
    {
    case INT:
    {
        int32_t val = strtol(int_str, &endp, 0);
        if (*endp)
            return errmsg("Value \"%s\" is invalid for integer parameter",
                          int_str);
        if (check_r)
        {
            int32_t smin = (int32_t)min;
            int32_t smax = (int32_t)max;
            if (val < smin || val > smax)
                return errmsg("Value %d not in range (%d-%d)", val, smin, smax);
        }
        values[geo].i = (u_int32_t)val;
        break;
    }
    case UNS:
        values[geo].i = strtoul(int_str, &endp, 0);
        if (*endp)
            return errmsg("Value \"%s\" is invalid for unsigned parameter",
                          int_str);
        if (check_r  &&  (values[geo].i < min || values[geo].i > max))
            return errmsg("Value 0x%x not in range (0x%x-0x%x)",
                          values[geo].i, min, max);
        break;
    case INT64:
    {
        int64_t val = strtoll(int_str, &endp, 0);
        if (*endp)
            return errmsg("Value \"%s\" is invalid for integer parameter",
                          int_str);
        if (check_r)
        {
            int64_t smin = (int64_t)lmin;
            int64_t smax = (int64_t)lmax;
            if (val < smin || val > smax)
                return errmsg("Value %"U64L"d not in range (%"U64L"d-%"U64L"d)", val, smin, smax);
        }
        values[geo].l = (u_int64_t)val;
        break;
    }
    case UNS64:
        values[geo].l = strtoull(int_str, &endp, 0);
        if (*endp)
            return errmsg("Value \"%s\" is invalid for unsigned parameter",
                          int_str);
        if (check_r  &&  (values[geo].l < lmin || values[geo].l > lmax))
            return errmsg("Value 0x%"U64L"x not in range (0x%"U64L"x-0x%"U64L"x)",
                          values[geo].l, lmin, lmax);
        break;
    case ENUM:
        if (enump->vals.find(sval) == enump->vals.end())
            return errmsg("Value \"%s\" is invalid for enum of type \"%s\"",
                          sval.c_str(), enump->name.c_str());
        values[geo].i = enump->vals[sval];
        values[geo].s = sval;
        break;
    case ASCII:
        if (sval.length() >= slength)
            return errmsg("Value \"%s\" is too long", sval.c_str());
        values[geo].s = sval;
        break;
    case BOOL:
        if (!strcasecmp(sval.c_str(), "true"))
            values[geo].i = 1;
        else if (!strcasecmp(sval.c_str(), "false"))
            values[geo].i = 0;
        else
        {
            int32_t val = strtol(sval.c_str(), &endp, 0);
            if (*endp)
                return errmsg("Value \"%s\" is invalid for bool parameter",
                              sval.c_str());
            if (val != 0  &&  val != 1)
                return errmsg("Value \"%s\" is invalid for bool parameter",
                              sval.c_str());
            values[geo].i = (val == 1);
        }
        break;
    }

    if (warn && !values[geo].where_f.empty())
    {
        printf("-W- Parameter %s.%s already assigned in %s. ",
               group->name.c_str(), name.c_str(),values[geo].where_f.c_str());
        if (values[geo].where_l != -1)
            printf(" (line:%d)", values[geo].where_l);
        switch (repr)
        {
        case ASCII:
            printf("Old = \"%s\"; new = \"%s\"\n",
                   s_old.c_str(), values[geo].s.c_str());
            break;
        case INT:
            printf("Old = %d; new = %d\n",
                   i_old, values[geo].i);
            break;
        case UNS:
            printf("Old = 0x%x; new = 0x%x\n",
                   i_old, values[geo].i);
            break;
        case INT64:
            printf("Old = %"U64L"d; new = %"U64L"d\n",
                   l_old, values[geo].l);
            break;
        case UNS64:
            printf("Old = 0x%"U64L"x; new = 0x%"U64L"x\n",
                   l_old, values[geo].l);
            break;
        case BOOL:
            printf("Old = %s; new = %s\n",
                   i_old?"true":"false", values[geo].i?"true":"false");
            break;
        case ENUM:
            printf("Old = 0x%x (%s); new = 0x%x (%s)\n",
                   i_old, s_old.c_str(), values[geo].i, values[geo].s.c_str());
            break;
        }
    }

    values[geo].where_f = w_f;
    values[geo].where_l = w_l;
    values[geo].ext_val = ext_assignment;
    return true;
} // Param::assign

////////////////////////////////////////////////////////////////////////
string Param::get(const u_int32_t geo)
{
    char buf[64];

    if (!values.contains(geo))
    {
        errmsg("Parameter %s.%s doesn't contain GEO ID %08x",
               group->name.c_str(), name.c_str(), geo);
        return "";
    }

    switch(repr)
    {
    case INT:
        sprintf(buf, "%d", values[geo].i);
        return buf;
    case UNS:
        sprintf(buf, "0x%x", values[geo].i);
        return buf;
    case INT64:
        sprintf(buf, "%"U64L"d", values[geo].l);
        return buf;
    case UNS64:
        sprintf(buf, "0x%"U64L"x", values[geo].l);
        return buf;
    case ENUM:
    case ASCII:
        return values[geo].s;
    case BOOL:
        return values[geo].i ? "true" : "false";
    }
    return "";  // Just to shutup stupid compiler warning

} // Param::get

////////////////////////////////////////////////////////////////////////
string Param::geti(const u_int32_t geo)
{
    char buf[64];

    if (!values.contains(geo))
    {
        errmsg("Parameter %s.%s doesn't contain GEO ID %08x",
               group->name.c_str(), name.c_str(), geo);
        return "";
    }

    switch(repr)
    {
    case INT:
        sprintf(buf, "%d", values[geo].i);
        return buf;
    case ENUM:
    case UNS:
        if (dev_type == MT47396)
	    sprintf(buf, "0x%x", values[geo].i);
	else
	    sprintf(buf, "%d", values[geo].i);
        return buf;
    case INT64:
        sprintf(buf, "%"U64L"d", values[geo].l);
        return buf;
    case UNS64:
	if (dev_type == MT47396)  
	    sprintf(buf, "0x%"U64L"x", values[geo].l);
	else
	    sprintf(buf, "%"U64L"d", values[geo].l);
        return buf;
    case ASCII:
        return values[geo].s;
    case BOOL:
        return values[geo].i ? "1" : "0";
    }
    return "";  // Just to shutup stupid compiler warning
} // Param::geti

////////////////////////////////////////////////////////////////////////
u_int32_t Param::get32(const u_int32_t geo)
{
    if (!values.contains(geo))
    {
        errmsg("Parameter %s.%s doesn't contain GEO ID %08x",
               group->name.c_str(), name.c_str(), geo);
        return 0;
    }

    switch(repr)
    {
    case INT:
    case ENUM:
    case UNS:
        return values[geo].i;
    case BOOL:
        return values[geo].i ? 1 : 0;
    default:
        errmsg("Parameter %s.%s can't be retrieved as 32-bit integer.",
               group->name.c_str(), name.c_str());
        return 0;
    }

    return 0; // Just to shutup stupid compiler warning
} // Param::get32

////////////////////////////////////////////////////////////////////////
u_int64_t Param::get64(const u_int32_t geo)
{
    if (!values.contains(geo))
    {
        errmsg("Parameter %s.%s doesn't contain GEO ID %08x",
               group->name.c_str(), name.c_str(), geo);
        return 0;
    }

    switch(repr)
    {
    case INT:
    case ENUM:
    case UNS:
        return values[geo].i;
    case BOOL:
        return values[geo].i ? 1 : 0;
    case INT64:
    case UNS64:
        return values[geo].l;
    default:
        errmsg("Parameter %s.%s can't be retrieved as 64-bit integer.",
               group->name.c_str(), name.c_str());
        return 0;
    }

    return 0; // Just to shutup stupid compiler warning
} // Param::get64


string Param::GetAssignmentDescription(const string& line_prefix) {
    string sret;
    ostringstream s;
    string base;


    switch (repr) {
    case ENUM:
        sret = line_prefix;

        sret += "Enum parameter. Possible values: ";
        //FOR(list<string>, it, enump->keys) {
        for (list<string>::iterator it = enump->keys.begin(); it != enump->keys.end(); ++it) {
            sret += *it ;
            sret += ", ";
        }

        sret.erase(sret.size() -2);

        sret += "\n";

        break;

    case BOOL:

        sret = line_prefix;
        sret += "Boolean parameter. Possible values: true, false .\n";

        break;

    case UNS:
    case INT:
    case UNS64:
    case INT64:

        if (!check_r) {
            // Assign min and max to full range:
            if (repr == UNS || repr == INT) {
                min = 0;
                max = (u_int32_t)-1;
            } else {
                lmin = 0;
                lmax = (u_int64_t)-1;
            }
        }

        s << "Integer parameter. Values range ";
        if (repr == UNS || repr == UNS64) {
            s << hex; 
            base = "0x";
        }
        
        if (repr == UNS || repr == INT) {
            s << ": " << base << min << " - " << base << max << ".\n";
        } else {
            s << "(64 bits)";
            s << ": " 
                #ifndef __WIN__
                << base 
                << lmin 
                << " - " 
                << base 
                << lmax 
                #else
                // TODO: FIX
                << "NOT YET IMPLEMENTED - MIC on windows does not support 64 bits parameters range description."
                #endif 
                << ".\n";
        }
        
        sret = line_prefix + s.str();
        

        break;
    case ASCII:

        s << "ASCII parameter. Max length: " << slength << " characters.\n";
        sret = line_prefix + s.str();
        break;
    }

    return sret;
}


//-----------------------------------------
//------------------------------ Enum class
//-----------------------------------------
//
////////////////////////////////////////////////////////////////////////
bool Enum::append(const std::string& enum_name, const u_int32_t enum_val)
{
    if (vals.find(enum_name) != vals.end())
        return errmsg("Value %s already defined", enum_name.c_str());

    vals[enum_name] = enum_val;
    keys.push_back(enum_name);
    return true;
} // Enum::append
