/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  MultiFile.cpp - Multifile (read-only file with subsections) class
 *
 *  Version: $Id: MultiFile.cpp 2752 2006-01-19 14:40:17Z mst $
 *
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <zlib.h>

#include <iostream>
#include <fstream>

#include "MultiFile.h"
#include "utils.h"
#include "compatibility.h"

namespace std {}; using namespace std;

////////////////////////////////////////////////////////////////////////
static bool getline(gzFile& gf, ifstream& fp, string& rbuf, bool compressed)
{
    const int BLEN = 1024;
    char      buf[BLEN];

    rbuf = "";
    if (compressed)
    {
        do
        {
            char *p = gzgets(gf, buf, BLEN);
            if (p)
                rbuf += buf;
            else
                return !rbuf.empty();
        } while (strlen(buf) >= BLEN-1);
        return true;
    }
    else
    {
        bool rc;
        if( getline(fp, rbuf)) 
            rc = true;
        else 
            rc = false;

        rbuf += '\n';
        return rc;
    }
} // getline


////////////////////////////////////////////////////////////////////////
MultiFile::MultiFile(DeviceType dev_type) : _fname(0), _dev_type(dev_type)
{
    sprintf(_magic, MAGIC_STRING_FORMAT, _dev_type);
} // MultiFile::MultiFile


////////////////////////////////////////////////////////////////////////
MultiFile::~MultiFile()
{
    clear();
} // MultiFile::~MultiFile


////////////////////////////////////////////////////////////////////////
void MultiFile::clear()
{
    // Delete sections
    for (map<string, MSection*>::iterator mit = _sects.begin();
         mit != _sects.end(); ++mit)
        delete mit->second;
    _sects.clear();

    _fname = 0;
} // MultiFile::clear


////////////////////////////////////////////////////////////////////////
MSection *MultiFile::operator[](const char *name) const
{
    map<string, MSection*>::const_iterator s = _sects.find(name);
    return s == _sects.end() ? 0 : s->second;
} // MultiFile::operator


////////////////////////////////////////////////////////////////////////
list<string> MultiFile::sections() const
{
    list<string> rc;
    for(map<string, MSection*>::const_iterator i=_sects.begin(); i!=_sects.end(); ++i)
        if (i->second->bin)
            rc.push_back(i->first);
    return rc;
} // MultiFile::sections

////////////////////////////////////////////////////////////////////////
list<string> MultiFile::non_boot_sections() const
{
    list<string> rc;
    for(map<string, MSection*>::const_iterator i=_sects.begin(); i!=_sects.end(); ++i)
        if (i->second->bin && i->second->crstart > MSection::BOOT3)
            rc.push_back(i->first);
    return rc;
} // MultiFile::sections

////////////////////////////////////////////////////////////////////////
bool MultiFile::read(const char *fname)
{
    gzFile   gf;
    ifstream fp;

    clear();

    _fname = (char *)fname;

    bool compressed = false;
    if (_magic)
    {
        // Open and try read - may be file compressed?
        // TODO: May leak.
        char* buf = new char[strlen(_magic)+1];
        int fd = COMP_OPEN(_fname, O_RDONLY);
        if (fd == -1)
            return errmsg("Can't open file \"%s\": %s", _fname, strerror(errno));

        // Probe uncompressed
        compressed = !(COMP_READ(fd, buf, strlen(MAGIC_STRING_PREFIX)) == (int)strlen(MAGIC_STRING_PREFIX)
                       && !strncmp(buf, _magic, strlen(MAGIC_STRING_PREFIX)));
        COMP_CLOSE(fd);

        // Probe compressed, do correct open (or via ifstream or via gzopen)
        if (compressed)
        {
            gf = gzopen(_fname, "rb");
            if (!gf)
            {
                int        errnum;
                const char *errtxt = gzerror(gf, &errnum);
                return errmsg("Can't open compressed file \"%s\": %s",_fname,
                              errnum == Z_ERRNO ? strerror(errno) : errtxt);
            }
            if (gzgets(gf, buf, strlen(MAGIC_STRING_PREFIX)+1) && !strcmp(buf, MAGIC_STRING_PREFIX))
            {
                compressed = true;
                gzrewind(gf);
            }
            else
                return errmsg("File \"%s\" has invalid format.", _fname);
        }
        else
        {
            fp.open(_fname);
            if (!fp)
                return errmsg("Can't open file \"%s\": %s", _fname, strerror(errno));
        }

        delete [] buf;

        //
        // We know the format (compressed or not). Now check Device ID match.
        //

        string file_magic;
        getline(gf, fp, file_magic, compressed);
        file_magic.erase(file_magic.size() - 1 , file_magic.size() - 1); // remove newline

        if (file_magic != _magic) {
            return errmsg("Bad FW file stamp. Expected \"%s\", found \"%s\"", 
                          _magic + strlen(MAGIC_STRING_PREFIX),
                          file_magic.c_str() + strlen(MAGIC_STRING_PREFIX));
        }

        if (compressed) {
            gzrewind(gf);
        } else {
            fp.seekg (0, ios::beg);
        }

    }

    string   rbuf;
    string   sect_name;
    MSection *sect=0;
    for (int line=1; getline(gf, fp, rbuf, compressed); ++line)
    {
        bool sect_end   = false;
        bool sect_start = false;
        bool sect_bin   = false;

        // Examine start/end of section
        if (rbuf.find("<section") == 0)
        {
            sect_start = true;
            sect_bin = false;
        }
        else if (rbuf.find("<bsection") == 0)
        {
            sect_start = true;
            sect_bin = true;
        }
        else if (rbuf.find("</section>") == 0)
        {
            sect_end = true;
            sect_bin = false;
        }
        else if (rbuf.find("</bsection>") == 0)
        {
            sect_end = true;
            sect_bin = true;
        }

        if (sect_start)
        {
            /*
             * Section start
             */
            if (sect)
                return errmsg("File: %s, Line:%d - nested sections not allowed.",
                              _fname, line);

            // Extract name and create section
            if (!get_section_header(line, rbuf, "name", sect_name ))
                return false;

            sect = new MSection(sect_name.c_str());
            sect->bin = sect_bin;
            sect->start_line = line+1;

            // Extract address (for binary sections)
            if (sect->bin)
            {
		bool ret = true;
		ret &=  get_section_header   (line, rbuf, "beg"    , sect->addr);
		ret &=  get_section_header   (line, rbuf, "start"  , sect->start);

		if (_dev_type != MT47396)   
		    ret &=  get_section_header_cr(line, rbuf, "crstart", sect->crstart);
                    
		if (ret == false)
		    return false;
            }
        }

        else if (sect_end)
        {
            /*
             * Section end
             */
            if (!sect)
                return errmsg("File: %s, Line:%d - unmatched sections end.",
                              _fname, line);
            if (sect->bin != sect_bin)
                return errmsg("File: %s, Line:%d - unmatched section/bsection tags",
                              _fname, line);

            _sects[sect_name] = sect;
            sect = 0;
        }
        else if (sect)
        {
            /*
             * Inside section
             */

            if (sect->bin)
            {
                const char *numbers = "1234567890abcdefABCDEF";

                // Truncate garbage/comments
                int n = rbuf.find_first_not_of("\t 1234567890abcdefABCDEF");
                if (n != -1)
                    rbuf = rbuf.substr(0, n);

                for(int n1=0; ;)
                {
                    char num[3];

                    n1 = rbuf.find_first_of(numbers, n1);
                    if (n1 == -1)
                        break;
                    int n2 = rbuf.find_first_not_of(numbers, n1);
                    if (n2-n1 != 2)
                        return errmsg("File: %s, Line:%d - invalid binary "
                                      "section format", _fname, line);
                    num[0] = rbuf[n1];
                    num[1] = rbuf[n1+1];
                    num[2] = 0;
                    sect->data.push_back(strtoul(num, 0, 16));
                    rbuf = rbuf.substr(n2+1);
                }
            }
            else
                sect->lines.push_back(rbuf);
        }
        else
        {
            /*
             * Outside section
             */

            // Do nothing
        }
    }

    // Close all (it isn't necessary for ifstream, but it is
    // necessary for gzFile)
    if (compressed)
        gzclose(gf);
    else
        fp.close();

    return true;
} // MultiFile::read


bool MultiFile::get_section_header   (int line, const string& header, const char* attr, string&  val) 
{
    string look_for = attr;
    look_for       += "=\"";

    const int N     = look_for.length();
    int n1, n2;

    if ((n1 = header.find(look_for)) == -1)
        return errmsg("File: %s, Line:%d - \"%s\" is omitted.",
                      _fname, line, attr);
    if ((n2 = header.find('"', n1+N)) == -1)
        return errmsg("File: %s, Line:%d - \"%s\" is invalid.",
                      _fname, line, attr);
    val = header.substr(n1+N, n2-n1-N);
    return true;
}

bool MultiFile::get_section_header   (int line, const string& header, const char* attr, u_int32_t& val)
{
    char      *endp;
    string sval;

    if (!get_section_header(line, header, attr, sval))
        return false;

    val = strtoul(sval.c_str(), &endp, 0);

    if (*endp)
        return errmsg("File: %s, Line:%d - \"%s\" is invalid.",
                      _fname, line, attr);
    return true;
}

bool MultiFile::get_section_header_cr(int line, const string& header, const char* attr, u_int32_t& val) 
{
    if (string(attr) != "crstart")
        return errmsg("Internal error: MultiFile::get_section_header_cr can be called for crstart attribute only. Was called for %s attr.\n", attr);

    string crstart;
    if (!get_section_header(line, header, attr, crstart))
        return false;

    if      (crstart == "BOOT0")
        val = MSection::BOOT0;
    else if (crstart == "BOOT2")
        val = MSection::BOOT2;
    else if (crstart == "BOOT3")
        val = MSection::BOOT3;
    else 
        return get_section_header(line, header, attr, val);

    return true;
}

