/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  Image.cpp - FW image manipulation class
 *
 *  Version: $Id: Image.cpp 2752 2006-01-19 14:40:17Z mst $
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <map>

#include "compatibility.h"

#include "Image.h"
#include "MultiFile.h"
#include "Param.h"
#include "ParamList.h"
#include "compatibility.h"

#define FOR(t,i,l) for(t::iterator i = l.begin(); i != l.end(); ++i)
#define numb_el(x) (sizeof(x)/sizeof(x[0]))
namespace std {}; using namespace std;

static const char      *_boot_name = "boot";
static const char      *_main_name = "main";
static const u_int32_t _code_irisc_offs   = 0x20000000;
static const u_int32_t _code_crspace_offs = 0x20000;
const int Image::NVRam::_fsizes[] = { 512*1024, 1048576, 2*1048576, 4*1048576 };

static struct {
    char           *fname;
    Image::FORMATS fvalue;
} formats[] = {
    { "image",   Image::OUT_IMAGE  },
    { "mlx",     Image::OUT_IMAGE  },
    { "mellanox",Image::OUT_IMAGE  },
    { "dwords",  Image::OUT_DWORDS },
    { "binary",  Image::OUT_BINARY }
};

bool         Image::fs_image = true;


//------------------------------------------
//------------------------------ Crc16 class
//------------------------------------------
//
////////////////////////////////////////////////////////////////////////
void Crc16::add(u_int32_t o)
{
    if (_debug)
        printf("Crc16::add(%08x)\n", o);
    for (int i=0; i<32; i++)
    {
        if (_crc & 0x8000)
            _crc = (((_crc<<1) | (o>>31)) ^  0x100b) & 0xffff;
        else
            _crc=((_crc<<1) | (o>>31)) & 0xffff;
        o = (o<<1) & 0xffffffff;
    }
} // Crc16::add


////////////////////////////////////////////////////////////////////////
void Crc16::finish()
{
    for (int i=0; i<16; i++)
    {
        if (_crc & 0x8000)
            _crc=((_crc<<1)  ^  0x100b) & 0xffff;
        else
            _crc=(_crc<<1) & 0xffff;
    }

    // Revert 16 low bits
    _crc = _crc ^ 0xffff;

} // Crc16::finish


//--------------------------------------------------
//------------------------------ Image::EEPROM class
//--------------------------------------------------
//
////////////////////////////////////////////////////////////////////////
void Image::NVRam::clear()
{
    _data.clear();
    _sects.clear();
} // Image::NVRam::clear


////////////////////////////////////////////////////////////////////////
void Image::NVRam::add(const u_int32_t addr, const u_int32_t val)
{
    if (addr >= _data.size())
    {
        _data.resize(addr, 0);
        for (u_int32_t i = 0; i < (addr - _data.size())/4; i++)
            _crc << 0;
        append(val);
    }
    else
    {
        u_int32_t v = __cpu_to_be32(val);
        u_int8_t  *pw = (u_int8_t*)&v;
        std::copy(pw, pw+sizeof(u_int32_t), &_data[addr]);
    }
} // Image::NVRam::add


////////////////////////////////////////////////////////////////////////
void Image::NVRam::append(const u_int32_t val)
{
    u_int32_t v = __cpu_to_be32(val);
    u_int8_t  *pw = (u_int8_t*)&v;
    _data.insert(_data.end(), pw, pw+sizeof(u_int32_t));
    _crc << val;
} // Image::NVRam::add


////////////////////////////////////////////////////////////////////////
void Image::NVRam::add_crc()
{
    _crc.finish();
    u_int32_t crc = _crc.get() & 0xffff;
    append(crc);
    _crc.clear();
} // Image::NVRam::add_crc


////////////////////////////////////////////////////////////////////////
void Image::NVRam::start(const char *name)
{
    _sects[_data.size()] = name;
} // Image::NVRam::start


////////////////////////////////////////////////////////////////////////
u_int8_t Image::NVRam::get(const u_int32_t addr, std::string& sname)
{
    if (_prev_addr+1 != addr)
    {
        // Reset iterator
        _isects = _sects.lower_bound(addr);
        if (_isects != _sects.begin())
            --_isects;
        sname = _isects->second;
    }
    else
    {
        map<u_int32_t, string>::iterator it = _isects;

        ++it;
        if (it != _sects.end()  &&  it->first == addr)
            _isects = it;
        sname = _isects->second;
    }
    _prev_addr = addr;
    return _data[addr];
} // Image::Eeprom::get

void Image::NVRam::pad_to_addr(u_int32_t addr) 
{
    if (_data.size() >= addr )
        return;

    start("PAD");
    _data.insert(_data.end(), addr - _data.size(), (u_int8_t) 0xff);
}


bool Image::NVRam::init(ParamList* plist)
{
    u_int32_t val;
    if (plist->exists("FW._Is_Flash") && plist->params["FW._Is_Flash"]->get32()) // ??? Should I error if these params don't exist ???
    {
        _is_flash = true;

        if (plist->exists("FW._Flash_SectorSize"))
            _sect_size = plist->params["FW._Flash_SectorSize"]->get32();
        else
            _sect_size = 64;

        _sect_size *= 1024;

        _nbanks = plist->params["FW._Flash_nBanks"]->get32();

        val = plist->params["FW._Flash_SizeIdx"]->get32();
        if (val >= sizeof(_fsizes)/sizeof(_fsizes[0]))
            return errmsg("_Flash_SizeIdx value (%d) exceeds available sizes (%d).\n",
                          val, (int)(sizeof(_fsizes)/sizeof(_fsizes[0])));

        _bank_size = _fsizes[val];
    }
//--    else
//--    {
//--         plist->getParameter("_EEPROM_I2C", val);
//--         fbank = val;
//--         plist->getParameter("_EEPROM_Num", val);
//--         count = val;
//--         plist->getParameter("_EEPROM_SizeIdx", val);
//--         if (val >= numbel(_esizes))
 //-         {
//--             Q_ASSERT(false);
//--        -    val = 0;
//--          }
//--          size = _esizes[val];
 //--    }

    return true;
} // Image::NVRam::init()



//------------------------------------------
//------------------------------ Image class
//------------------------------------------
//
////////////////////////////////////////////////////////////////////////
Image::Image(ParamList *plist, bool only_rmw_cr) :  _ee_amount(0), _plist(plist), _only_rmw_cr(only_rmw_cr)
{
} // Image::Image


////////////////////////////////////////////////////////////////////////
Image::~Image()
{
} // Image::~Image


////////////////////////////////////////////////////////////////////////
u_int8_t Image::getI2C(const u_int32_t addr)
{
    u_int32_t cur = 0;
    for (int i=0; i<EEPROMS; i++)
    {
        u_int32_t size = _ee_info[i] & 0xffffff;
        u_int8_t  i2c  = (u_int8_t) (_ee_info[i] >> 24);

        cur += size;
        if (addr < cur)
            return i2c;
    }
    return 0;
} // getI2C


////////////////////////////////////////////////////////////////////////
u_int32_t Image::getOffs(const u_int32_t addr)
{
    u_int32_t cur = 0;
    for (int i=0; i<EEPROMS; i++)
    {
        u_int32_t size = _ee_info[i] & 0xffffff;

        if (addr < cur + size)
            return addr - cur;
        cur += size;
    }
    return 0;
} // Image::getOffs


#define CHECK(p, args) do {                                             \
    if (plist.params.find(p) == plist.params.end())                        \
        return errmsg args ;                                               \
    if (plist.params[p]->repr == Param::ASCII)                             \
        return errmsg("Auto parameter \"%s\" can't be ASCII.", p.c_str()); \
    } while(0)
////////////////////////////////////////////////////////////////////////

// Anafa2 params check



#define CHECKP(p, args) do {                                          \
    if (plist.params.find(p) == plist.params.end())                                 \
        return errmsg args ;                                            \
    if (plist.params[p]->repr != Param::INT && plist.params[p]->repr != Param::UNS) \
        return errmsg("Parameter \"%s\" should be INT/UNS.", p);        \
    } while(0)
    

bool  Image::param_check (ParamList& plist)
{
        // Check EEPROM information
    CHECKP("EEPROM.amount", ("Parameter EEPROM.amount (amount of EEPROMS) "
          "must be specified."));
    for (unsigned int i = 1; i <= plist.params["EEPROM.amount"]->get32(); i++)
    {
        char buf[64];
        sprintf(buf, "EEPROM.eeprom%d_address", i);
        CHECKP(buf, ("Parameter %s (slave I2C address of EEPROM #%d ) must be "
              "specified.", buf, i));
        sprintf(buf, "EEPROM.eeprom%d_size", i);
        CHECKP(buf, ("Parameter %s (size of EEPROM #%d in Kbytes) must be "
              "specified.", buf, i));
    }

    // Check PLL information
    CHECKP("PLL.w1", ("Parameter PLL.w1 (first dword of PLL) must be specified."));
    CHECKP("PLL.w2", ("Parameter PLL.w2 (second dword of PLL) must be specified."));
    CHECKP("PLL.w3", ("Parameter PLL.w3 (third dword of PLL) must be specified."));
    CHECKP("PLL.w4", ("Parameter PLL.w4 (fourth dword of PLL) must be specified."));

    if (!_plist->exists(PSID_PARAM_NAME)) {
        _plist->add_psid(PSID_PARAM_NAME, "crspace", "0x3ff0c");
    }


    return true;
}




////////////////////////////////////////////////////////////////////////
bool Image::preproc(ParamList &plist, const MultiFile& mfile) {
    mfile.name();
    
    _plist = &plist;

    _nvram.init(_plist);

    /*
     * Add all parameters to CR-SPACE map or fix C-code
     */

    // Go thru group list
    for (list<Group*>::iterator gr_it = plist.grlist.begin();
         gr_it != plist.grlist.end(); ++gr_it)
    {

        // Go thru params list
        for (list<Param*>::iterator par_it = (*gr_it)->prlist.begin();
             par_it != (*gr_it)->prlist.end(); ++par_it)
        {
            Param *par = *par_it;
            switch (par->type)
            {
            case Param::CRSPACE:
            {
                // Sellect cr partition for this param:
                std::map<u_int32_t, CrSpace>* cr_partition;
                
                if (par->boot2) 
                {
                    cr_partition = &_cr_boot2;
                } 
                else 
                {
                    cr_partition = &_cr;
                }

                switch(par->repr)
                {
                case Param::ASCII:
                    for (int geo_idx=0; geo_idx < par->values.size(); geo_idx++)
                    {
                        u_int32_t geo = par->values.key(geo_idx);
                        CrSpace&  cur_cr = (*cr_partition)[geo];
                        string    svalue = par->get(geo);

                        if (par->err())
                            return errmsg("%s", par->err());

                        // CR-Space (Anafa2) is always big-endian!!!
                        for (unsigned i=0; i<par->slength; i++)
                        {
                            unsigned in_dword = 3 - i%4;
                            if (!cur_cr.add(par, par->offset*8 + par->bitoffs +
                                            ((i & ~0x03) + in_dword) * 8, 8,
                                            (i < svalue.size() ?
                                             svalue[i] : 0)))
                                return errmsg("%s", cur_cr.err());
                        }

                        //// For little-endian Cr-Space the code is follow:
                        //for (unsigned i=0; i<par->slength; i++)
                        //    if (!cur_cr.add(par, par->offset*8 + par->bitoffs + i*8,
                        //                    8, (i < svalue.size() ?
                        //                        svalue[i] : 0)))
                        //        return errmsg("%s", cur_cr.err());
                    }
                    break;
                case  Param::INT64:
                case  Param::UNS64:
                    for (int geo_idx=0; geo_idx < par->values.size(); geo_idx++)
                    {
                        u_int32_t geo = par->values.key(geo_idx);
                        CrSpace&  cur_cr = (*cr_partition)[geo];
                        u_int64_t lvalue = par->get64(geo);

                        if (par->err())
                            return errmsg("%s", par->err());

                        if (!cur_cr.add(par, par->offset*8 + par->bitoffs,
                                        par->bitsize, (u_int32_t)(lvalue>>32)))
                            return errmsg("%s", cur_cr.err());
                        if (!cur_cr.add(par, (par->offset+4)*8 + par->bitoffs,
                                        par->bitsize,
                                        (u_int32_t)(lvalue & 0xffffffff)))
                            return errmsg("%s", cur_cr.err());
                    }
                    break;
                case  Param::INT:
                case  Param::UNS:
                case  Param::ENUM:
                case  Param::BOOL:
                    for (int geo_idx=0; geo_idx < par->values.size(); geo_idx++)
                    {
                        u_int32_t geo = par->values.key(geo_idx);
                        CrSpace&  cur_cr = (*cr_partition)[geo];
                        u_int32_t ivalue = par->get32(geo);

                        if (par->err())
                            return errmsg("%s", par->err());

                        if (!cur_cr.add(par, par->offset*8 + par->bitoffs,
                                        par->bitsize, ivalue))
                            return errmsg("%s", cur_cr.err());
                    }
                    break;
                }
                break;
            }
            case Param::C:
            {
                u_int8_t*  buff;
                u_int32_t  buff_size;
                u_int32_t  param_size;
    


                if (par->repr == Param::INT  || par->repr == Param::UNS ||
                    par->repr == Param::ENUM || par->repr == Param::BOOL)
                {

                    u_int32_t  value = __cpu_to_be32(par->get32());
                    
                    buff       = (u_int8_t *)&value;
                    buff_size  = sizeof(u_int32_t);
                    param_size = sizeof(u_int32_t);

                    if (CrSpace::par_debug())
                    {
                        printf("Par:%s.%s; C %s:0x%x <= 0x%x\n",
                               par->name.c_str(), par->group->name.c_str(),
                               par->sect->name.c_str(),
                               par->offset - par->sect->addr, value);
                    }
                }
                else if (par->repr == Param::ASCII) 
                {
                    //printf("-D- Skipping C ASCII parameter %s\n", par->name.c_str());
                    //continue;

                    string val = par->get();
                    buff       = (u_int8_t*) val.c_str();
                    buff_size  = val.length();
                    param_size = par->slength;

                    if (CrSpace::par_debug())
                    {
                        printf("Par:%s.%s; C %s:0x%x <= \"%s\"\n",
                               par->name.c_str(), par->group->name.c_str(),
                               par->sect->name.c_str(),
                               par->offset - par->sect->addr, val.c_str());
                    }

                }
                else
                    return errmsg("File: %s, Line: %d - Only UNS/INT/ENUM/BOOL/ASCII"
                                  " parameters of C type are supported.",
                                  par->fname.c_str(), par->line);
                
                if (par->err())
                    return errmsg("%s", par->err());

                copy(buff, buff+buff_size, &par->sect->data[par->offset - par->sect->addr]);

                // Zero leftovers
                if (buff_size < param_size) 
                {
                    memset(&par->sect->data[par->offset - par->sect->addr + buff_size], 0, param_size - buff_size);
                }

                break;
            }
            case Param::DUMMY:
                for (int geo_idx=0; geo_idx < par->values.size(); geo_idx++)
                {
                    u_int32_t geo = par->values.key(geo_idx);
                    if (par->tag_id  &&  !_spec[geo].add(par, geo))
                        return errmsg("%s", _spec[geo].err());
                }
            }
        }
    }

    /*
     * Now go thru all CR-SPACEs map and format different data
     * initialization lists
     */
    for ( u_int32_t i = 0 ;i < 2 ; i++) 
    {
        std::map<u_int32_t, CrSpace>* crs[] = {&_cr, &_cr_boot2};

        for (map<u_int32_t, CrSpace>::iterator cr_it = crs[i]->begin();
              cr_it != crs[i]->end(); ++cr_it)
        {
            if (!cr_it->second.create_inits(_only_rmw_cr))
                return errmsg("%s", cr_it->second.err());
        }
    }

    return true;
}



bool Image::compile(ParamList& plist, const MultiFile& mfile)
{


    /*
     * Check and extract PLLs, calculate PLLs checksum
     */
    _pll[0] = plist.params["PLL.w1"]->get32();
    _pll[1] = plist.params["PLL.w2"]->get32();
    _pll[2] = plist.params["PLL.w3"]->get32();
    _pll[3] = plist.params["PLL.w4"]->get32();

    if (_pll[3] & 0xff)
        return errmsg("Fourth word of PLL (PLL.w4 parameter) - LSB byte should be zero.");
    u_int8_t chksumm = 0;
    for (int i=0; i<PLLS; i++)
    {
        chksumm += (_pll[i] & 0xff) + ((_pll[i]>>8) & 0xff) +
            ((_pll[i]>>16) & 0xff) + (_pll[i] >> 24);
    }
    chksumm &= 0xff;
    chksumm = 0xab - chksumm;
    _pll[PLLS-1] |= chksumm&0xff;

    /*
     * Create EEPROM order info
     */
    memset(&_ee_info[0], 0, sizeof(u_int32_t) * EEPROMS);
    _ee_amount = plist.params["EEPROM.amount"]->get32();
    if (_ee_amount > EEPROMS)
        return errmsg("Too many EEPROMs (%d) - Maximum is %d", _ee_amount, EEPROMS);
    if (_ee_amount & 0x01)
        return errmsg("EEPROMs amount (%d) must be even.", _ee_amount);
    for (int i=0; i<_ee_amount; i++)
    {
        char buf[30];
        sprintf(buf, "EEPROM.eeprom%d_address", i+1);
        _ee_info[i] = plist.params[buf]->get32() << 24;
        sprintf(buf, "EEPROM.eeprom%d_size", i+1);
        _ee_info[i] |= (plist.params[buf]->get32() * 1024) & 0xffffff;
    }
    // Debug prints
    if (getenv("MIC_EEPROM_DEBUG"))
    {
        printf("EEPROMs amount=%d\n", _ee_amount);
        for (int i=0; i<EEPROMS; i++)
            printf("EEPROM #%d: 0x%08x\n", i+1, _ee_info[i]);
    }

    /*
     * Create executable and start addresses list
     */
    bool         boot_defined = false;
    list<string> sect_list = mfile.sections();
    for (list<string>::iterator l=sect_list.begin(); l!=sect_list.end(); ++l)
    {
        string pstart_name = string("Auto.") + *l + "_start";
        CHECK(pstart_name, ("Auto parameter %s must be specified.",
              pstart_name.c_str()));
        if (!mfile[l->c_str()])
            return errmsg("\"%s\" section must be present.", l->c_str());
        if (!mfile[l->c_str()]->bin)
            return errmsg("\"%s\" section must be binary.", l->c_str());
        if (plist.params[pstart_name]->get32() < _code_irisc_offs)
            return errmsg("Section %s - start address (0x%x) must be greater "
                          "than 0x%x\n", l->c_str(),
                          plist.params[pstart_name]->get32(), _code_irisc_offs);
        if (mfile[l->c_str()]->addr < _code_irisc_offs)
            return errmsg("Section %s - load address (0x%x) must be greater "
                          "than 0x%x\n", l->c_str(),
                          mfile[l->c_str()]->addr, _code_irisc_offs);
        if (*l == string(_boot_name))
        {
            boot_defined = true;
            _boot.first = *l;
            _boot.second = plist.params[pstart_name]->get32();
        }
        else
            _exes.push_back(Exe(*l, plist.params[pstart_name]->get32()));
    }
    if (!boot_defined)
        return errmsg("\"%s\" executable must be specified.", _boot_name);

    // Debug prints
    if (getenv("MIC_EXE_DEBUG"))
    {
        printf("Exe: \"%s\"; start=0x%08x\n", _boot.first.c_str(), _boot.second);
        FOR (std::list<Exe>, l, _exes)
        {
            printf("Exe: \"%s\"; start=0x%08x\n", l->first.c_str(),l->second);
        }
    }

    return format(plist, mfile);
} // Image::compile

#define STRUCT_TOEEPROM(s) do {                                \
    u_int32_t *p = (u_int32_t *)(&s);                          \
    for (u_int32_t ii=0; ii<sizeof(s)/sizeof(u_int32_t); ii++) \
        _nvram << *p++;                                       \
    } while(0)
#define PTR_TOEEPROM(s) do {                                   \
    u_int32_t *p = (u_int32_t *)(s);                           \
    for (u_int32_t ii=0; ii<sizeof(s)/sizeof(u_int32_t); ii++) \
        _nvram << *p++;                                       \
    } while(0)
////////////////////////////////////////////////////////////////////////
bool Image::format(ParamList& plist, const MultiFile& mfile)
{
    _plist = &plist;
    _nvram.clear();

    /*
     * PLLs
     * ----
     */
    _nvram.start("PLL");
    for (int i=0; i<PLLS; i++)
        _nvram << _pll[i];
    _nvram.add(0x24, 0);


    /*
     * Boot section
     * ------------
     */
    _nvram.start("BOOT");
    _nvram.clear_crc();

    // Load address
    _nvram << mfile[_boot_name]->addr;

    // Size
    _nvram << mfile[_boot_name]->data.size() / 4;

    // Boot code
    MSection *sect = mfile[_boot_name];
    if (sect->data.size() & 0x03)
        return errmsg("Boot section - data size (0x%lx) must be 4-bytes aligned.",
                      (long) sect->data.size());
    for (unsigned i=0; i < sect->data.size(); i += sizeof(u_int32_t))
    {
        if (i < EEPROMS*sizeof(u_int32_t))
        {
            // First EEPROMS words - EEPROM order info
            _nvram << _ee_info[i/sizeof(u_int32_t)];
        }
        else
        {
            u_int32_t word;
            u_int8_t  *pw = (u_int8_t*)&word;
            copy(&sect->data[i], &sect->data[i+sizeof(u_int32_t)], pw);
            _nvram << __be32_to_cpu(word);
        }
    }

    // Boot jump address
    _nvram << _boot.second;

    // CRC16
    _nvram.add_crc();


    /*
     * Special structures
     * ------------------
     */
    DATA_TAG tag;
    for (map<u_int32_t, Special>::iterator sit=_spec.begin(); sit!=_spec.end(); ++sit)
    {
        char      buf[64];
        u_int32_t geo_id = sit->first;
        Special&  spec = sit->second;

        // Calculate total size
        u_int32_t total = 1;  // Crc
        for (map<int, vector<u_int8_t> >::iterator vit = spec.data.begin();
             vit != spec.data.end(); ++vit)
            total += vit->second.size()/4 + 1;  // Data + TAGID

        tag.reset();
        tag.geo_id = geo_id;
        tag.length = (u_int16_t)total;
        tag.type = TYPE_SPC;

        unsigned crc_begin = _nvram.size();

        unsigned sysguid_loc = 0;
        unsigned guid_loc = 0;
        unsigned bsn_loc = 0;

        sprintf(buf, "SPECIAL-GEO:0x%08x", geo_id);
        _nvram.start(buf);
        _nvram.clear_crc();
        STRUCT_TOEEPROM(tag);
        for (map<int, vector<u_int8_t> >::iterator vit = spec.data.begin();
             vit != spec.data.end(); ++vit)
        {
            // STRUCT_ID
            _nvram << vit->first;

            // Some special info need be collected here: BSN and GUID locations
            switch (vit->first)
            {
            case SYS_GUID_ID:
                sysguid_loc = _nvram.size();
                break;
            case NODE_GUID_ID:
                guid_loc = _nvram.size();
                break;
            case BSN_ID:
                bsn_loc = _nvram.size();
                break;
            default:
                break;
            }

            // STRUCT itself
            for (unsigned i=0; i < vit->second.size(); i += sizeof(u_int32_t))
            {
                u_int32_t word;
                u_int8_t  *pw = (u_int8_t*)&word;
                copy(&vit->second[i], &vit->second[i+sizeof(u_int32_t)], pw);
                _nvram << word;
            }
        }
        unsigned crc_end = _nvram.size() - 4;
        unsigned crc_loc = _nvram.size();
        _nvram.add_crc();

        //printf("F: CRC_b:%08x/%02x CRC_e:%08x/%02x CRC_l:%08x/%02x GUID_L:%08x/%02x BSH_L:%08x/%02x\n",
        //       getOffs(crc_begin), getI2C(crc_begin),
        //       getOffs(crc_end),   getI2C(crc_end),
        //       getOffs(crc_loc),   getI2C(crc_loc),
        //       getOffs(guid_loc),  getI2C(guid_loc),
        //       getOffs(bsn_loc),   getI2C(bsn_loc));
        if (sysguid_loc != 0 || guid_loc != 0  ||  bsn_loc != 0)
        {

            unsigned second_image_offset = 0;
            for (int i=0; i<_ee_amount/2; i++)
                second_image_offset += _ee_info[i] & 0xffffff;

            int images = fs_image ? 2 : 1;

            for (int i=0 ; i < images ; i++) {
                int offset = i * second_image_offset;
                
                _crc_repl.push_back(CRC_Replacement(
                                      EE_loc(getOffs(offset + crc_begin), getI2C(offset + crc_begin)),
                                      EE_loc(getOffs(offset + crc_end),   getI2C(offset + crc_end)),
                                      EE_loc(getOffs(offset + crc_loc),   getI2C(offset + crc_loc))));

                if (sysguid_loc != 0)
                    _sysguid_repl.push_back(EE_loc(getOffs(offset + sysguid_loc), getI2C(offset + sysguid_loc)));
                if (guid_loc != 0)
                    _guid_repl.push_back(EE_loc(getOffs(offset + guid_loc), getI2C(offset + guid_loc)));
                if (bsn_loc != 0)
                    _bsn_repl.push_back(EE_loc(getOffs(offset + bsn_loc), getI2C(offset + bsn_loc)));
            }
        }
    }


    /*
     * CR-SPACEs per GEO_ID
     * --------------------
     */
    for (map<u_int32_t, CrSpace>::iterator cr_it = _cr.begin();
         cr_it != _cr.end(); ++cr_it)
    {
        u_int32_t geo_id = cr_it->first;
        CrSpace&  cur_cr = cr_it->second;
        char      buf[64];

        /*
         * CR-SPACE - Individual words initialization
         * ------------------------------------------
         */
        if (!cur_cr.iwi_init.empty())
        {
            tag.reset();
            tag.geo_id = geo_id;
            tag.length = cur_cr.iwi_init.size() + 1;
            tag.type = TYPE_IWI;

            sprintf(buf, "CRSPACE-IWI-GEO:0x%08x", geo_id);
            _nvram.start(buf);
            _nvram.clear_crc();
            STRUCT_TOEEPROM(tag);
            for (unsigned i=0; i<cur_cr.iwi_init.size(); i += 2)
            {
                _nvram << cur_cr.iwi_init[i];
                _nvram << cur_cr.iwi_init[i+1];
            }
            _nvram.add_crc();
        }


        /*
         * CR-SPACE - Homogeneous block initialization
         * -------------------------------------------
         */
        if (!cur_cr.hbi_init.empty())
        {
            int  n=1;
            for (unsigned i=0; i<cur_cr.hbi_init.size(); i += 3)
            {
                tag.reset();
                tag.geo_id = geo_id;
                tag.length = 4;
                tag.type = TYPE_HBI;

                sprintf(buf, "CRSPACE-HBI-%d-GEO:0x%08x", n++, geo_id);
                _nvram.start(buf);
                _nvram.clear_crc();
                STRUCT_TOEEPROM(tag);
                _nvram << cur_cr.hbi_init[i];
                _nvram << cur_cr.hbi_init[i+1]*4;
                _nvram << cur_cr.hbi_init[i+2];

                _nvram.add_crc();
            }
        }

        /*
         * CR-SPACE - Non-homogeneous block initialization
         * -------------------------------------------
         */
        if (!cur_cr.nbi_init.empty())
        {
            int  n=1;
            for (unsigned i=0; i<cur_cr.nbi_init.size(); )
            {
                unsigned cnt = cur_cr.nbi_init[i+1];
                tag.reset();
                tag.geo_id = geo_id;
                tag.length = cnt + 2;
                tag.type = TYPE_NBI;

                sprintf(buf, "CRSPACE-NBI-%d-GEO:0x%08x", n++, geo_id);
                _nvram.start(buf);
                _nvram.clear_crc();
                STRUCT_TOEEPROM(tag);
                _nvram << cur_cr.nbi_init[i];
                i += 2;
                for (unsigned j=0; j<cnt; j++, i++)
                    _nvram << cur_cr.nbi_init[i];
                _nvram.add_crc();
            }
        }


        /*
         * CR-SPACE - Read-modify-write block initialization
         * -------------------------------------------------
         */
        if (!cur_cr.rmw_init.empty())
        {
            tag.reset();
            tag.geo_id = geo_id;
            tag.length = cur_cr.rmw_init.size() + 1;
            tag.type = TYPE_RMW;

            sprintf(buf, "CRSPACE-RMW-GEO:0x%08x", geo_id);
            _nvram.start(buf);
            _nvram.clear_crc();
            STRUCT_TOEEPROM(tag);
            for (unsigned i=0; i<cur_cr.rmw_init.size(); i += 3)
            {
                _nvram << cur_cr.rmw_init[i];
                _nvram << cur_cr.rmw_init[i+1];
                _nvram << cur_cr.rmw_init[i+2];
            }
            _nvram.add_crc();
        }
    }


    u_int32_t m_start = 0xffffffff;
    MSection *m = mfile[_main_name];
    if (m)
    {
        string pstart_name = string("Auto.") + _main_name + "_start";
        m_start = plist.params[pstart_name]->get32();
        u_int32_t m_load = m->addr - _code_irisc_offs + _code_crspace_offs;

        /*
         * JUMP
         * ----
         */
        if (m_start != 0xffffffff)
        {
            tag.reset();
            tag.length = 2;
            tag.type = TYPE_JMP;
            _nvram.start("JUMP");
            _nvram.clear_crc();
            STRUCT_TOEEPROM(tag);
            _nvram << m_start;
            _nvram.add_crc();
        }


        /*
         * MAIN
         * ----
         */
        tag.reset();
        tag.length = m->data.size()/4 + 2;  // Data + addr + Crc16
        tag.type = TYPE_COD;

        _nvram.start("MAIN");
        _nvram.clear_crc();

        // Data tag
        STRUCT_TOEEPROM(tag);

        // Addr
        _nvram << m_load;

        // Data
        for (unsigned i=0; i < m->data.size(); i += sizeof(u_int32_t))
        {
            u_int32_t word;
            u_int8_t  *pw = (u_int8_t*)&word;
            copy(&m->data[i], &m->data[i+sizeof(u_int32_t)], pw);
            _nvram << __be32_to_cpu(word);
        }

        // CRC16
        _nvram.add_crc();
    }


    /*
     * LAST
     * ----
     */
    tag.reset();
    tag.length = 1;
    tag.type = TYPE_LST;
    _nvram.start("LAST");
    _nvram.clear_crc();
    STRUCT_TOEEPROM(tag);
    _nvram.add_crc();


    /*
     * Check image size
     * ----------------
     */
    if (fs_image)
    {
        // FailSafe image
        unsigned size1=0, size2=0;
        for (int i=0; i<_ee_amount/2; i++)
            size1 += _ee_info[i] & 0xffffff;
        for (int i=_ee_amount/2; i<_ee_amount; i++)
            size2 += _ee_info[i] & 0xffffff;
        if (_nvram.size() > size1)
            return errmsg("Image too big (0x%x bytes) to fit to first half of "
                          "EEPROMs (0x%x bytes)", _nvram.size(),  size1);
        if (_nvram.size() > size2)
            return errmsg("Image too big (0x%x bytes) to fit to second half of "
                          "EEPROMs (0x%x bytes)", _nvram.size(),  size1);
    }
    else
    {
        // Single image (all EEPROMs)
        unsigned size=0;
        for (int i=0; i<_ee_amount; i++)
            size += _ee_info[i] & 0xffffff;
        if (_nvram.size() > size)
            return errmsg("Image too big (0x%x bytes) to fit to EEPROMs "
                          "(0x%x bytes)", _nvram.size(),  size);
    }

    _nvram.finish();
    return true;
} // Image::format


////////////////////////////////////////////////////////////////////////
bool Image::write(const char *fname, const char *format_name, DeviceType dev_type)
{
    // Determine output image format
    FORMATS frm = OUT_IMAGE;     // Default
    if (format_name)
    {
        unsigned i;
        for (i=0; i<numb_el(formats); i++)
        {
            if (!strcasecmp(formats[i].fname, format_name))
            {
                frm = formats[i].fvalue;
                break;
            }
        }
        if (i >= numb_el(formats))
            return errmsg("Output image format \"%s\" is not supported.\n", format_name);
    }

    FILE *fp = 0;

    // Open file and write info (only for non-binary images)
    if (frm != OUT_BINARY)
    {
        fp = fopen(fname, "w");
        if (!fp)
            return errmsg("Can't open file \"%s\": %s", fname, strerror(errno));

        #ifndef __WIN__
        extern char    *g_version;
        fprintf(fp, "# Created by MIC %s\n", g_version);


        struct timeval t;
        gettimeofday(&t, 0);
        fprintf(fp, "# At %s\n\n", ctime(&t.tv_sec));
        #endif


        fprintf(fp, "\n");


        dump_info(fp, frm, dev_type); // Additional info header (if needed)
        if (fs_image) 
            fs_info_dump(fp);
    }

    // Write image

    if (!write_image(fp, fname, frm))
        return false;



    if (frm != OUT_BINARY)
        fclose(fp);

    _nvram.finish();
    return true;
} // Image::write

bool Image::write_image(FILE* fp, const char* fname, FORMATS frm) 
{
    if (fs_image)
    {
        // FailSafe image
        if (!dump(fp, fname, 0, frm))
            return false;
        if (!dump(fp, fname, _ee_amount/2, frm))
            return false;
    }
    else
    {
        // Single image (all EEPROMs)
        if (!dump(fp, fname, 0, frm))
            return false;
    }
    return true;
}

bool Image::get_fw_ver (FwVer v) {

    char* ver_params[] = {
        "FWInfo.FW_VER_MAJOR",
        "FWInfo.FW_VER_MINOR",
        "FWInfo.FW_VER_SUBMINOR"
    };

    map<string, Param*>::iterator pit;

    for (u_int32_t i = 0; i < numb_el (ver_params); i++ ){
        if ((pit = _plist->params.find(ver_params[i])) != _plist->params.end())
            v[i] = pit->second->get32();
        else
            return errmsg("Can't get version info: Param %s not found\n", ver_params[i]);
    }

    return true;
}



////////////////////////////////////////////////////////////////////////
bool Image::dump_info(FILE *fp, const Image::FORMATS frm, DeviceType dev_type)
{
    map<string, Param*>::iterator pit;

    if (frm != OUT_IMAGE)
        return false;

    fprintf(fp, "DEVICE_TYPE MT%d\n", dev_type);
    if ((pit = _plist->params.find("FWInfo.FW_VER_MAJOR")) != _plist->params.end())
        fprintf(fp, "MAJ_FW_VER %d\n", pit->second->get32());
    if ((pit = _plist->params.find("FWInfo.FW_VER_MINOR")) != _plist->params.end())
        fprintf(fp, "MIN_FW_VER %d\n", pit->second->get32());
    if ((pit = _plist->params.find("FWInfo.FW_VER_SUBMINOR")) != _plist->params.end())
        fprintf(fp, "SUB_MIN_FW_VER %d\n", pit->second->get32());

    //
    // Pring device mapping
    //
    fprintf(fp, "\n");
    for (int i = 0 ; i < _ee_amount ; i++) {
        fprintf(fp, "DEVMAP %08x %02x\n", 
                _ee_info[i] & 0x00ffffff,
                _ee_info[i] >> 24);
    }
    fprintf(fp, "\n");

    for (list<EE_loc>::iterator it=_guid_repl.begin(); it!=_guid_repl.end(); ++it)
        fprintf(fp, "REPLACE GUID %08x %02x\n", it->first, it->second);
    for (list<EE_loc>::iterator it=_guids_repl.begin(); it!=_guids_repl.end(); ++it)
        fprintf(fp, "REPLACE GUIDS %08x %02x\n", it->first, it->second);
    for (list<EE_loc>::iterator it=_sysguid_repl.begin(); it!=_sysguid_repl.end(); ++it)
        fprintf(fp, "REPLACE SYSGUID %08x %02x\n", it->first, it->second);
    for (list<EE_loc>::iterator it=_bsn_repl.begin(); it!=_bsn_repl.end(); ++it)
        fprintf(fp, "REPLACE BSN  %08x %02x\n", it->first, it->second);
     for (list<EE_loc>::iterator it=_vsd_repl.begin(); it!=_vsd_repl.end(); ++it)
        fprintf(fp, "REPLACE VSD  %08x %02x\n", it->first, it->second);
    //for (list<EE_loc>::iterator it=_psid_repl.begin(); it!=_psid_repl.end(); ++it)
    //    fprintf(fp, "REPLACE PSID %08x %02x\n", it->first, it->second);
    for (list<CRC_Replacement>::iterator it=_crc_repl.begin(); it!=_crc_repl.end(); ++it)
        fprintf(fp, "REPLACE CRC  %08x %02x %08x %02x %08x %02x\n",
                it->from.first, it->from.second,
                it->to.first,   it->to.second,
                it->loc.first,  it->loc.second);
    fprintf(fp, "\n");
    return true;
} // Image::dump_info

bool Image::fs_info_dump(FILE* fp) 
{

    u_int32_t ee_index[]     = {0      , _ee_amount/2};

    
    // Go through both PtrSects
    for (u_int32_t i = 0; i < 2; ++i) { 	
        fprintf(fp, "IMAGE %d %08x %02x\n",i , 0 , _ee_info[ee_index[i]] >> 24 );
    }
    
    fprintf(fp, "\n");
    return true;
}


////////////////////////////////////////////////////////////////////////
bool Image::dump(FILE *fp, const char *fname, const unsigned int idx, const Image::FORMATS frm)
{
    switch(frm)
    {
    case OUT_IMAGE:
        return mlx_dump(fp, idx);
    case OUT_DWORDS:
        return dwords_dump(fp, idx);
    case OUT_BINARY:
        return binary_dump(fname, idx);
    }
    return errmsg("Invalid output file format");
} // Image::dump


////////////////////////////////////////////////////////////////////////
bool Image::binary_dump(const char *basename, const unsigned int idx)
{
    unsigned int ee_idx = idx;
    unsigned int ee_ofs = 0;
    unsigned int i;
    FILE         *fp;
    char*        fname = new char[strlen(basename)+10];

    sprintf(fname, "%s_%02x.bin", basename, _ee_info[ee_idx] >> 24);
    fp = fopen(fname, "wb");
    if (!fp)
        return errmsg("Can't open file \"%s\": %s", fname, strerror(errno));
    for (i = 0; i < _nvram.size(); i++)
    {
        u_int8_t data = _nvram[i];
        fwrite(&data, 1, 1, fp);
        if (++ee_ofs >= (_ee_info[ee_idx] & 0xffffff))
        {
            // Next EEPROM
            ee_ofs = 0;
            ee_idx++;
            fclose(fp);
            sprintf(fname, "%s_%02x.bin", basename, _ee_info[ee_idx] >> 24);
            fp = fopen(fname, "wb");
            if (!fp)
                return errmsg("Can't open file \"%s\": %s", fname, strerror(errno));
        }
    }
    fclose(fp);
    delete [] fname;
    return true;
} // Image::binary_dump


////////////////////////////////////////////////////////////////////////
bool Image::dwords_dump(FILE *fp, const unsigned int idx)
{
    unsigned int ee_idx = idx;
    unsigned int ee_ofs = 0;
    unsigned int gl_ofs = 0;
    string       sname, old_sname;

    _nvram.get(0, sname);
    fprintf(fp, "#\n# Section %s start, EEPROM:%02x OFFSET:%08x (%08x)\n",
            sname.c_str(), _ee_info[ee_idx] >> 24, ee_ofs, gl_ofs);
    old_sname = sname;
    for (unsigned i = 0; i < _nvram.size(); i += 4)
    {
        _nvram.get(i, sname);

        // Check for next section
        if (sname != old_sname)
        {
            fprintf(fp, "# Section %s end\n#\n", old_sname.c_str());
            old_sname = sname;
            fprintf(fp, "# Section %s start, EEPROM:%02x OFFSET:%08x (%08x)\n",
                    sname.c_str(), _ee_info[ee_idx] >> 24, ee_ofs, gl_ofs);
        }

        // Dump data
        u_int32_t data;
        _nvram.copy(i, sizeof(u_int32_t), (u_int8_t*)&data);
        data = __cpu_to_be32(data);
        fprintf(fp, "0x%08x  # EEPROM:%02x OFFSET:%08x (%08x)\n",
                data, _ee_info[ee_idx] >> 24, ee_ofs, gl_ofs);

        ee_ofs += sizeof(u_int32_t);
        gl_ofs += sizeof(u_int32_t);

        // Check for next EEPROM
        if (ee_ofs >= (_ee_info[ee_idx] & 0xffffff))
        {
            ee_ofs = 0;
            ee_idx++;
            fprintf(fp, "# Section %s end\n#\n", sname.c_str());
            fprintf(fp, "# Section %s start, EEPROM:%02x OFFSET:%08x (%08x)\n",
                    sname.c_str(), _ee_info[ee_idx] >> 24, ee_ofs, gl_ofs);
        }
    }
    fprintf(fp, "# Section %s end\n#\n", sname.c_str());
    return true;
} // Image::dwords_dump


////////////////////////////////////////////////////////////////////////
bool Image::mlx_dump(FILE *fp, const unsigned int idx)
{
    unsigned int ee_idx = idx;
    unsigned int ee_ofs = 0;
    unsigned int i, lpos=0;
    string       sname, old_sname;

    _nvram.get(0, sname);
    fprintf(fp, "START %s %08x %02x\n", sname.c_str(),
            ee_ofs, _ee_info[ee_idx] >> 24);
    old_sname = sname;
    for (i = 0; i < _nvram.size(); i++, lpos++)
    {
        u_int8_t data = _nvram.get(i, sname);

        // Check for next section
        if (sname != old_sname)
        {
            fprintf(fp, "%sEND %s\n\n", lpos % IN_LINE ? "\n" : "", old_sname.c_str());
            old_sname = sname;
            fprintf(fp, "START %s %08x %02x\n", sname.c_str(),
                    ee_ofs, _ee_info[ee_idx] >> 24);
            lpos = 0;
        }

        // Dump data
        fprintf(fp, " %02x%s", data & 0xff, (lpos+1) % IN_LINE ? "" : "\n");

        // Check for next EEPROM
        if (++ee_ofs >= (_ee_info[ee_idx] & 0xffffff))   //orenk: what does this check do ???
        {
            ee_ofs = 0;
            ee_idx++;
            fprintf(fp, "%sEND %s\n\n", lpos % IN_LINE ? "\n" : "", sname.c_str());
            fprintf(fp, "START %s %08x %02x\n", sname.c_str(),
                    ee_ofs, _ee_info[ee_idx] >> 24);
            lpos = 0xffffffff;
        }
    }
    fprintf(fp, "%sEND %s\n\n", lpos % IN_LINE ? "\n" : "", sname.c_str());
    return true;
} // Image::mlx_dump


////////////////////////////////////////////////////////////////////////
void Image::raw_dump(FILE *fp, const unsigned int idx)
{
    unsigned int ee_idx = idx;
    unsigned int ee_ofs = 0;
    unsigned int i;

    fprintf(fp, "START EEPROM %08x %02x\n", ee_ofs, _ee_info[ee_idx] >> 24);
    for (i = 0; i < _nvram.size(); i++)
    {
        fprintf(fp, " %02x%s", _nvram[i] & 0xff, (i+1) % IN_LINE ? "" : "\n");
        if (++ee_ofs >= (_ee_info[ee_idx] & 0xffffff))
        {
            // Next EEPROM
            ee_ofs = 0;
            ee_idx++;
            fprintf(fp, "%sEND EEPROM\n\n", i % IN_LINE ? "\n" : "");
            fprintf(fp, "START EEPROM %08x %02x\n",
                    ee_ofs, _ee_info[ee_idx] >> 24);
        }
    }
    fprintf(fp, "%sEND EEPROM\n\n", i % IN_LINE ? "\n" : "");
} // Image::raw_dump



//////////////////////////////////////////////////////////////////
//
//  ImSection class
//
//////////////////////////////////////////////////////////////////

bool      Image::ImSection::SetWord(u_int32_t byte_addr, u_int16_t data)
{
    // NOTE!!! Should be co-ordinated with endianess

    _data[byte_addr + 3]  =  data       & 0x00ff;
    _data[byte_addr + 2]  = (data >> 8) & 0x00ff;
    return true;
}

bool      Image::ImSection::CalcCrc() 
{
    vector<u_int8_t> v;
    Pack(v);

    _crc.clear();
    for (u_int32_t i = 0 ; i < (v.size() - 4) ; i += 4 ) 
    {
        u_int32_t* dw    = (u_int32_t*) &(v[i]);  // !!!!!!!!!!!!!!!!! Check endianess !!!
        u_int32_t  dw_be = __cpu_to_be32(*dw); 
        _crc << dw_be;
    }

    _crc.finish();
    return true;
}

u_int32_t Image::ImSection::GetCrc() {
    return _crc.get();
}

