#!/bin/bash
#
# Copyright (c) 2006 Mellanox Technologies. All rights reserved.
# 
# This Software is licensed under one of the following licenses:
#
# 1) under the terms of the "Common Public License 1.0" a copy of which is
#    available from the Open Source Initiative, see
#    http://www.opensource.org/licenses/cpl.php.
#
# 2) under the terms of the "The BSD License" a copy of which is
#    available from the Open Source Initiative, see
#    http://www.opensource.org/licenses/bsd-license.php.
#
# 3) under the terms of the "GNU General Public License (GPL) Version 2" a
#    copy of which is available from the Open Source Initiative, see
#    http://www.opensource.org/licenses/gpl-license.php.
#
# Licensee has the right to choose one of the above licenses.
#
# Redistributions of source code must retain the above copyright
# notice and one of the license notices.
#
# Redistributions in binary form must reproduce both the above copyright
# notice, one of the license notices in the documentation
# and/or other materials provided with the distribution.
#
#
#  $Id: install.sh 7833 2006-06-08 13:31:12Z vlad $
# 

usage()
{
cat << EOF

    Usage: `basename $0` [--prefix PREFIX] [--batch] [--help] [--tmpdir DIR]
                         [-kver <kernel version>] [-k|--ksrc DIR]
                         [--memtrack]
                         [--nosu] [--clean]
                         [--packages module1 module2 ... -- | --all]

                         [--with-all-libs]   : include all user level libraries (all above --with-...)
                         [--debug]  : enable debug option for IB modules
                         
                         [--build_root BUILD_ROOT]


                         Available Packages: ib_verbs ib_mthca ib_ipoib ib_sdp ib_srp ib_iser ib_rds ib_ipath

                         Install script supports configure options
                                                              
                         `$(dirname $0)/configure --help`
EOF
}


###############################################################################
#                          Default values assigment                           #
###############################################################################

# remove any aliases
unalias -a
# Find the directory where this script resides
installdir=`cd ${0%*/*};pwd`
prog=$0
package_name=$(echo ${installdir##*/} | cut -d '-' -f 1 | tr -d '[:space:]')
tmpdir=/tmp/openib_gen2
prefix=/usr/local/ofed
clean=no
batch=no
nosu=no
SET_ENV=1
MODPROBE_UPDATE=1
PACKAGES=""
ALL_PACKAGES="ib_verbs ib_mthca ib_ipoib ib_sdp ib_rds ib_srp" # ib_iser ib_ipath
ARCH=`uname -m`
BUILD_MODULES=1
WITH_PACKAGES=""
CONFIG_PARAMS=""
# WITH_ALLPACKAGES="--with-libibcm --with-libsdp --with-dapl --with-management-libs --with-osm --with-diags --with-mpi --with-perftest --with-mstflint --with-sdp-zcopy"
WITH_ALLPACKAGES="--with-libibverbs --with-libmthca --with-libibcm --with-libsdp --with-librdmacm --with-management-libs --with-opensm --with-openib-diags --with-perftest --with-mstflint"

NULL=/dev/null
IB_CONF_DIR=/etc/infiniband
NETWORK_CONF_DIR=/etc/infiniband
BUILD_ROOT=""
QUILT_PATCH=1
WITHMPI=0
LIBSDP=0

# Check if quilt is installed
QUILT=$(/usr/bin/which quilt  2> /dev/null)
if [ -z ${QUILT} ] || [ ! -x ${QUILT} ]; then
        QUILT_PATCH=0
fi

verbs=n
mthca=n
ipath=n
ipoib=n
sdp=n
srp=n
srp_target=n
iser=n
rds=n

# Handle the case there is no makedepend...
export PATH=${PATH}:${installdir}
# COLOR SETTINGS
SETCOLOR_SUCCESS="echo -en \\033[1;34m"
SETCOLOR_FAILURE="echo -en \\033[1;31m"
SETCOLOR_WARNING="echo -en \\033[1;35m"
SETCOLOR_NORMAL="echo -en \\033[0;39m"

###############################################################################
#                          Parsing command line                               #
###############################################################################
while [ ! -z "$1" ]; do
        case "$1" in
                -t|--tmpdir)
                        tmpdir=$2
                        shift 2
                        ;;
                --kernel-version=*)
                        KVERSION=`expr "x$1" : 'x[^=]*=\(.*\)'`
                        WITH_PACKAGES="${WITH_PACKAGES} --kernel-version ${KVERSION}"
                        shift
                        ;;
                -kver|--kernel-version*)
                        KVERSION=$2
                        WITH_PACKAGES="${WITH_PACKAGES} --kernel-version ${KVERSION}"
                        shift 2
                        ;;
                --kernel-sources=*)
                        KSRC=`expr "x$1" : 'x[^=]*=\(.*\)'`
                        WITH_PACKAGES="${WITH_PACKAGES} --kernel-sources ${KSRC}"
                        shift
                        ;;
                -k|--ksrc|--kernel-sources)
                        KSRC=$2
                        WITH_PACKAGES="${WITH_PACKAGES} --kernel-sources ${KSRC}"
                        shift 2
                        ;;
                -libdir | --libdir)
                        LIBDIR=$2
                        WITH_PACKAGES="${WITH_PACKAGES} --libdir ${LIBDIR}"
                        shift 2
                        ;;
                -libdir=* | --libdir=*)
                        LIBDIR=`expr "x$1" : 'x[^=]*=\(.*\)'`
                        WITH_PACKAGES="${WITH_PACKAGES} --libdir ${LIBDIR}"
                        shift
                        ;;
                -h|--help)
                        usage
                        shift
                        exit 0
                        ;;
                -b|--batch)
                        batch=yes
                        shift
                        ;;
                -c|--clean)
                        clean=yes
                        shift
                        ;;
                --nosu)
                        nosu=yes
                        shift
                        ;;
                --build_root)
                        BUILD_ROOT=$2
                        shift 2
                        ;;
                --build_only)
                        SET_ENV=0
                        shift
                        ;;
                --memtrack)
                        WITH_PACKAGES="${WITH_PACKAGES} --with-memtrack"
                        shift
                        ;;
                --debug)
                        WITH_PACKAGES="${WITH_PACKAGES} --with-mthca_debug-mod --with-ipoib_debug-mod --with-ipoib_debug_data-mod --with-sdp_debug-mod --with-sdp_debug_data-mod --with-rds_debug-mod"
                        shift
                ;;
                -p|--prefix)
                        prefix=$2
                        shift 2
                        ;;
                --packages)
                        shift
                        BUILD_MODULES=1
                        while [ "$1" != "--" ]
                        do
                            PACKAGES="$PACKAGES $1"
                            shift
                        done    
                        if [ -z "$PACKAGES" ]; then
                                BUILD_MODULES=0
                        fi
                        shift
                        ;;
                --with-all-libs)
                        WITH_PACKAGES="${WITH_PACKAGES} ${WITH_ALLPACKAGES}"
                        LIBSDP=1
                        shift
                ;;
                --without-modprobe)
                        MODPROBE_UPDATE=0
                        shift
                ;;
                --with*)
                        WITH_PACKAGES="${WITH_PACKAGES} $1"
                        case "$1" in
                                *-mod)
                                        BUILD_MODULES=1
                                ;;
                                --with-libsdp)
                                        LIBSDP=1
                                ;;
                        esac
                        shift
                ;;
                --all)
                        PACKAGES=$ALL_PACKAGES
                        BUILD_MODULES=1
                        shift
                        ;;
                *)
                        echo "Wrong parameter error: $1"
                        usage
                        exit 1
                        ;;
        esac
done

if [ "$nosu" == "no" ]; then
        if [ $UID -ne 0 ] && [ $SET_ENV -eq 1 ]; then
                echo You must be root to install
                exit 1
        fi
fi

# Set default value for PACKAGES  
if [ -z "$PACKAGES" ] && [ $BUILD_MODULES -eq 1 ]; then
        PACKAGES=$ALL_PACKAGES
fi

###############################################################################
#                          Global environment                                 #
###############################################################################

HOST_INFO=${tmpdir}/host_info.log
DEBUG_INFO=${tmpdir}/debug_info.tgz
KVERSION=${KVERSION:-$(uname -r)}
KVERSION_TWO_DIGITS=2$(echo ${KVERSION} | cut -d'.' -f2)
KSRC=${KSRC:-/lib/modules/${KVERSION}/build}
KERNEL_TREE_ROOT=${KSRC%*/*}
MAKE_KERNEL_DIR=${KSRC}
BUILD_IB_KERNEL_HOME=${tmpdir}/infiniband
IB_KERNEL_HOME=${KSRC}/drivers/infiniband

LIBDIR32="${prefix}/lib"
LIBDIR64="${prefix}/lib64"

case ${ARCH} in
        i?86|ia64)
        LIBDIR=${LIBDIR:-"${LIBDIR32}"}
        ;;
        *)
        LIBDIR=${LIBDIR:-"${LIBDIR64}"}
        ;;
esac

# if [ "${ARCH}" == "x86_64" ]; then
#     LIBDIR=${LIBDIR:-"${LIBDIR64}"}
# else
#     LIBDIR=${LIBDIR:-"${LIBDIR32}"}
# fi


# make sure we have the tmpdir
/bin/rm -rf ${tmpdir}
mkdir -p ${tmpdir}

##Special case for SuSE9.1 and SlES9.0 build tree
#if [ -d ${KERNEL_TREE_ROOT}/source/ ]; then
#       KSRC=${KERNEL_TREE_ROOT}/source
#       MAKE_KERNEL_DIR=${KERNEL_TREE_ROOT}/build
#fi

MODULES_CONF_PARAMS=""
WITH_MAKE_PARAMS=""

###############################################################################
#                          Global functions                                   #
###############################################################################
# trap 'cleanup' 2 9 15

err_echo()
{
        echo
        [ "$TT_COLOR" = "yes" ] && $SETCOLOR_FAILURE
        echo "ERROR: $@" 
        [ "$TT_COLOR" = "yes" ] && $SETCOLOR_NORMAL
        echo "ERROR: $@" >> $LOGFILE
}

pass_echo()
{
        echo
        [ "$TT_COLOR" = "yes" ] && $SETCOLOR_SUCCESS
        echo "$@"
        [ "$TT_COLOR" = "yes" ] && $SETCOLOR_NORMAL
        echo  "$@" >> $LOGFILE
}

# Create debug info
get_debug_info()
{
        /bin/rm -f $HOST_INFO
        touch $HOST_INFO
        echo "Hostname: `hostname -s`" >> $HOST_INFO
        echo "openib BUILD_ID=`cat ${installdir}/BUILD_ID`" >> $HOST_INFO
        test -e /etc/issue && echo "OS: `cat /etc/issue`" >> $HOST_INFO
        echo "Current kernel: `uname -r`" >> $HOST_INFO
        echo "Building modules for kernel: ${KVERSION}" >> $HOST_INFO
        echo "Architecture: `uname -m`" >> $HOST_INFO
        echo "GCC version: `gcc --version`"  >> $HOST_INFO
        echo "CPU: `cat /proc/cpuinfo | /bin/grep -E \"model name|arch\" | head -1`" >> $HOST_INFO
        echo "`cat /proc/meminfo | /bin/grep \"MemTotal\"`" >> $HOST_INFO
        echo "Chipset: `/sbin/lspci 2> /dev/null | head -1 | cut -d ':' -f 2-`" >> $HOST_INFO
        echo >> $HOST_INFO
        echo "############# LSPCI ##############" >> $HOST_INFO
        /sbin/lspci 2> /dev/null >> $HOST_INFO
        echo >> $HOST_INFO
        echo "############# LSPCI -N ##############" >> $HOST_INFO
        /sbin/lspci -n 2> /dev/null >> $HOST_INFO

        echo                                 
        cd ${tmpdir}
        tar czf ${DEBUG_INFO} *.log
        echo "Please open an issue in the http://openib.org/bugzilla and attach $DEBUG_INFO"
}

# Clean on failure
cleanup()
{
        trap 'echo Please wait ...' 2
        printf "\nRunning cleanup ..."
        # Restore Gen2 sources
        if [ -f ${BUILD_ROOT}/${prefix}/backup/infiniband-${KVERSION}.tgz ]; then
                rm -rf /lib/modules/${KVERSION}/kernel/drivers/infiniband
                tar xzfP ${BUILD_ROOT}/${prefix}/backup/infiniband-${KVERSION}.tgz
                /bin/rm -f ${BUILD_ROOT}/${prefix}/backup/infiniband-${KVERSION}.tgz
        fi

        if [ -f ${BUILD_ROOT}/${prefix}/backup/infiniband_ko-${KVERSION}.tgz ]; then
                rm -rf /lib/modules/${KVERSION}/kernel/drivers/infiniband
                tar xzfP ${BUILD_ROOT}/${prefix}/backup/infiniband_ko-${KVERSION}.tgz
                /bin/rm -f ${BUILD_ROOT}/${prefix}/backup/infiniband_ko-${KVERSION}.tgz
                /sbin/depmod -r -ae ${KVERSION}
                   
        fi

        # Delete binaries
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/mwrite
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/mread
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/mstflint
        
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/ib_rdma_lat
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/ib_rdma_bw
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/ib_write_lat
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/ib_write_bw
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/ib_send_lat
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/ib_send_bw
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/ib_read_lat
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/ib_read_bw
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/runme
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/ib_write_bw_postlist
        /bin/rm -f ${BUILD_ROOT}/${prefix}/bin/ib_clock_test

        tmpprefix=$(echo -n $prefix | sed -e 's/\///g')
        if [[ ${tmpprefix} != "usrlocal" && ${tmpprefix} != "usr" && ${tmpprefix} != "" ]]; then
                # Don't delete /usr/local and /usr and /
                /bin/rm -rf ${BUILD_ROOT}/${prefix}
        fi
        printf "\r\t\t\t\t\t (done)\n"

}

# Execute and print command. Put output into the LOGFILE. Check exit status.
ex()
{
        echo "Running: $@" >> $LOGFILE
        eval $@ >> $LOGFILE 2>&1
        if [ $? -ne 0 ]; then
                err_echo "Failed to execute: $@"
                echo "See $LOGFILE for more details"
                echo
                #cleanup
                get_debug_info
                exit 1
        fi
}


check_prev_install()
{
        # Check prefix and kernel version using info file:
        INFO=/etc/infiniband/info
        if [ -x ${INFO} ]; then
                info_prefix=$(${INFO} | grep -w prefix | cut -d '=' -f 2)
                if [ -x ${info_prefix}/uninstall_gen2.sh ] \
                        || [ -x ${info_prefix}/uninstall.sh ]; then
                        if [ "$batch" == "no" ]; then
                                echo -n "Remove currently installed components (${info_prefix}/uninstall.sh) (y/n) [y] ? "
                                read force
                        else
                                force="yes"
                        fi
                        if [ "x$force" == "xy" ] \
                                || [ "x$force" == "xyes" ] \
                                || [ "x$force" == "xY" ] \
                                || [ "x$force" == "xYES" ] \
                                || [ "x$force" == "x" ]  ; then
                                echo "Clean old distribution at ${info_prefix}"
                                echo "Running: yes | ${info_prefix}/uninstall.sh" >> $LOGFILE
                                # Save ${tmpdir} from uninstall
                                [ -d ${tmpdir} ] && mv ${tmpdir} ${tmpdir}.save
                                yes | ${info_prefix}/uninstall*.sh > /dev/null 2>&1
                                [ -d ${tmpdir}.save ] && mv ${tmpdir}.save ${tmpdir}
                        fi

                fi
        fi

   # Uninstall vapi if exist
        if [ -d /usr/mellanox ]; then
                echo "Found /usr/mellanox!"
                if [ "$batch" == "no" ]; then
                        if test -x /usr/mellanox/uninstall.sh; then
                                echo -n "Remove currently installed components (/usr/mellanox/uninstall.sh) (y/n) [y] ? "
                                read force
                        else
                                force="no"
                        fi
                else
                        force="yes"
                fi

                if [ "x$force" == "xy" ] \
                        || [ "x$force" == "xyes" ] \
                        || [ "x$force" == "xY" ] \
                        || [ "x$force" == "xYES" ] \
                        || [ "x$force" == "x" ]  ; then
                        echo "Clean old distribution at /usr/mellanox"
                        echo "Running: yes | /usr/mellanox/uninstall.sh" >> $LOGFILE
                        yes | /usr/mellanox/uninstall.sh >> $LOGFILE 2>&1

                        if [ $? -ne 0 ]; then
                                if [ "$batch" == "no" ]; then
                                        err_echo "/usr/mellanox/uninstall.sh failed"
                                        exit 1
                                else
                                        if [ "${prefix}" == "/usr/mellanox" ]; then
                                                echo "Forcing uninstall (rm -rf ${prefix}):"
                                                ex "/bin/rm -rf ${prefix}"
                                        else
                                                echo "WARNING: Leaving /usr/mellanox. \
                                                Continuing installation..."
                                        fi
                                fi
                        fi
                fi

        fi

        # Uninstall OFED if exist
        old_prefix=/usr/local/ofed
        if [ -d $old_prefix ]; then
                if [ "$batch" == "no" ]; then
                        echo "OFED installation found under ${old_prefix}!"
                        if ! test -x ${old_prefix}/uninstall.sh; then
                                echo "Please remove installation from ${old_prefix} and rerun `basename $0`"
                                exit 1
                        fi
                        echo -n "Remove currently installed components (${old_prefix}/uninstall.sh) (y/n) [y] ? "
                        read force
                else
                        force="yes"
                fi

                if [ "x$force" == "xy" ] \
                        || [ "x$force" == "xyes" ] \
                        || [ "x$force" == "xY" ] \
                        || [ "x$force" == "xYES" ] \
                        || [ "x$force" == "x" ]  ; then
                        echo "Clean old installation at ${old_prefix}"
                        echo "Running: yes | ${old_prefix}/uninstall.sh" >> $LOGFILE
                        yes | ${old_prefix}/uninstall.sh >> $LOGFILE 2>&1
                        if [ $? -ne 0 ]; then
                                if [ "$batch" == "no" ]; then
                                        err_echo "${old_prefix}/uninstall.sh failed"
                                        exit 1
                                else
                                        echo "Forcing uninstall (rm -rf ${old_prefix}):"
                                        ex "/bin/rm -rf ${old_prefix}"
                                fi
                        fi
                else
                        echo "Aborting installation"
                        exit 1
                fi

        fi
}


set_deps()
{
        for pack in $PACKAGES
        do
                        case $pack in
                        ib_verbs)
                                verbs=y
                        ;;
                        ib_mthca)
                                verbs=y
                                mthca=y
                        ;;
                        ib_ipath)
                                verbs=y
                                ipath=y
                        ;;
                        ib_ipoib)
                                ipoib=y
                        ;;
                        ib_sdp)
                                ipoib=y
                                sdp=y
                        ;;
                        ib_srp)
                                ipoib=y
                                srp=y
                        ;;
                        ib_srp_target)
                                ipoib=y
                                srp_target=y
                        ;;
                        ib_iser)
                                case ${KVERSION} in
                                        2.6.1[6-9]*)
                                        iser=y
                                        ;;
                                        *)
                                        echo "ISER is not supported for this kernel"
                                        ;;
                                esac
                        ;;
                        ib_rds)
                                ipoib=y
                                rds=y
                        ;;
                        *)
                                echo "Wrong package name: $pack"
                                usage
                                exit 2
                        ;;    
                
                esac
        done


        # Build MODULES_CONF_PARAMS

        if [ "$verbs" == "y" ]; then
                MODULES_CONF_PARAMS="$MODULES_CONF_PARAMS CONFIG_INFINIBAND=m CONFIG_INFINIBAND_USER_MAD=m CONFIG_INFINIBAND_USER_ACCESS=m"
                MODULES_CONF_PARAMS="$MODULES_CONF_PARAMS CONFIG_INFINIBAND_ADDR_TRANS=y"
        fi

        if [ "$mthca" == "y" ]; then
                MODULES_CONF_PARAMS="$MODULES_CONF_PARAMS CONFIG_INFINIBAND_MTHCA=m"
        fi

        if [ "$ipath" == "y" ]; then

                case $(uname -m) in
                        i?86)
                                echo "IPATH is not supported on x86 architecture"
                                exit 1
                        ;;
                        *)
                        ;;
                esac

                MODULES_CONF_PARAMS="$MODULES_CONF_PARAMS CONFIG_IPATH_CORE=m CONFIG_INFINIBAND_IPATH=m"
        fi
    
        if [ "$ipoib" == "y" ]; then
                MODULES_CONF_PARAMS="$MODULES_CONF_PARAMS CONFIG_INFINIBAND_IPOIB=m"
        fi
    
        if [ "$sdp" == "y" ]; then
                MODULES_CONF_PARAMS="$MODULES_CONF_PARAMS CONFIG_INFINIBAND_SDP=m"
        fi
    
        if [ "$srp" == "y" ]; then
                MODULES_CONF_PARAMS="$MODULES_CONF_PARAMS CONFIG_INFINIBAND_SRP=m"
        fi
    
        if [ "$srp_target" == "y" ]; then
                MODULES_CONF_PARAMS="$MODULES_CONF_PARAMS CONFIG_INFINIBAND_SRP_TARGET=m"
        fi
     
        if [ "$iser" == "y" ]; then
                MODULES_CONF_PARAMS="$MODULES_CONF_PARAMS CONFIG_INFINIBAND_ISER=m"
        fi
     
        if [ "$rds" == "y" ]; then
                MODULES_CONF_PARAMS="$MODULES_CONF_PARAMS CONFIG_INFINIBAND_RDS=m"
        fi
}


print_selected_packages()
{
        echo
        echo "The following packages will be installed:"
        echo
        [ "$verbs" == "y" ]             && echo ib_verbs
        [ "$mthca" == "y" ]             && echo ib_mthca
        [ "$ipath" == "y" ]             && echo ib_ipath
        [ "$ipoib" == "y" ]             && echo ib_ipoib
        [ "$sdp" == "y" ]               && echo ib_sdp
        [ "$srp" == "y" ]               && echo ib_srp
        [ "$srp_target" == "y" ]        && echo ib_srp_target
        [ "$iser" == "y" ]              && echo ib_iser
        [ "$rds" == "y" ]               && echo ib_rds
        
}

###############################################################################
#                          Create build tree                                  #
###############################################################################
create_build_tree()
{
        echo
        echo Creating build tree under ${tmpdir}
        ex cp -a ${installdir}/ ${tmpdir}/${package_name}
}

count_ib_ports()
{
        local cnt=0
        local tmp_cnt=0
    
        tmp_cnt=$(/sbin/lspci -n 2> /dev/null | grep "15b3:6282" | wc -l | tr -d '[:space:]') # Arbel mode
        cnt=$[ $cnt + 2*${tmp_cnt} ]
    
        tmp_cnt=$(/sbin/lspci -n 2> /dev/null | grep -E "15b3:5e8c|15b3:6274" | wc -l | tr -d '[:space:]') # Sinai
        cnt=$[ $cnt + ${tmp_cnt} ]

        tmp_cnt=$(/sbin/lspci -n 2> /dev/null | grep -E "15b3:5a44|15b3:6278" | wc -l | tr -d '[:space:]') # Tavor mode
        cnt=$[ $cnt + 2*${tmp_cnt} ]
    
        return $cnt
}

ipoib_config()
{
        # Network configuration
        if [ -f /etc/redhat-release ]; then
                DISTRIBUTION="RedHat"
        elif [ -f /etc/rocks-release ]; then
                DISTRIBUTION="Rocks"
        elif [ -f /etc/SuSE-release ]; then
                DISTRIBUTION="SuSE"
        else
                DISTRIBUTION=$(ls /etc/*-release | head -n 1 | xargs -iXXX basename XXX -release 2> /dev/null)
        fi

        INTERFACES=`/sbin/ifconfig | grep eth | awk '{print$1}' | sort -n | tr -s '\n' '\ '`
        for INTERFACE in $INTERFACES
        do
                ADDRS=`/sbin/ifconfig $INTERFACE | grep "inet addr"`
                if [[ -n $ADDRS ]]; then
                        LAN_INTERFACE=$INTERFACE
                        break
                fi
        done

        LAN_INTERFACE=${LAN_INTERFACE:-eth0}


        count_ib_ports
        ports_num=$?

        DEVICES=""
                           
        for (( i=0 ; i < $ports_num ; i++ ))
        do
                DEVICES="$DEVICES ib${i}"
        done    
        
        # Set default number of devices in case that no IB device currently installed on the system
        [ "$DEVICES" == "" ] && DEVICES="ib0 ib1"

        high_IP=`/sbin/ifconfig | sed -e "/\($LAN_INTERFACE\).*/{ N; s/.*inet addr:\([0-9]*\).*/\1/; }" | grep ^[0-9] | head -1`
        let high_IP++
        low_IP=`/sbin/ifconfig | sed -e "/\($LAN_INTERFACE\).*/{ N; s/.*inet addr:[0-9]*\.\([0-9]*\.[0-9]*\.[0-9]*\).*/\1/; }" | grep ^[0-9] | head -1`
        IPADDR=${high_IP}.${low_IP}

        NETMASK=`/sbin/ifconfig | sed -e "/\($LAN_INTERFACE\).*/{ N; s/.*Mask:\([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*\).*/\1/; }" | grep ^[0-9] | head -1`
        low_Bcast=`/sbin/ifconfig | sed -e "/\($LAN_INTERFACE\).*/{ N; s/.*Bcast:[0-9]*\.\([0-9]*\.[0-9]*\.[0-9]*\).*/\1/; }" | grep ^[0-9] | head -1`
        BROADCAST=${high_IP}.${low_Bcast}
        ONBOOT=1
        case "$NETMASK" in
                "255.255.255.0")
                        NETWORK=${IPADDR%*.*}.0
                ;;
                "255.255.0.0")
                        NETWORK=${IPADDR%*.*.*}.0.0
                ;;
                "255.0.0.0")
                        NETWORK=${IPADDR%*.*.*.*}.0.0.0
                ;;
                *)
                        NETWORK=${IPADDR%*.*.*.*}.0.0.0
                ;;
        esac

        IP_A=${IPADDR%*.*.*.*}
        IP_BCD=${IPADDR#*.}
        NET_BCD=${NETWORK#*.}
        BR_BCD=${BROADCAST#*.}

        for DEVICE in $DEVICES
        do
                target=${NETWORK_CONF_DIR}/ifcfg-${DEVICE}
                cat $NULL > $target

                if [ "$DISTRIBUTION" == "SuSE" ]; then
                        echo BOOTPROTO=\'static\' >> $target
                        echo IPADDR=\'${IP_A}.${IP_BCD}\' >> $target
                        echo NETMASK=\'$NETMASK\' >> $target
                        echo NETWORK=\'${IP_A}.${NET_BCD}\' >> $target
                        echo BROADCAST=\'${IP_A}.${BR_BCD}\' >> $target
                        echo REMOTE_IPADDR=\'\' >> $target
                        if [ "$ONBOOT" == "1" ]; then
                                echo STARTMODE=\'onboot\' >> $target
                        else
                                echo STARTMODE=\'manual\' >> $target
                        fi
                        echo WIRELESS=\'no\' >> $target
                else
                        echo DEVICE=$DEVICE >> $target
                        echo BOOTPROTO=static >> $target
                        echo IPADDR=${IP_A}.${IP_BCD} >> $target
                        echo NETMASK=$NETMASK >> $target
                        echo NETWORK=${IP_A}.${NET_BCD} >> $target
                        echo BROADCAST=${IP_A}.${BR_BCD} >> $target
                        if [ "$ONBOOT" == "1" ]; then
                                echo ONBOOT=yes >> $target
                        else
                                echo ONBOOT=no >> $target
                        fi
                fi

                let IP_A++

        done

}

###############################################################################
#                          MAIN area                                          #
###############################################################################
main()
{
        if [ "${KVERSION_TWO_DIGITS}" == "24" ]; then
                printf "\n${KVERSION} kernel is not supported by this package\n"
                exit 1
        fi

        if [ -e ${installdir}/BUILD_ID ]; then
            echo "BUILD_ID: $(cat ${installdir}/BUILD_ID)"
        fi
            

        if [ $SET_ENV -eq 1 ]; then
                LOGFILE=${tmpdir}/check_prev_install.log
                check_prev_install
        fi
    
        LOGFILE=${tmpdir}/pre_install.log

        set_deps
    
        print_selected_packages
    
        if [ ! -d ${BUILD_ROOT}/etc ]; then
                mkdir -p ${BUILD_ROOT}/etc
        fi

                      
        LOGFILE=${tmpdir}/create_build_tree.log
        create_build_tree

        
        LOGFILE=${tmpdir}/configure.log
        cd ${tmpdir}/${package_name}

        printf "\n Running configure ..."

        case ${ARCH} in
                ppc64)
                    CONFIG_PARAMS='CFLAGS="$CFLAGS -g -O2 -m64" CXXFLAGS="$CXXFLAGS -g -O2 -m64" FFLAGS="$FFLAGS -g -O2 -m64" LDFLAGS="$LDFLAGS -g -O2 -m64 -L/usr/lib64"'
                ;;
        esac

        ex env ${MODULES_CONF_PARAMS} $CONFIG_PARAMS ./configure --prefix=${prefix} --libdir=${LIBDIR} ${WITH_PACKAGES}
        printf "\r\t\t\t\t\t (done)"

        if [ $BUILD_MODULES -eq 1 ]; then
                LOGFILE=${tmpdir}/make_kernel.log
                printf "\n Running make kernel ..."
                ex make kernel
                printf "\r\t\t\t\t\t (done)"
        fi

        LOGFILE=${tmpdir}/make_user.log
        printf "\n Running make user ..."
        ex env $CONFIG_PARAMS make user
        printf "\r\t\t\t\t\t (done)"

        if [ $BUILD_MODULES -eq 1 ]; then
                if [ $SET_ENV -eq 1 ]; then
                        LOGFILE=${tmpdir}/make_install_kernel.log
                        printf "\n Running make install_kernel ..."
                        ex make install_kernel MODULES_DIR=/lib/modules/${KVERSION}
                        printf "\r\t\t\t\t\t (done)"
                else
                        LOGFILE=${tmpdir}/make_install_kernel.log
                        printf "\n Running make copy_kernel ..."
                        ex make copy_kernel MODULES_DIR=/lib/modules/${KVERSION} DESTDIR=${BUILD_ROOT}
                        printf "\r\t\t\t\t\t (done)"
                fi
        fi

        LOGFILE=${tmpdir}/make_install_user.log
        printf "\n Running make install_user ..."
        ex make DESTDIR=${BUILD_ROOT} install_user 
        printf "\r\t\t\t\t\t (done)"

        if [ $LIBSDP -eq 1 ]; then
                if [ "${ARCH}" == "x86_64" ] || [ "${ARCH}" == "ppc64" ]; then
                        # Making OFED libsdp in ${tmpdir} (32 bit version )
                        LOGFILE=${tmpdir}/libsdp32.log
                        cd ${tmpdir}/${package_name}/src/userspace/libsdp
                        make clean >> ${LOGFILE} 2>&1 && 
                        ./configure --prefix=${prefix} --libdir=${LIBDIR32} --disable-libcheck \
                        CFLAGS="-g -O2 -m32" CXXFLAGS="-g -O2 -m32" FFLAGS="-g -O2 -m32" >> ${LOGFILE} 2>&1 && 
                        make >> ${LOGFILE} 2>&1 && 
                        make DESTDIR=${BUILD_ROOT} install >> ${LOGFILE} 2>&1
                        if [ $? -ne 0 ]; then
                                echo
                                echo "The compilation of libsdp 32 bit version failed. For more information look at: ${LOGFILE}"
                                echo "Skipping libsdp 32 bit version..."
                        fi

                fi
        fi

        if [ $WITHMPI -eq 1 ]; then
            if [ -d ${tmpdir}/${package_name}/src/userspace/mpi/mvapich-gen2 ]; then
                LOGFILE=${tmpdir}/make_mpi.log
                printf "\n Running make mpi ..."
                cd ${tmpdir}/${package_name}/src/userspace/mpi/mvapich-gen2
                ex ./mvapich.make --install_path ${BUILD_ROOT}/${prefix} --include_path ${BUILD_ROOT}/${prefix}/include --lib_path ${BUILD_ROOT}/${LIBDIR}
                printf "\r\t\t\t\t\t (done)"
                cd ${tmpdir}/${package_name}
            else
                echo
                echo "MPI is not a part of this package. Skipping..."
                echo
            fi    
        fi


        LOGFILE=${tmpdir}/make_install_user.log
        # Add info file to include installation information                                  
        [ ! -d ${BUILD_ROOT}${IB_CONF_DIR} ] && mkdir -p ${BUILD_ROOT}${IB_CONF_DIR}
        [ ! -d ${BUILD_ROOT}${NETWORK_CONF_DIR} ] && mkdir -p ${BUILD_ROOT}${NETWORK_CONF_DIR}

        INFO=${BUILD_ROOT}${IB_CONF_DIR}/info
        /bin/rm -f ${INFO}
        touch ${INFO}

cat >> ${INFO} << EOFINFO
#!/bin/bash

echo prefix=${prefix}
echo Kernel=${KVERSION}
echo
echo "MODULES: ${MODULES_CONF_PARAMS}"
echo
echo "User level: ${WITH_PACKAGES}"
echo
EOFINFO

chmod +x ${INFO} > /dev/null 2>&1


        
        # Run this part to update running environment
        if [ $SET_ENV -eq 1 ]; then

                if [ -e ${installdir}/BUILD_ID ]; then
                    cp -f ${installdir}/BUILD_ID ${BUILD_ROOT}/${prefix}
                fi
                
                if [ -s ${prefix}/etc/dat.conf ]; then
                        # udapl configuration file
                        rm -f /etc/dat.conf
                        perl -ni -e  "s@/usr/lib@${LIBDIR}@;print" ${prefix}/etc/dat.conf
                        install -m 0644 ${prefix}/etc/dat.conf /etc/dat.conf
                        rm -f ${prefix}/etc/dat.conf
                fi
                
if [ -d /etc/profile.d ]; then
cat > /etc/profile.d/ofed.sh << EOF
if ! echo \${PATH} | grep -q ${prefix}/bin ; then
        PATH=\${PATH}:${prefix}/bin
fi
EOF
cat > /etc/profile.d/ofed.csh << EOF
if ( "\${path}" !~ *${prefix}/bin* ) then
        set path = ( \$path ${prefix}/bin )
endif
EOF
fi

                # Update /etc/ld.so.conf depending on architecture                                                     
        
                if [ -d  ${LIBDIR32} ]; then
                        echo "${LIBDIR32}" > /etc/ld.so.conf.d/ofed.conf
                fi
        
                if [ -d  ${LIBDIR64} ]; then
                        echo "${LIBDIR64}" >> /etc/ld.so.conf.d/ofed.conf
                fi

                ex /sbin/ldconfig

                printf "\n Running depmod ..."
                if [ "${KVERSION_TWO_DIGITS}" == "24" ]; then
                        ex /sbin/depmod -r -ae -F /boot/System.map-${KVERSION} ${KVERSION}
                else
                        ex /sbin/depmod -r -ae ${KVERSION}
                fi
                printf "\r\t\t\t\t\t (done)"
        
                [ ! -d ${IB_CONF_DIR} ] && mkdir -p ${IB_CONF_DIR}

                install -m 0644 ${installdir}/openib.conf ${IB_CONF_DIR}

                if [ "$verbs" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load UCM module" >> ${IB_CONF_DIR}/openib.conf
                        echo "UCM_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load RDMA_CM module" >> ${IB_CONF_DIR}/openib.conf
                        echo "RDMA_CM_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load RDMA_UCM module" >> ${IB_CONF_DIR}/openib.conf
                        echo "RDMA_UCM_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi

                if [ "$mthca" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load MTHCA module" >> ${IB_CONF_DIR}/openib.conf
                        echo "MTHCA_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                fi
 
                if [ "$ipath" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load IPATH module" >> ${IB_CONF_DIR}/openib.conf
                        echo "IPATH_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                fi
                                                
                if [ "$ipoib" == "y" ]; then
                        printf "\n Configuring IPoIB..."
                        ipoib_config

                        echo >> ${IB_CONF_DIR}/openib.conf                                                
                        echo "# Load IPoIB" >> ${IB_CONF_DIR}/openib.conf
                        echo "IPOIB_LOAD=yes" >> ${IB_CONF_DIR}/openib.conf
                        printf "\r\t\t\t\t\t (done)"

if [ $MODPROBE_UPDATE -eq 1 ]; then
for (( i=0 ; i < $ports_num ; i++ ))
do
cat >> /etc/modprobe.conf << EOF
alias ib${i} ib_ipoib
EOF
done
fi

                fi
        
                if [ "$sdp" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load SDP module" >> ${IB_CONF_DIR}/openib.conf
                        echo "SDP_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
if [ $MODPROBE_UPDATE -eq 1 ]; then
cat >> /etc/modprobe.conf << EOF
alias net-pf-27 ib_sdp
EOF
fi
                fi
        
                if [ "$srp" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load SRP initiator module" >> ${IB_CONF_DIR}/openib.conf
                        echo "SRP_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi
        
                if [ "$srp_target" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load SRP target module" >> ${IB_CONF_DIR}/openib.conf
                        echo "SRP_TARGET_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi
        
                if [ "$iser" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load ISER module" >> ${IB_CONF_DIR}/openib.conf
                        echo "ISER_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi
        
                if [ "$rds" == "y" ]; then
                        echo >> ${IB_CONF_DIR}/openib.conf
                        echo "# Load RDS module" >> ${IB_CONF_DIR}/openib.conf
                        echo "RDS_LOAD=no" >> ${IB_CONF_DIR}/openib.conf
                fi

                if [ -d /etc/udev/rules.d ]; then
                        install -m 0644 ${installdir}/90-ib.rules /etc/udev/rules.d/
                fi

                install -m 0755 ${installdir}/openibd /etc/init.d/
                if [ -f /etc/SuSE-release ]; then
                        perl -i -ne 'if (m@^#!/bin/bash@) {
                        print q@#!/bin/bash
### BEGIN INIT INFO
#  Provides:       openibd
#  Required-Start: $local_fs $network
#  Required-Stop: opensmd
#  Default-Start:  2 3 5
#  Default-Stop: 0 1 2 6
#  Description:    Activates/Deactivates InfiniBand Driver to \
#                  start at boot time.
### END INIT INFO
@;
                 } else {
                     print;
                 }' /etc/init.d/openibd


                        /sbin/insserv openibd
                else
                        perl -i -ne 'if (m@^#!/bin/bash@) {
                        print q@#!/bin/bash
#
# Bring up/down openib
# 
#  chkconfig: 2345 15 85
#  description: Activates/Deactivates InfiniBand Driver to \
#               start at boot time.
# 
### BEGIN INIT INFO
# Provides:       openibd
### END INIT INFO
@;
                 } else {
                     print;
                 }' /etc/init.d/openibd

                        /sbin/chkconfig --del openibd
                        /sbin/chkconfig --add openibd

                fi

        fi

        printf "\n Installation finished successfully \n"

}

main
