/*
 *	MINIXEMU	An emulator for Minix binaries.
 *
 *	VM86 is used to process all the 8086 mode code. We trap up to
 *	386 mode for system call emulation and naughties.
 *
 *
 *	Copyright (C) 1995 Alan Cox (alan@lxorguk.ukuu.org.uk)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/vm86.h>
#include "minix.h" 

extern int minix_syscall(minix_msg *m, int syscall, int fs);
volatile struct vm86_struct minix_cpu;
static unsigned char minix_space[128*1024+15];	/* The Linux vm will deal with not allocating the unused pages */
static unsigned char *minix_base;		/* Paragraph aligned */

#define dbprintf(x)

static void minix_init()
{
	minix_cpu.screen_bitmap=0;
	minix_cpu.cpu_type = CPU_286;
	/*
	 *	All INT xx calls are trapped.
	 */
	memset((void *)&minix_cpu.int_revectored,0xFF, sizeof(minix_cpu.int_revectored));
}

static int minix_take_interrupt(int arg)
{
	minix_msg m;
	unsigned short mtype;
	if(arg!=0x80 && arg!=32)
	{
		dbprintf(("Took an int %d\n", arg));
		fflush(stderr);
		kill(getpid(), SIGILL);
		return -1;
	}
	/* Build a minix message pointer and do the syscall */
	
	dbprintf(("%d:syscall AX=%X BX=%X CX=%X\n",
		getpid(),
		(unsigned short)minix_cpu.regs.eax,
		(unsigned short)minix_cpu.regs.ebx,
		(unsigned short)minix_cpu.regs.ecx));
		
	/* AX=src/dest BX=message pointer CX=send/recv/both */
	memcpy(&m, MINIX_DSEG(minix_cpu.regs.ebx)+4, sizeof(m));
	
	mtype=*(unsigned short *)(MINIX_DSEG(minix_cpu.regs.ebx)+2);
	
	dbprintf(("%d:minix syscall %d to %d.\n",getpid(),mtype, minix_cpu.regs.eax&0xFFFF));
	
	mtype=minix_syscall(&m,mtype,minix_cpu.regs.eax&0xFFFF);
	memcpy(MINIX_DSEG(minix_cpu.regs.ebx)+4, &m, sizeof(m));
	*(unsigned short *)(MINIX_DSEG(minix_cpu.regs.ebx)+2)=mtype;
	*(unsigned short *)(MINIX_DSEG(minix_cpu.regs.ebx))=minix_cpu.regs.eax;
	dbprintf(("%d:minix syscall returned %d %d %d\n",getpid(),
		mtype,m.msg.m1.m1i1,m.msg.m1.m1p1));
	/* Happen to leave mtype in EAX */
	minix_cpu.regs.eax=(mtype&0xFFFF);
	/* Finally return to vm86 state */

	return 0;
}


static int load_minix(int fd)
{
	/* Load the minix binary image and set it up in a suitable VM86 segment. Load CS and DS/SS
	   according to image type. chmem is ignored we always use 64K segments */
	struct minix_exec_hdr mh;
	unsigned char *dsp;
	if(read(fd, &mh,sizeof(mh))!=sizeof(mh))
		return -ENOEXEC;
	if(mh.hlen!=EXEC_HEADER_SIZE)
		return -ENOEXEC;
	if(mh.type!=MINIX_COMBID&&mh.type!=MINIX_SPLITID)
		return -ENOEXEC;
/*	fprintf(stderr,"Minix binary - %lX. tseg=%ld dseg=%ld bss=%ld\n",
		mh.type,mh.tseg,mh.dseg,mh.bseg);*/
	if(read(fd,minix_base,mh.tseg)!=mh.tseg)
		return -ENOEXEC;
	if(mh.type==MINIX_COMBID)
		dsp=minix_base+mh.tseg;
	else
		dsp=minix_base+65536;
	if(read(fd,dsp,mh.dseg)!=mh.dseg)
		return -ENOEXEC;
	memset(dsp+mh.dseg,0, mh.bseg);
	/*
	 *	Load the VM86 registers
	 */
	 
/*	if(mh.type==MINIX_COMBID)
		dsp=minix_base;*/
	minix_cpu.regs.ds=PARAGRAPH(dsp);
	minix_cpu.regs.es=PARAGRAPH(dsp);
	minix_cpu.regs.ss=PARAGRAPH(dsp);
	minix_cpu.regs.esp=65134;	/* Args stacked later */
	minix_cpu.regs.cs=PARAGRAPH(minix_base);
	minix_cpu.regs.eip=0;		/* Run from 0 */
	
	/*
	 *	Loaded
	 */
	return 0;
}


int run_minix()
{
	/*
	 *	Execute 8086 code for a while.
	 */
	int err=vm86((struct vm86_struct *)&minix_cpu);
	switch(VM86_TYPE(err))
	{
		/*
		 *	Signals are just re-starts of emulation (yes the handler might alter
		 *	minix_cpu)
		 */
		case VM86_SIGNAL:
			return 0;
		case VM86_UNKNOWN:
			/* Simulate a seg fault */
			signal(SIGSEGV, SIG_DFL);
			kill(getpid(), SIGSEGV);
			exit(1);
		case VM86_INTx:
			return minix_take_interrupt(VM86_ARG(err));
		case VM86_STI:
			return 0;	/* Shouldnt be seen */
	}
	return 0;
}

unsigned short stack_str(char *xp)
{
	unsigned char *psp=(unsigned char *)MINIX_DSEG(minix_cpu.regs.esp);
	psp-=strlen(xp)+1;
	strcpy(psp,xp);
	minix_cpu.regs.esp-=strlen(xp)+1;
	return minix_cpu.regs.esp;
}	

int main(int argc, char *argv[], char *envp[])
{
	int fd;
	char **p,**q;
	long l=0;
	unsigned short *psp;
	int ct=0;
	dbprintf(("MinixEMU 0.03 Alpha\n"));
	if(argc==1)
	{
		fprintf(stderr,"minixemu cmd args.....\n");
		exit(1);
	}
	minix_init();
	minix_base=(unsigned char *)((((unsigned long)minix_space)+15)&~15);
	
	fd=open(argv[1], O_RDONLY);
	if(fd==-1)
	{
		perror(argv[1]);
		exit(1);
	}
	
	if(load_minix(fd)==-1)
	{
		fprintf(stderr,"Not a minix binary.\n");
		exit(1);
	}
	
	close(fd);
	
	argv+=2;
	
	p=envp;
	while(*p)
		l+=strlen(*p++)+1;
	p--;
	q=argv;
	while(*q)
		l+=strlen(*q++)+1;
	q--;
	
/*	fprintf(stderr,"strlen=%d\n",l);*/
/*	fprintf(stderr,"SP=%d\n",minix_cpu.regs.esp);*/
	/* PSP is the bottom of the argument vectors */
	psp=(unsigned short *)(MINIX_DSEG(minix_cpu.regs.esp)-l-2*ct);		
	/* Space used for texts */

/*	printf("String base at %p\n", psp);*/
	/* Stack arguments */
	*psp--=0;
/*	printf("Env end mark at %p\n",psp);*/
	ct++;
	do
	{
		ct++;
		*psp--=stack_str(*p--);
/*		printf("String %s based at %d (vec %d)\n",p[1],psp[1],((unsigned long)psp)+2-(unsigned long)(MINIX_DSEG(0)));*/
	}
	while(p!=envp-1);
	*psp--=0;
/*	printf("Argv end mark at %p\n",psp);*/
	ct++;	
	do
	{
		ct++;
		*psp--=stack_str(*q--);
/*		printf("String %s based at %d (vec %d)\n",q[1],psp[1],((unsigned long)psp)+2-(unsigned long)(MINIX_DSEG(0)));*/
	}
	while(q!=argv-1);
	*psp=argc-2;		/* The book is wrong here, and forgets to tell
				   you about this one */
	minix_cpu.regs.esp= (unsigned long)psp-(unsigned long)MINIX_DSEG(0);
/*	printf("Final SP=%d\n",minix_cpu.regs.esp);*/
	
	while(1)
		run_minix();
}

