/* 
 * tools.c
 * Copyright (C) 2002, AVM GmbH. All rights reserved.
 * 
 * This Software is  free software. You can redistribute and/or
 * modify such free software under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * The free software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this Software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA, or see
 * http://www.opensource.org/licenses/lgpl-license.html
 * 
 * Contact: AVM GmbH, Alt-Moabit 95, 10559 Berlin, Germany, email: info@avm.de
 */

#include <linux/slab.h>
#include <linux/string.h>
#include <linux/vmalloc.h>
#include <linux/kernel.h>
#include "defs.h"
#include "tools.h"

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
struct __q {

	unsigned	head, tail, mask;
	unsigned	size, free;
	unsigned	bnum, blen;
	void **		item;
	char *		bptr;
} ;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
typedef struct __header {

	unsigned	type;
#if !defined (NDEBUG)
	unsigned	size;
	unsigned	from;
	unsigned	tag;
#endif
} header_t;

#define	TYPE_NONE	'?'
#define TYPE_KMALLOCED	'k'
#define	TYPE_VMALLOCED	'v'
#define KMALLOC_LIMIT	131072

static nomem_handler_t	handler		= NULL;

#if !defined (NDEBUG)
#include <asm/atomic.h>

#define	FENCE_TAG	0xDEADBEEF
#define	FENCE1_OK(h,m)	((h)->tag==FENCE_TAG)
#define	FENCE2_OK(h,m)	(*(unsigned *)(((char *) m)+(h)->size)==FENCE_TAG)

static atomic_t		alloc_count	= ATOMIC_INIT (0);

#if !defined (NDEBUG) && defined (LOG_TIMER)
static struct timeval	zero_time;
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned hallocated (void) {
    
	return atomic_read (&alloc_count);
} /* hallocated */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned hallocator (void * mem) {
	header_t * hdr;

	if (mem != NULL) {
		hdr = ((header_t *) mem) - 1;
		return FENCE1_OK(hdr, mem) ? 0 : hdr->from;
	} else {
		return 0;
	}
} /* hallocator */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int hvalid (void * mem) {
	header_t *	hdr;
	int		flag = TRUE;

	if (mem != NULL) {
		hdr  = ((header_t *) mem) - 1;
		flag = FENCE1_OK(hdr, mem) && FENCE2_OK(hdr, mem);
	} 
	return flag;
} /* hvalid */
#endif

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void * ALLOCATE (unsigned n, unsigned * t, int p) {
	void *		temp;
	unsigned	type;

	if (n <= KMALLOC_LIMIT) {
		temp = kmalloc (n, p);
		type = TYPE_KMALLOCED;
	} else {
		temp = vmalloc (n);
		type = TYPE_VMALLOCED;
	}
	*t = (temp != NULL) ? type : TYPE_NONE;
	return temp;
} /* ALLOCATE */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
#if !defined (NDEBUG)
#define	PATCH(n)	sizeof(header_t)+sizeof(unsigned)+((n)?(n):1)
#else
#define	PATCH(n)	sizeof(header_t)+((n)?(n):1)
#endif

static void * halloc (unsigned size, unsigned addr, int prio) {
	unsigned	n, t = TYPE_NONE;
	void *		mem;
	header_t *	hdr;

	n = PATCH(size);
	if (NULL == (hdr = (header_t *) ALLOCATE (n, &t, prio))) {
		log ("Memory request (%u/%u bytes) failed.\n", size, n);
		mem = NULL;
		if (handler != NULL) {
			(*handler) (size);
		}
	} else {
		mem = (void *) (hdr + 1);
		hdr->type = t;
#if !defined (NDEBUG)
		hdr->size = size ? size : 1;
		hdr->from = addr;
		hdr->tag  = FENCE_TAG;
		* (unsigned *) (((char *) mem) + size) = FENCE_TAG;
		atomic_add (hdr->size, &alloc_count);
#if defined (LOG_ALLOC)
		log (
			"malloc (%u[%u,%c]) = %p @ %08x\n", 
			hdr->size, 
			n, 
			hdr->type, 
			mem, 
			hdr->from
		);
#endif
#else
		UNUSED_ARG (addr);
#endif
	}
	return mem;
} /* halloc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * hmalloc (unsigned size) {

	return halloc (size, *(((unsigned *) &size) - 1), GFP_ATOMIC);
} /* hmalloc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * hcalloc (unsigned size) {
	void * mem;

	mem = halloc (size, *(((unsigned *) &size) - 1), GFP_ATOMIC);
	if ((mem != NULL) && (size != 0)) {
		memset (mem, 0, size);
	}
	return mem;
} /* hcalloc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * hmalloc_kernel (unsigned size) {

	return halloc (size, *(((unsigned *) &size) - 1), GFP_KERNEL);
} /* hmalloc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void * hcalloc_kernel (unsigned size) {
	void * mem;

	mem = halloc (size, *(((unsigned *) &size) - 1), GFP_KERNEL);
	if ((mem != NULL) && (size != 0)) {
		memset (mem, 0, size);
	}
	return mem;
} /* hcalloc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void hfree (void * mem) {
	header_t *	hdr;

	if (mem != NULL) {
		hdr = ((header_t *) mem) - 1;
#if !defined (NDEBUG)
		if (!(FENCE1_OK(hdr, mem) && FENCE2_OK(hdr, mem))) {
			log (
				"FENCE VIOLATED (%u/0x%08X)!\n", 
				hdr->size, 
				hdr->from
			);
			return; /* Oops. */
		} 
		atomic_sub (hdr->size, &alloc_count);
#if defined (LOG_ALLOC)
		log (
			"free (%p), %u bytes, alloc @ %08x\n",
			mem,
			hdr->size,
			hdr->from
		);
#endif
#endif
		assert ((hdr->type == TYPE_KMALLOCED) || (hdr->type == TYPE_VMALLOCED));
		switch (hdr->type) {
		
		case TYPE_KMALLOCED: 
			kfree (hdr);
			break;
		case TYPE_VMALLOCED: 
			vfree (hdr);
			break;
		default:
			log ("Could not free memory @ %p.\n", mem);
		}
	}
} /* hfree */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
nomem_handler_t hset_handler (nomem_handler_t hndf) {
	nomem_handler_t	oldf = handler;
	
	handler = hndf;
	return oldf;
} /* hset_handler */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/

#if !defined (NDEBUG) && defined (LOG_TIMER)

#include <linux/time.h>

void setup_timer (dbg_timer * t, long dsec, long dusec) {
	
	assert (t != NULL);
	memset (&t->t, 0, sizeof (t->t));
	t->d.tv_sec  = dsec;
	t->d.tv_usec = dusec;
} /* setup_timer */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int check_timer (dbg_timer * t) {
	int		res = 1;
	struct timeval	now;
	struct timeval	delta;
	
	assert (t != NULL);
	do_gettimeofday (&now);
	timeval_less (now, zero_time, &delta);
	now = delta;
	timeval_less (now, t->t, &delta);
	if ((delta.tv_sec > t->d.tv_sec) 
	|| ((delta.tv_sec == t->d.tv_sec) && (delta.tv_usec > t->d.tv_usec))
	) {
		lprintf (
			KERN_INFO, 
			"Timer '%s' exceeded: %ld s, %ld s\n", 
			t->name,
			delta.tv_sec,
			delta.tv_usec
		);
		res = 0;
	} 
	return res;
} /* check_timer */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int check_timer_cb (dbg_timer * t, void (* callback) (dbg_timer *, struct timeval *)) {
	int		res = 1;
	struct timeval	now;
	struct timeval	delta;
	
	assert (t != NULL);
	do_gettimeofday (&now);
	timeval_less (now, zero_time, &delta);
	now = delta;
	timeval_less (now, t->t, &delta);
	if ((delta.tv_sec > t->d.tv_sec) 
	|| ((delta.tv_sec == t->d.tv_sec) && (delta.tv_usec > t->d.tv_usec))
	) {
		if (callback != NULL) {
			(*callback) (t, &delta);
		}
		res = 0;
	} 
	return res;
} /* check_timer_cb */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void touch_timer (dbg_timer * t) {
	struct timeval	temp, delta;
	
	assert (t != NULL);
	do_gettimeofday (&temp);
	timeval_less (temp, zero_time, &delta);
	t->t = delta;
} /* touch_timer */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void start_watch (dbg_timer * w) {
	struct timeval	temp, delta;
	
	assert (w != NULL);
	do_gettimeofday (&temp);
	timeval_less (temp, zero_time, &delta);
	w->t = delta;
} /* start_watch */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void stop_watch (dbg_timer * w) {
	struct timeval	temp, delta;
	
	assert (w != NULL);
	do_gettimeofday (&temp);
	timeval_less (temp, zero_time, &delta);
	temp = delta;
	timeval_less (temp, w->t, &delta);
	w->t = delta;
} /* stop_watch */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int timers_start (void) {

	do_gettimeofday (&zero_time);
	return 1;
} /* timers_start */
     
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void timers_stop (void) {

} /* timers_stop */

#endif /* !NDEBUG && LOG_TIMER */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void vlprintf (const char * level, const char * fmt, va_list args) {
	char line[__BUFSIZE];

	VSNPRINTF (line, sizeof (line), fmt, args);
	printk ("%s%s: %s", level, TARGET, line); 
} /* vlprintf */
 
/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void lprintf (const char * level, const char * fmt, ...) {
	va_list args;

	va_start (args, fmt);
	vlprintf (level, fmt, args);
	va_end (args);
} /* lprintf */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned long atomic_xchg (volatile atomic_t * v, unsigned value) {

	return __xchg (value, &v->counter, sizeof (unsigned));
} /* atomic_xchg */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
ptr_queue_p q_make (unsigned max) {
	ptr_queue_p	qp;
	unsigned	mask = 1;
	
	if (NULL == (qp = (ptr_queue_p) hmalloc (sizeof (ptr_queue_t)))) {
		return NULL;
	}
	if (NULL == (qp->item = (void **) hmalloc (max * sizeof (void *)))) {
		hfree (qp);
		return NULL;
	}
	qp->bptr = NULL;
	while (mask < max) {
		mask <<= 1;
	}
	assert (mask == max);
	--mask;
	qp->head = 0;
	qp->tail = 0;
	qp->mask = mask;
	qp->size = max;
	qp->free = max;
	return qp;
} /* q_make */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void q_remove (ptr_queue_p * qpp) {
	ptr_queue_p	qp;
	
	assert (qpp != NULL);
	qp = *qpp;
	assert (qp != NULL);
	assert (qp->item != NULL);
	if (qp->bptr != NULL) {
		hfree (qp->bptr);
	}
	hfree (qp->item);
	hfree (qp);
	*qpp = NULL;
} /* q_remove */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void q_reset (ptr_queue_p qp) {

	assert (qp != NULL);
	qp->head = qp->tail = 0;
	qp->free = qp->size;
} /* q_reset */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int q_attach_mem (ptr_queue_p qp, unsigned n, unsigned len) {
	void *		buf;

	assert (qp != NULL);
	assert (qp->bptr == 0);
	assert ((n * len) != 0);
	if (NULL == (buf = hmalloc (n * len))) {
		return FALSE;
	} 
	qp->bnum = n;
	qp->blen = len;
	qp->bptr = buf;
	return TRUE;
} /* q_attach_mem */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int q_enqueue (ptr_queue_p qp, void * p) {

	assert (qp != NULL);
	if (qp->free == 0) {
		return FALSE;
	}
	assert (qp->head < qp->size);
	qp->item[qp->head++] = p;
	qp->head &= qp->mask;
	assert (qp->head < qp->size);
	qp->free--;
	return TRUE;
} /* q_enqueue */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int q_enqueue_mem (ptr_queue_p qp, void * m, unsigned len) {
	unsigned	ix;
	char *		mp;
	
	assert (qp != NULL);
	assert (qp->bptr != NULL);
	assert (qp->blen >= len);
	if (qp->free == 0) {
		return FALSE;
	}
	assert (qp->head < qp->size);
	ix = qp->head++;
	qp->head &= qp->mask;
	qp->item[ix] = mp = &qp->bptr[ix * len];
	assert (mp != NULL);
	memcpy (mp, m, len);
	assert (qp->head < qp->size);
	qp->free--;
	return TRUE;
} /* q_enqueue_mem */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int q_dequeue (ptr_queue_p qp, void ** pp) {

	assert (qp != NULL);
	if (qp->free == qp->size) {
		return FALSE;
	}
	assert (qp->tail < qp->size);
	assert (pp != NULL);
	*pp = qp->item[qp->tail++];
	qp->tail &= qp->mask;
	assert (qp->tail < qp->size);
	qp->free++;
	return TRUE;
} /* q_dequeue */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
int q_peek (ptr_queue_p qp, void ** pp) {

	assert (qp != NULL);
	if (qp->free == qp->size) {
		return FALSE;
	}
	assert (qp->tail < qp->size);
	assert (pp != NULL);
	*pp = qp->item[qp->tail];
	return TRUE;
} /* q_peek */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
unsigned q_get_count (ptr_queue_p qp) {

	assert (qp != NULL);
	return qp->size - qp->free;
} /* q_get_count */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void memdump (const void * mem, unsigned len, unsigned start, const char * msg) {
        unsigned        max, min, idx;
        unsigned char * data = (unsigned char *) mem;
        char            hex[50], chr[20];

	lprintf (KERN_INFO, "Memory dump %s:\n", msg);
        min = 0;
        while (min < len) {
                max = ((min + 16) > len ? len : min + 16);
                idx = 0;
                while ((min + idx) < max) {
                        sprintf (hex + 3 * idx, "%02x ", *data);
                        sprintf (chr + idx, "%c", ((' ' <= *data) &&
                                        (*data <= '~')) ? *data : '.');
                        ++idx;
                        ++data;
                }
                while (idx < 16) {
                        strcpy (hex + 3 * idx++, "   ");
                }
                lprintf (KERN_INFO, "%08x: %s  %s\n", min + start, hex, chr);
                min = max;
        }
} /* memdump */

