/*
 * ocfsipc.c
 *
 * IPC infrastructure to support ipcdlm
 *
 * Copyright (C) 2002 Oracle Corporation.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 *
 * Authors: Neeraj Goyal, Suchit Kaura, Kurt Hackel, Sunil Mushran,
 *          Manish Singh, Wim Coekaerts
 */

#include <ocfs.h>

/* Tracing */
#define  OCFS_DEBUG_CONTEXT  OCFS_DEBUG_CONTEXT_IPC

ocfs_ipc_ctxt OcfsIpcCtxt;

/*
 * ocfs_cleanup_ipc()
 *
 */
int ocfs_cleanup_ipc (void)
{
	return 0;
}				/* ocfs_cleanup_ipc */

/*
 * ocfs_init_ipc()
 *
 */
int ocfs_init_ipc (void)
{
	return 0;
}				/* ocfs_init_ipc */

/*
 * ocfs_recv_thread()
 *
 */
int ocfs_recv_thread (void *unused)
{
	int status;
	ocfs_recv_ctxt *recv_ctxt = NULL;

	LOG_ENTRY ();

#define LISTENER_PROCESS_NAME	"ocfslsnr"
	ocfs_daemonize (LISTENER_PROCESS_NAME, strlen(LISTENER_PROCESS_NAME));
	OcfsIpcCtxt.task = current;

	while (1) {
		recv_ctxt = ocfs_malloc (sizeof (ocfs_recv_ctxt));
		if (recv_ctxt == NULL) {
			LOG_ERROR_STATUS (-ENOMEM);
			goto bail;
		}

		memset (recv_ctxt, 0, sizeof (ocfs_recv_ctxt));
		recv_ctxt->msg_len = OCFS_MAX_DLM_PKT_SIZE;

		status = ocfs_recv_udp_msg (recv_ctxt);
		if (status < 0) {
			ocfs_safefree (recv_ctxt);
			if (status != -EBADF) {
				LOG_ERROR_STATUS (status);
			} else {
				/* Thread is being killed. */
				goto finally;
			}
		}
	}

      finally:
	if (OcfsIpcCtxt.send_sock) {
		sock_release (OcfsIpcCtxt.send_sock);
		OcfsIpcCtxt.send_sock = NULL;
	}

	if (OcfsIpcCtxt.recv_sock) {
		sock_release (OcfsIpcCtxt.recv_sock);
		OcfsIpcCtxt.recv_sock = NULL;
	}

	/* Flush all scheduled tasks */
	flush_scheduled_tasks ();

	/* signal main thread of ipcdlm's exit */
	complete (&(OcfsIpcCtxt.complete));

      bail:
	LOG_EXIT ();
	return 0;
}				/* ocfs_recv_thread */

/*
 * ocfs_init_udp()
 *
 */
int ocfs_init_udp (void)
{
	int status = 0;
	int child_pid;

	LOG_ENTRY ();

	/* start the listener thread */
	status = ocfs_init_udp_sock (&OcfsIpcCtxt.send_sock,
				     &OcfsIpcCtxt.recv_sock);
	if (status < 0) {
		LOG_ERROR_STATUS (status);
		goto bail;
	}

	child_pid = kernel_thread (ocfs_recv_thread, NULL,
				   CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
	if (child_pid < 0) {
		status = -EFAIL;
		LOG_ERROR_ARGS ("unable to launch ocfslsnr thread, error=%d",
				child_pid);
		goto bail;
	} else {
		init_completion (&(OcfsIpcCtxt.complete));
	}

      bail:
	LOG_EXIT_STATUS (status);
	return status;
}				/* ocfs_init_udp */

/*
 * ocfs_init_ipc_dlm()
 *
 */
int ocfs_init_ipc_dlm (ocfs_protocol protocol)
{
	int status = 0;

	LOG_ENTRY ();

	ocfs_init_ipc ();

	OcfsIpcCtxt.dlm_msg_size = OCFS_DLM_MAX_MSG_SIZE;
	OcfsIpcCtxt.version = OCFS_IPC_DLM_VERSION;

	switch (protocol) {
	    case OCFS_TCP:
		    status = -EINVAL;
		    break;

	    case OCFS_UDP:
		    ocfs_init_udp ();
		    break;

	    default:
		    status = -EINVAL;
		    break;
	}

	LOG_EXIT_STATUS (status);
	return status;
}				/* ocfs_init_ipc_dlm */

/*
 * ocfs_send_udp_msg()
 *
 */
int ocfs_send_udp_msg (ocfs_ipc_config_info * send, void *msg, __u32 msglen,
		       wait_queue_head_t * event)
{
	struct sockaddr_in sin;
	int status;

	LOG_ENTRY ();

	memset (&sin, 0, sizeof (sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = in_aton(send->ip_addr);
	sin.sin_port = htons(send->ip_port);

	LOG_TRACE_ARGS ("about to send to %s:%u\n", send->ip_addr,
			send->ip_port);

	if (!OcfsIpcCtxt.send_sock) {
		LOG_ERROR_STATUS (status = -EFAIL);
		goto bail;
	}

	status = ocfs_send_to (OcfsIpcCtxt.send_sock, (struct sockaddr *) &sin,
			       sizeof (sin), msg, msglen);
	if (status < 0) {
		LOG_ERROR_STATUS (status);
		goto bail;
	}

      bail:
	LOG_EXIT_STATUS (status);
	return status;
}				/* ocfs_send_udp_msg */

/*
 * ocfs_recv_udp_msg()
 *
 */
int ocfs_recv_udp_msg (ocfs_recv_ctxt * recv_ctxt)
{
	struct sockaddr_in sin;
	int sinlen;
	int status = -EFAIL;

	LOG_ENTRY ();

	/* Initialize the workitem with our worker routine and Q it. */
	INIT_TQUEUE (&recv_ctxt->ipc_tq, ocfs_dlm_recv_msg, recv_ctxt);

	sinlen = sizeof (struct sockaddr_in);
	memset (&sin, 0, sinlen);

	status = ocfs_recv_from (OcfsIpcCtxt.recv_sock,
				 (struct sockaddr *) &sin, &sinlen,
				 recv_ctxt->msg, (int *)&recv_ctxt->msg_len);
	if (status < 0) {
		if (status != -EBADF)
			LOG_ERROR_STATUS (status);
		goto bail;
	}

	LOG_TRACE_ARGS ("Received packet from: %d.%d.%d.%d\n",
			NIPQUAD (sin.sin_addr.s_addr));

	if (status == 0)
		schedule_task (&recv_ctxt->ipc_tq);

      bail:
	LOG_EXIT_STATUS (status);
	return status;
}				/* ocfs_recv_udp_msg */

/*
 * ocfs_send_bcast()
 *
 */
int ocfs_send_bcast (ocfs_super * osb, __u64 votemap, ocfs_dlm_msg * dlm_msg)
{
	int status = 0;
	__u32 nodemap;
	__u32 nodenum;
	ocfs_node_config_info *node;

	LOG_ENTRY_ARGS ("(votemap=0x%x)\n", LO(votemap));

	for (nodemap = LO(votemap), nodenum = 0; nodemap != 0;
	     					nodemap >>= 1, nodenum++) {
		if (nodenum == osb->node_num)
			continue;

		if (!(nodemap & 0x1))
			continue;

		node = osb->node_cfg_info[nodenum];
		if (node) {
			LOG_TRACE_ARGS ("Sending msg to node=%u, name=%s\n",
					nodenum, node->node_name);
			ocfs_dlm_send_msg (osb, &node->ipc_config, dlm_msg);
		}
	}

	LOG_EXIT_STATUS (status);
	return status;
}				/* ocfs_send_bcast */

/*
 * ocfs_dlm_send_msg()
 *
 */
void ocfs_dlm_send_msg (ocfs_super * osb, ocfs_ipc_config_info * send,
			ocfs_dlm_msg * dlm_msg)
{
	LOG_ENTRY ();

	ocfs_send_udp_msg (send, dlm_msg, dlm_msg->msg_len, NULL);

	LOG_EXIT ();
	return;
}				/* ocfs_dlm_send_msg */

/*
 * ocfs_init_udp_sock()
 *
 */
int ocfs_init_udp_sock (struct socket **send_sock, struct socket **recv_sock)
{
	struct sockaddr_in sin;
	int status = -EFAIL;
	ocfs_comm_info *comm;

	LOG_ENTRY ();

	/* Create Send Socket */
	status = sock_create (PF_INET, SOCK_DGRAM, IPPROTO_UDP, send_sock);
	if (status < 0) {
		LOG_ERROR_ARGS ("unable to create socket, error=%d", status);
		goto bail;
	}

	/* Bind Send Socket */
	memset (&sin, 0, sizeof (sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl (INADDR_ANY);
	sin.sin_port = htons (0);

	status = (*send_sock)->ops->bind (*send_sock, (struct sockaddr *) &sin,
					 sizeof (sin));
	if (status < 0) {
		LOG_ERROR_ARGS ("unable to bind socket, error=%d", status);
		goto bail;
	}

	/* Create Receive Socket */
	status = sock_create (PF_INET, SOCK_DGRAM, IPPROTO_UDP, recv_sock);
	if (status < 0) {
		LOG_ERROR_ARGS ("unable to create socket, error=%d", status);
		goto bail;
	}

	comm = &(OcfsGlobalCtxt.comm_info);

	/* Bind Receive Socket */
	memset (&sin, 0, sizeof (sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl (INADDR_ANY);
	sin.sin_port = htons (comm->ip_port);

	status = (*recv_sock)->ops->bind (*recv_sock, (struct sockaddr *) &sin,
					 sizeof (sin));
	if (status < 0) {
		LOG_ERROR_ARGS ("unable to bind socket, error=%d", status);
		goto bail;
	}

      bail:
	LOG_EXIT_STATUS (status);
	return status;
}				/* ocfs_init_udp_sock */

/*
 * ocfs_send_to()
 * 
 */
int ocfs_send_to (struct socket *sock, struct sockaddr *addr, int addrlen,
		  char *buf, int buflen)
{
	int error;
	struct msghdr msg;
	struct iovec iov;
	mm_segment_t oldfs;
	int status = 0;

	LOG_ENTRY ();

	iov.iov_base = buf;
	iov.iov_len = buflen;
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	msg.msg_control = NULL;
	msg.msg_controllen = 0;
	msg.msg_name = addr;
	msg.msg_namelen = addrlen;
	msg.msg_flags = 0;

	oldfs = get_fs ();
	set_fs (get_ds ());
	error = sock_sendmsg (sock, &msg, buflen);
	set_fs (oldfs);

	if (error < 0) {
		LOG_ERROR_ARGS ("unable to sendmsg, error=%d", error);
		status = -EFAIL;
	}

	LOG_EXIT_STATUS (status);
	return status;
}				/* OcfsSentTo */

/*
 * ocfs_recv_from()
 *
 */
int ocfs_recv_from (struct socket *sock, struct sockaddr *addr, int *addrlen,
		    char *buf, int *buflen)
{
	struct iovec iov;
	struct msghdr msg;
	int error;
	mm_segment_t oldfs;
	int status = -EFAIL;

	LOG_ENTRY ();

	msg.msg_control = NULL;
	msg.msg_controllen = 0;
	msg.msg_iovlen = 1;
	msg.msg_iov = &iov;
	iov.iov_len = *buflen;
	iov.iov_base = buf;
	msg.msg_name = addr;
	msg.msg_namelen = *addrlen;
	msg.msg_flags = 0;

	oldfs = get_fs ();
	set_fs (get_ds ());
	error = sock_recvmsg (sock, &msg, *buflen, msg.msg_flags);
	set_fs (oldfs);
	if (error < 0) {
		if (error == -ERESTARTSYS) {
			status = -EBADF;
			LOG_TRACE_STR ("Shutting down ipcdlm");
			goto bail;
		} else {
			LOG_ERROR_ARGS ("unable to recvmsg, error=%d", error);
			goto bail;
		}
	} else {
		if (msg.msg_namelen) {
			*addrlen = msg.msg_namelen;
			*buflen = iov.iov_len;
			status = 0;
			goto bail;
		}
	}

      bail:
	LOG_EXIT_STATUS (status);
	return status;
}				/* ocfs_recv_from */
