/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Enterprise Fibre Channel Host Bus Adapters.                     *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003 Emulex Corporation.                          *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

#include "fc_os.h"

#include "fc_hw.h"
#include "fc.h"

#include "fcdiag.h"
#include "fcfgparm.h"
#include "fcmsg.h"
#include "fc_crtn.h"   /* Core - external routine definitions */
#include "fc_ertn.h"   /* Environment - external routine definitions */

extern fc_dd_ctl_t  DD_CTL;
extern iCfgParam icfgparam[];
extern int lpfc_nethdr;

/* Routine Declaration - Local */
_local_ int        fc_mbuf_to_iocb(fc_dev_ctl_t *p_dev_ctl, fcipbuf_t *p_mbuf);
_local_ fcipbuf_t *fc_txq_put(fc_dev_ctl_t *p_dev_ctl, RING *rp,
                     fcipbuf_t *p_mbuf);
/* End Routine Declaration - Local */

/*****************************************************************************/
/*
 * NAME:     fc_ringtx_put
 *
 * FUNCTION: put xmit iocb onto the ring transmit queue.
 *
 * EXECUTION ENVIRONMENT: process and interrupt level.
 *
 * CALLED FROM:
 *      fc_els_cmd
 *
 * INPUT:
 *      binfo           - pointer to the device info area
 *  iocbq       - pointer to iocbq entry of xmit iocb
 *
 * RETURNS:  
 *  none
 */
/*****************************************************************************/
_static_ void
fc_ringtx_put(
RING        *rp,
IOCBQ       *iocbq)     /* pointer to iocbq entry */
{
   FC_BRD_INFO  * binfo;

   binfo = (FC_BRD_INFO * )rp->fc_binfo;
   if (rp->fc_tx.q_first) {
      ((IOCBQ * )rp->fc_tx.q_last)->q = (uchar * )iocbq;
      rp->fc_tx.q_last = (uchar * )iocbq;
   } else {
      rp->fc_tx.q_first = (uchar * )iocbq;
      rp->fc_tx.q_last = (uchar * )iocbq;
   }

   iocbq->q = NULL;
   rp->fc_tx.q_cnt++;

   return;

}   /* End fc_ringtx_put */


/*****************************************************************************/
/*
 * NAME:     fc_ringtx_get
 *
 * FUNCTION: get a packet off the ring transmit queue.
 *
 * EXECUTION ENVIRONMENT: interrupt level.
 *
 * CALLED FROM:
 *      fc_els_cmd
 *
 * INPUT:
 *      rp       - pointer to the ring to get an iocb from
 *
 * RETURNS:  
 *  NULL - no iocbs found
 *  iocb pointer - pointer to an iocb to transmit
 */
/*****************************************************************************/
_static_ IOCBQ *
fc_ringtx_get(
RING         *rp)
{
   FC_BRD_INFO  * binfo;
   NODELIST     * nlp;
   IOCBQ        * p_first = NULL;
   IOCBQ        * prev = NULL;
   uchar        * daddr;
   ushort xri;

   binfo = (FC_BRD_INFO * )rp->fc_binfo;
   if (rp->fc_tx.q_first) {
      p_first = (IOCBQ * )rp->fc_tx.q_first;

      /* Make sure we already have a login and exchange to the remote node */
      while (p_first->iocb.ulpCommand == 0) {
         if (rp->fc_ringno == FC_IP_RING) {
            NETHDR    * np;

            /* check to see if nlplist entry exists yet */
            np = (NETHDR * )(fcdata(((fcipbuf_t * )(p_first->bp))));
            daddr = np->fc_destname.IEEE;
            if ((xri = fc_emac_lookup(binfo, daddr, &nlp))) {
               /* exchange to destination already exists */
               if (binfo->fc_flag & FC_SLI2)
                  p_first->iocb.ulpCommand = CMD_XMIT_SEQUENCE64_CX;
               else
                  p_first->iocb.ulpCommand = CMD_XMIT_SEQUENCE_CX;
               p_first->iocb.ulpContext = xri;
               p_first->info = (uchar * )nlp;
               break;
            }
         }

         /* loop past continuation iocbs */
         while (p_first->iocb.ulpLe == 0) {
            prev = p_first;
            if ((p_first = (IOCBQ * )p_first->q) == 0) {
               return(0);
            }
         }
         prev = p_first;
         if ((p_first = (IOCBQ * )p_first->q) == 0) {
            return(0);
         }
      }

      /* adjust last if necessary */
      if (p_first->q == 0) {
         rp->fc_tx.q_last = (uchar * )prev;
      }

      /* remove iocb chain to process */
      if (prev == 0) {
         rp->fc_tx.q_first = p_first->q;
      } else {
         prev->q = (uchar * )p_first->q;
      }

      p_first->q = NULL;
      rp->fc_tx.q_cnt--;
   }
   return(p_first);

}   /* End fc_ringtx_get */


/*****************************************************************************/
/*
 * NAME:     fc_ringtx_drain
 *
 * FUNCTION: get all packets off the ring transmit queue.
 *
 * EXECUTION ENVIRONMENT: interrupt level.
 *
 * NOTES:
 *
 * CALLED FROM:
 *      fc_els_cmd
 *
 * INPUT:
 *      binfo       - pointer to the device info area
 *
 * RETURNS:  
 *  NULL - no match found
 *  mbuf pointer - pointer to a mbuf chain which contains a packet.
 */
/*****************************************************************************/
_static_ IOCBQ *
fc_ringtx_drain(
RING         *rp)
{
   FC_BRD_INFO  * binfo;
   IOCBQ        * p_first;
   IOCBQ        * prev;

   binfo = (FC_BRD_INFO * )rp->fc_binfo;
   p_first = (IOCBQ * )rp->fc_tx.q_first;
   if (p_first) {
      prev = (IOCBQ * )p_first->q;

      /* remove iocb chain to process */
      if (prev == 0) {
         rp->fc_tx.q_first = 0;
         rp->fc_tx.q_last =  0;
      } else {
         rp->fc_tx.q_first = (uchar * )prev;
      }

      p_first->q = NULL;
      rp->fc_tx.q_cnt--;
   }

   return(p_first);

}   /* End fc_ringtx_drain */




/*****************************************************************************/
/*
 * NAME:     fc_ringtxp_put
 *
 * FUNCTION: put xmit iocb onto the ring pending queue.
 *
 * EXECUTION ENVIRONMENT: process and interrupt level.
 *
 * CALLED FROM:
 *      fc_elsp_cmd
 *
 * INPUT:
 *      rp          - pointer to the ring
 *  iocbq       - pointer to iocbq entry of xmit iocb
 *
 * RETURNS:  
 *  none
 */
/*****************************************************************************/
_static_ void
fc_ringtxp_put(
RING    *rp,
IOCBQ   *iocbq)     /* pointer to iocbq entry */
{
   fc_dev_ctl_t * p_dev_ctl;
   FC_BRD_INFO  * binfo;
   unsigned long iflag;

   binfo = (FC_BRD_INFO * )rp->fc_binfo;
   p_dev_ctl = (fc_dev_ctl_t *)binfo->fc_p_dev_ctl;

   iflag = lpfc_q_disable_lock(p_dev_ctl);
   if (rp->fc_txp.q_first) {
      ((IOCBQ * )rp->fc_txp.q_last)->q = (uchar * )iocbq;
      rp->fc_txp.q_last = (uchar * )iocbq;
   } else {
      rp->fc_txp.q_first = (uchar * )iocbq;
      rp->fc_txp.q_last = (uchar * )iocbq;

      /* start watchdog timer on first xmit only */
      if (rp->fc_ringno != FC_FCP_RING) {
         lpfc_q_unlock_enable(p_dev_ctl, iflag);
         RINGTMO = fc_clk_set((fc_dev_ctl_t *)(binfo->fc_p_dev_ctl),
            rp->fc_ringtmo, fc_cmdring_timeout, (void *)rp, 0);
         iflag = lpfc_q_disable_lock(p_dev_ctl);
      }
   }

   iocbq->q = NULL;
   rp->fc_txp.q_cnt++;

   lpfc_q_unlock_enable(p_dev_ctl, iflag);
   return;

}   /* End fc_ringtxp_put */


/*****************************************************************************/
/*
 * NAME:     fc_ringtxp_get
 *
 * FUNCTION: get a packet off the ring pending queue.
 *
 * EXECUTION ENVIRONMENT: interrupt level.
 *
 * CALLED FROM:
 *      fc_els_cmd
 *
 * INPUT:
 *      rp       - pointer to the ring
 *
 * RETURNS:  
 *  NULL - no match found
 *  iocbq pointer - pointer to iocbq which matches the iotag
 */
/*****************************************************************************/
_static_ IOCBQ *
fc_ringtxp_get(
RING    *rp,
ushort  iotag)      /* tag to match i/o */
{
   fc_dev_ctl_t * p_dev_ctl;
   FC_BRD_INFO   * binfo;
   IOCBQ         * iocbq;       /* pointer to iocbq entry */
   IOCBQ         * pq;          /* pointer to previous iocbq entry */
   IOCBQ         * save;            /* pointer to iocb entry of chain */
   unsigned long   iflag;

   binfo = (FC_BRD_INFO * )rp->fc_binfo;
   p_dev_ctl = (fc_dev_ctl_t *)binfo->fc_p_dev_ctl;
   pq = 0;
   save = 0;

   /* Right now this just loops through the linked list looking
    * for a match on iotag. This can get optimized in the future
    * to have iotag just index into an array.
    */
   iflag = lpfc_q_disable_lock(p_dev_ctl);
   iocbq = (IOCBQ * )(rp->fc_txp.q_first);
   while (iocbq) {
      /* do we match on iotag */
      if ((iocbq->iocb.ulpIoTag == iotag) || (iotag == 0)) {
         /* loop past continuation iocbs */
         while (iocbq->iocb.ulpLe == 0) {
            rp->fc_txp.q_cnt--;
            save = iocbq;
            if ((iocbq = (IOCBQ * )iocbq->q) == 0) {
               iocbq = save;
               break;
            }
         }
         save = iocbq;
         iocbq = (IOCBQ * )iocbq->q;

         save->q = 0;   /* NULL terminate iocb chain */

         /* Remove iocbq chain from list, adjust first, last and cnt */
         if (iocbq == 0)
            rp->fc_txp.q_last = (uchar * )pq;

         if (pq) {
            save = (IOCBQ * )pq->q;
            pq->q = (uchar * )iocbq;
         } else {
            save = (IOCBQ * )rp->fc_txp.q_first;
            rp->fc_txp.q_first = (uchar * )iocbq;
         }
         rp->fc_txp.q_cnt--;

         /* stop watchdog timer */
         if(RINGTMO) {
            lpfc_q_unlock_enable(p_dev_ctl, iflag);
            fc_clk_can((fc_dev_ctl_t *)(binfo->fc_p_dev_ctl), RINGTMO);
            iflag = lpfc_q_disable_lock(p_dev_ctl);
            RINGTMO = 0;
         }

         /* if xmits are still pending, restart the watchdog timer */
         if (rp->fc_txp.q_cnt > 0) {
            /* start watchdog timer */
            if (rp->fc_ringno != FC_FCP_RING) {
               lpfc_q_unlock_enable(p_dev_ctl, iflag);
               RINGTMO = fc_clk_set((fc_dev_ctl_t *)(binfo->fc_p_dev_ctl),
                  rp->fc_ringtmo, fc_cmdring_timeout, (void *)rp, 0);
               iflag = lpfc_q_disable_lock(p_dev_ctl);
            }
         }
         break;
      }

      pq = iocbq;
      iocbq = (IOCBQ * )iocbq->q;
   }

   lpfc_q_unlock_enable(p_dev_ctl, iflag);
   return(save);
}   /* End fc_ringtxp_get */


/*****************************************************************************/
/*
 * NAME:     fc_xmit
 *
 * FUNCTION: Fibre Channel driver output routine.
 *
 * EXECUTION ENVIRONMENT: process only
 *
 * NOTES:
 *
 * CALLED FROM:
 *      fc_output  fc_intr
 *
 * INPUT:
 *      p_dev_ctl       - pointer to device information.
 *  p_mbuf      - pointer to a mbuf (chain) for outgoing packets
 *
 * RETURNS:  
 *  0 - successful
 *  EAGAIN - transmit queue is full
 */
/*****************************************************************************/
int
fc_xmit(
fc_dev_ctl_t *p_dev_ctl,
fcipbuf_t *p_mbuf)
{
   fcipbuf_t     * p_cur_mbuf;
   fcipbuf_t     * buf_tofree;
   RING        * rp;
   FC_BRD_INFO * binfo;
   iCfgParam         * clp;

   binfo = &BINFO;
   clp = DD_CTL.p_config[binfo->fc_brd_no];
   if(clp[CFG_NETWORK_ON].a_current == 0)
      return(EIO);

   rp = &binfo->fc_ring[FC_IP_RING];
   buf_tofree = fc_txq_put(p_dev_ctl, rp, p_mbuf);
   if (NDDSTAT.ndd_xmitque_max < rp->fc_tx.q_cnt) {
      NDDSTAT.ndd_xmitque_max = rp->fc_tx.q_cnt;
   }

   /* xmit queue was totally full */
   if (buf_tofree == p_mbuf) {
      while (p_mbuf) {
         NDDSTAT.ndd_xmitque_ovf++;
         NDDSTAT.ndd_opackets_drop++;
         p_mbuf = fcnextpkt(p_mbuf);
      }

      /* send the packet(s) on the xmit queue */
      issue_iocb_cmd(binfo, rp, 0);

      return(EAGAIN);
   }

   /* xmit queue could not fit entire chain */
   while ((p_cur_mbuf = buf_tofree) != NULL) {
      NDDSTAT.ndd_xmitque_ovf++;
      NDDSTAT.ndd_opackets_drop++;
      buf_tofree = fcnextpkt(buf_tofree);
      fcnextpkt(p_cur_mbuf) = NULL;
      m_freem(p_cur_mbuf);
   }

   /* send the packet(s) on the xmit queue */
   issue_iocb_cmd(binfo, rp, 0);

   return(0);
}   /* End fc_xmit */


/*****************************************************************************/
/*
 * NAME:     fc_txq_put
 *
 * FUNCTION: put packets onto the transmit queue.
 *
 * EXECUTION ENVIRONMENT: process and interrupt level.
 *
 * NOTES:
 *
 * CALLED FROM:
 *      fc_xmit
 *
 * INPUT:
 *      p_dev_ctl       - pointer to the device information area
 *      rp              - pointer to the device information area
 *  p_mbuf      - pointer to a mbuf chain
 *
 * RETURNS:  
 *  NULL - all mbufs are queued.
 *  mbuf pointer - point to a mbuf chain which contains packets
 *             that overflows the transmit queue.
 */
/*****************************************************************************/
_local_ fcipbuf_t *
fc_txq_put(
fc_dev_ctl_t *p_dev_ctl,
RING         *rp,
fcipbuf_t  *p_mbuf)     /* pointer to a mbuf chain */
{
   FC_BRD_INFO  * binfo;
   fcipbuf_t  * p_last, *p_over, *p_next;
   int  room;

   room = rp->fc_tx.q_max - NDDSTAT.ndd_xmitque_cur;
   binfo = &BINFO;
   if (room > 0) {
      p_over = 0;
      p_next = p_mbuf;
      while (p_next) {
         p_last = fcnextpkt(p_next);
         fcnextpkt(p_next) = NULL;
         if (fc_mbuf_to_iocb(p_dev_ctl, p_next)) {
            fcnextpkt(p_next) = p_last;
            p_over = p_next;
            break;
         }
         p_next = p_last;
         if ( --room <= 0) {
            p_over = p_next;
            break;
         }
      }
      binfo->fc_flag &= ~FC_NO_ROOM_IP;
   } else {
      FCSTATCTR.xmitnoroom++;
      p_over = p_mbuf;

      if(!(binfo->fc_flag & FC_NO_ROOM_IP)) {
         /* No room on IP xmit queue */
         fc_log_printf_msg_vargs( binfo->fc_brd_no,
                &fc_msgBlk0605,                   /* ptr to msg structure */
                 fc_mes0605,                      /* ptr to msg */
                  fc_msgBlk0605.msgPreambleStr,   /* begin varargs */
                   FCSTATCTR.xmitnoroom);         /* end varargs */
      }
      binfo->fc_flag |= FC_NO_ROOM_IP;
   }

   return(p_over);

}   /* End fc_txq_put */



/*****************************************************************************/
/*
 * NAME:     fc_mbuf_to_iocb
 *
 * FUNCTION: converts and mbuf into an iocb cmd chain and put on transmit q
 *
 * EXECUTION ENVIRONMENT: process and interrupt
 *
 * NOTES:
 *
 * CALLED FROM:
 *
 * INPUT:
 *  p_dev_ctl   - pointer to the device information area
 *  p_mbuf      - pointer to a packet in mbuf
 *
 * RETURNS:  
 *  0 - OK
 *  -1 - error occurred during transmit
 */
/*****************************************************************************/
_local_ int
fc_mbuf_to_iocb(
fc_dev_ctl_t *p_dev_ctl,
fcipbuf_t  *p_mbuf)     /* pointer to the packet in mbuf */
{
   FC_BRD_INFO  * binfo;
   iCfgParam    * clp;
   uchar        * daddr;
   RING         * rp;
   IOCBQ        * temp;
   IOCBQ        * qhead, * qtail;
   IOCB     * cmd;
   NODELIST     * nlp;
   fcipbuf_t    * p_cur_mbuf;   /* pointer to current packet in mbuf */
   fcipbuf_t    * m_net;
   ushort   * sp1, * sp2;
   ULP_BDE64    * bpl, * topbpl;
   MATCHMAP * bmp;
   MATCHMAP * bmphead, *bmptail;
   MATCHMAP     * savebmp;
   void     * handle;
   emac_t       * ep;
   NETHDR       * np;
   int          i, j, mapcnt;
   int          count, firstbuflen;
   int          num_iocbs, num_bdes, numble;
   ushort       leftover, xri;
   uchar    isbcast, ismcast;

   binfo = &BINFO;
   rp = &binfo->fc_ring[FC_IP_RING];
   clp = DD_CTL.p_config[binfo->fc_brd_no];

   /* First get a temporary iocb buffers. temp will be
    * used for the first iocb entry XMIT_SEQUENCE, and will
    * be used for each successive IOCB_CONTINUE entry.
    * qhead will be saved for the return
    */
   if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == 0) {
      m_freem(p_mbuf);
      return(0);
   }

   fc_bzero((void *)temp, sizeof(IOCBQ));  /* initially zero the iocb entry */
   cmd = &temp->iocb;
   mapcnt = 0;
   numble = 0;
   qhead = 0;
   qtail = 0;
   leftover = 0;
   bmp = 0;
   topbpl = 0;
   if (binfo->fc_flag & FC_SLI2) {
      bmphead = 0;
      bmptail = 0;
      /* Allocate buffer for Buffer ptr list */
      if ((bmp = (MATCHMAP * )fc_mem_get(binfo, MEM_BPL)) == 0) {
         fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         m_freem(p_mbuf);
         return(0);
      }
      bpl = (ULP_BDE64 * )bmp->virt;
      cmd->un.xseq64.bdl.ulpIoTag32 = (uint32)0;
      cmd->un.xseq64.bdl.addrHigh = (uint32)putPaddrHigh(bmp->phys);
      cmd->un.xseq64.bdl.addrLow = (uint32)putPaddrLow(bmp->phys);
      cmd->un.xseq64.bdl.bdeFlags = BUFF_TYPE_BDL;
      temp->bpl = (uchar *)bmp;
   }
   else {
      bpl = 0;
      bmphead = 0;
      bmptail = 0;
   }

   if(lpfc_nethdr == 0) {
      ep = (emac_t * )(fcdata(p_mbuf));
      daddr = ep->dest_addr;

      /* We need to convert 802.3 header (14 bytes) into
       * fc network header (16 bytes). Since the header is at
       * the begining of the buffer, we need to allocate extra space.
       */

      count = fcpktlen(p_mbuf) + 2; /* total data in mbuf after copy */
      firstbuflen = fcdatalen(p_mbuf);
      /* Assume first data buffer holds emac and LLC/SNAP at a minimun */
      if (firstbuflen < sizeof(struct fc_hdr )) {
         FCSTATCTR.mbufcopy++;
         fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         if (bmp) {
            fc_mem_put(binfo, MEM_BPL, (uchar * ) bmp);
         }
         m_freem(p_mbuf);
         return(0);
      }


      /* Allocate a buffer big enough to hold the Fibre Channel header
        * and the LLC/SNAP header.
        */
      if ((m_net = (fcipbuf_t * )m_getclustm(M_DONTWAIT, MT_DATA,
          (sizeof(NETHDR) + sizeof(snaphdr_t)))) == 0) {
         fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
         if (bmp) {
            fc_mem_put(binfo, MEM_BPL, (uchar * ) bmp);
         }
         return(EIO);
      }
      fcsethandle(m_net, 0);

      np = (NETHDR * )fcdata(m_net);

      /* Copy data from emac_t header to network header */
      sp1 = (ushort * ) & np->fc_destname;
      *sp1++ = 0;
      np->fc_destname.nameType = NAME_IEEE;      /* IEEE name */
      sp2 = (ushort * )ep->dest_addr;

      if (*sp2 & SWAP_DATA16(0x8000))   /* Check for multicast */ {
         ismcast = 1;
         if (*sp2 != 0xffff)  /* Check for broadcast */
            isbcast = 0;
         else
            isbcast = 1;
      } else {
         ismcast = 0;
         isbcast = 0;
      }

      /* First copy over the dest IEEE address */
      *sp1++ = *sp2++;
      if (isbcast && (*sp2 != 0xffff))
         isbcast = 0;
      *sp1++ = *sp2++;
      if (isbcast && (*sp2 != 0xffff))
         isbcast = 0;
      *sp1++ = *sp2++;

      /* Next copy over the src IEEE address */
      sp1 = (ushort * ) & np->fc_srcname;
      *sp1++ = 0;
      np->fc_srcname.nameType = NAME_IEEE;       /* IEEE name */
      sp2 = (ushort * )binfo->fc_portname.IEEE;
      *sp1++ = *sp2++;
      *sp1++ = *sp2++;
      *sp1++ = *sp2++;

      sp2 = (ushort * )((uchar *)ep + sizeof(emac_t));

      /* Now Copy LLC/SNAP */
      *sp1++ = *sp2++;
      *sp1++ = *sp2++;
      *sp1++ = *sp2++;
      *sp1++ = *sp2++;

      p_cur_mbuf = m_net;
      fcsetdatalen(m_net, (sizeof(NETHDR) + sizeof(snaphdr_t)));

      fcincdatalen(p_mbuf, (-(sizeof(struct fc_hdr ))));

      fcdata(p_mbuf) += sizeof(struct fc_hdr );

      /* Fixup mbuf chain so data is in line */
      fcnextdata(m_net) = p_mbuf;
   }
   else {
      np = (NETHDR * )(fcdata(((fcipbuf_t * )(p_mbuf))));
      daddr = np->fc_destname.IEEE;
      count = fcpktlen(p_mbuf); 
      p_cur_mbuf = p_mbuf;
      m_net = p_mbuf;

      sp2 = (ushort * )daddr;
      if (*sp2 & SWAP_DATA16(0x8000))   /* Check for multicast */ {
         ismcast = 1;
         if (*sp2 != 0xffff)  /* Check for broadcast */
            isbcast = 0;
         else
            isbcast = 1;
      } else {
         ismcast = 0;
         isbcast = 0;
      }
   }

   num_iocbs = 0;  /* count number of iocbs needed to xmit p_mbuf */
   num_bdes = 2;   /* Will change to 3 for IOCB_CONTINUE */
   nlp = 0;

   /* 
    * While there's data left to send and we are not at the end of
    * the mbuf chain, put the data from each mbuf in the chain into
    * a seperate iocb entry.
    */
   while (count && p_cur_mbuf) {
      if (binfo->fc_flag & FC_SLI2) {
         qhead = temp;
         qtail = temp;
         /* Set to max number of ULP_BDE64's that fit into a bpl */
         /* Save the last BDE for a continuation ptr,  if needed */
         num_bdes = ((FCELSSIZE / sizeof(ULP_BDE64)) - 1);
         numble = 0;
         if (bmphead == 0) {
            bmphead = bmp;
            bmptail = bmp;
         } else {
            bmptail->fc_mptr = (uchar * )bmp;
            bmptail = bmp;
         }
         bmp->fc_mptr = 0;
      } else {
         if (qhead == 0) {
            qhead = temp;
            qtail = temp;
         } else {
            qtail->q = (uchar * )temp;
            qtail = temp;
         }
      }
      temp->q = 0;
      /*
        * copy data pointers into iocb entry 
        */
      for (i = 0; i < num_bdes; i++) {
         /* Skip mblk's with 0 data length */
         while (p_cur_mbuf && (fcdatalen(p_cur_mbuf) == 0))
            p_cur_mbuf = fcnextdata(p_cur_mbuf);  /* goto next mbuf in chain */

         if ((count <= 0) || (p_cur_mbuf == 0))
            break;

         if (leftover == 0) {
            mapcnt = fc_bufmap(p_dev_ctl, (uchar * )(fcdata(p_cur_mbuf)),
                (uint32)fcdatalen(p_cur_mbuf), binfo->physaddr, binfo->cntaddr, &handle);

            /* fill in BDEs for command */
            if (mapcnt <= 0) {
               cmd->ulpBdeCount = i;
               goto out;
            }

            /* Save dmahandle if one was returned */
            fcsethandle(p_cur_mbuf, handle);
         }

         for (j = leftover; j < mapcnt; j++) {
            if ((i + j - leftover) >= num_bdes) {
               i = num_bdes;
               leftover = j;
               goto lim;
            }
            if (binfo->fc_flag & FC_SLI2) {
               bpl->addrHigh = (uint32)putPaddrHigh(binfo->physaddr[j]);
               bpl->addrHigh = PCIMEM_LONG(bpl->addrHigh);
               bpl->addrLow = (uint32)putPaddrLow(binfo->physaddr[j]);
               bpl->addrLow = PCIMEM_LONG(bpl->addrLow);
               bpl->tus.f.bdeSize = binfo->cntaddr[j];
               bpl->tus.f.bdeFlags = BDE64_SIZE_WORD;
               bpl->tus.w = PCIMEM_LONG(bpl->tus.w);
               bpl++;
               numble++;
            } else {
               cmd->un.cont[i+j-leftover].bdeAddress = (uint32)putPaddrLow(binfo->physaddr[j]);
               cmd->un.cont[i+j-leftover].bdeSize = binfo->cntaddr[j];
               cmd->un.cont[i+j-leftover].bdeAddrHigh = 0;
               cmd->un.cont[i+j-leftover].bdeReserved = 0;
            }
         }

         i = i + j - leftover - 1;
         count -= fcdatalen(p_cur_mbuf);       /* adjust count of data left */
         leftover = 0;
         p_cur_mbuf = fcnextdata(p_cur_mbuf);  /* goto next mbuf in chain */
      }

lim:
      /* Fill in rest of iocb entry, all non-zero fields */

      cmd->ulpBdeCount = i;

      /* Setup command to use accordingly */
      if (++num_iocbs > 1) {
         if (!(binfo->fc_flag & FC_SLI2)) {
            cmd->ulpCommand = CMD_IOCB_CONTINUE_CN;
            temp->bp = 0;
            temp->info = 0;
         }
      } else {
         /* set up an iotag so we can match the completion to an iocb/mbuf */
         cmd->ulpIoTag = rp->fc_iotag++;
         if (rp->fc_iotag == 0) {
            rp->fc_iotag = 1;
         }

         /* Setup fibre channel header information */
         cmd->un.xrseq.w5.hcsw.Fctl = 0;
         cmd->un.xrseq.w5.hcsw.Dfctl = FC_NET_HDR;   /* network headers */
         cmd->un.xrseq.w5.hcsw.Rctl = FC_UNSOL_DATA;
         cmd->un.xrseq.w5.hcsw.Type = FC_LLC_SNAP;

         if (isbcast) {
            if (++NDDSTAT.ndd_ifOutBcastPkts_lsw == 0)
               NDDSTAT.ndd_ifOutBcastPkts_msw++;
            if (binfo->fc_flag & FC_SLI2)
               cmd->ulpCommand = CMD_XMIT_BCAST64_CN;
            else
               cmd->ulpCommand = CMD_XMIT_BCAST_CN;
            cmd->ulpContext = 0;
            nlp = 0;
         } else if (ismcast) {
            if (++NDDSTAT.ndd_ifOutMcastPkts_lsw == 0)
               NDDSTAT.ndd_ifOutMcastPkts_msw++;
            if (binfo->fc_flag & FC_SLI2)
               cmd->ulpCommand = CMD_XMIT_BCAST64_CN;
            else
               cmd->ulpCommand = CMD_XMIT_BCAST_CN;
            cmd->ulpContext = 0;
            nlp = 0;
         } else {
            if (++NDDSTAT.ndd_ifOutUcastPkts_lsw == 0)
               NDDSTAT.ndd_ifOutUcastPkts_msw++;

            /* data from upper layer has a full MAC header on it. We
              * need to match the destination address with the portname
              * field in our nlp table to determine if we already have an
              * exchange opened to this destination.
              */
            if (((xri = fc_emac_lookup(binfo, daddr, &nlp)) != 0) && 
                !(nlp->nlp_action & NLP_DO_RSCN) && 
                (nlp->nlp_bp == 0)) {
               /* exchange to destination already exists */
               if (binfo->fc_flag & FC_SLI2)
                  cmd->ulpCommand = CMD_XMIT_SEQUENCE64_CX;
               else
                  cmd->ulpCommand = CMD_XMIT_SEQUENCE_CX;
               cmd->ulpContext = xri;
               nlp->nlp_type |= NLP_IP_NODE;
            } else {    /* need to wait for exchange to destination */
               FCSTATCTR.frameXmitDelay++;
               cmd->ulpCommand = 0;
               cmd->ulpContext = 0;

               if ((binfo->fc_flag & FC_LNK_DOWN) || 
                   (binfo->fc_ffstate < rp->fc_xmitstate))
                  goto out;

               if (nlp == 0) {
                  /* A partial entry doesn't even exist, so initiate
                    * ELS login by sending a FARP
                    */
                  /* Add FARP code here */
                  fc_els_cmd(binfo, ELS_CMD_FARP, (void *)daddr,
                      (uint32)0, (ushort)0, (NODELIST *)0);
               } else {
                  if ((nlp->nlp_DID != Bcast_DID)  && 
                      !(nlp->nlp_action & NLP_DO_ADDR_AUTH) &&
                      !(nlp->nlp_action & NLP_DO_RSCN) &&
                      !(nlp->nlp_flag & (NLP_FARP_SND | NLP_REQ_SND | NLP_RPI_XRI))) {
                     /* If a cached entry exists, PLOGI first */
                     if ((nlp->nlp_state == NLP_LIMBO) ||
                         (nlp->nlp_state == NLP_LOGOUT)) {
                        fc_els_cmd(binfo, ELS_CMD_PLOGI,
                           (void *)((ulong)nlp->nlp_DID), (uint32)0, (ushort)0, nlp);
                     }
                     /* establish a new exchange */
                     if ((nlp->nlp_Rpi) && (nlp->nlp_Xri == 0)) {
                        nlp->nlp_flag |= NLP_RPI_XRI;
                        fc_create_xri(binfo, &binfo->fc_ring[FC_ELS_RING], nlp);
                     }
                  }
               }

               cmd = &temp->iocb;
               if (binfo->fc_flag & FC_SLI2) {
                  while (bpl && (bpl != (ULP_BDE64 * )bmp->virt)) {
                     bpl--;
                     fc_bufunmap(p_dev_ctl,
                        (uchar *)getPaddr(bpl->addrHigh, bpl->addrLow), 0, bpl->tus.f.bdeSize);
                  }
                  if (bmp) {
                     fc_mem_put(binfo, MEM_BPL, (uchar * ) bmp);
                  }
               } else {
                  for (i = 0; i < (int)cmd->ulpBdeCount; i++) {
                     fc_bufunmap(p_dev_ctl, (uchar *)((ulong)cmd->un.cont[i].bdeAddress), 0, (uint32)cmd->un.cont[i].bdeSize);
                  }
               }
               if(lpfc_nethdr == 0) {
                  /* Free Resources */
                  fcnextdata(m_net) = 0;
                  fcfreehandle(p_dev_ctl, m_net);
                  m_freem(m_net);

                  /* Put p_mbuf back the way it was, without NETHDR */
                  fcincdatalen(p_mbuf, sizeof(struct fc_hdr ));
                  fcdata(p_mbuf) -= sizeof(struct fc_hdr );
               }

               fcfreehandle(p_dev_ctl, p_mbuf);

               fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);

               /* save buffer till ELS login completes */

               if (nlp == 0) {
                  m_freem(p_mbuf);
                  return(0);
               }

               if (nlp->nlp_bp == 0) {
                  nlp->nlp_bp = (uchar * )p_mbuf;
               } else {
                  /* Only keep one mbuf chain per node "on deck" */
                  p_cur_mbuf = (fcipbuf_t * )nlp->nlp_bp;
                  nlp->nlp_bp = (uchar * )p_mbuf;
                  m_freem(p_cur_mbuf);
               }
               return(0);
            }
            cmd->ulpClass = nlp->id.nlp_ip_info;
         }

         num_bdes = 3;  /* in case IOCB_CONTINUEs are needed */
         temp->bp = (uchar * )m_net;
         temp->info = (uchar * )nlp;
      }

      cmd->ulpOwner = OWN_CHIP;

      /* is this the last iocb entry we will need */
      if ((count == 0) || (p_cur_mbuf == 0)) {
         temp = 0;
         cmd->ulpLe = 1;
         /* if so queue cmd chain to last iocb entry in xmit queue */
         if (rp->fc_tx.q_first == 0) {
            rp->fc_tx.q_first = (uchar * )qhead;
            rp->fc_tx.q_last  = (uchar * )qtail;
         } else {
            ((IOCBQ * )(rp->fc_tx.q_last))->q  = (uchar * )qhead;
            rp->fc_tx.q_last  = (uchar * )qtail;
         }
         rp->fc_tx.q_cnt += num_iocbs;
         NDDSTAT.ndd_xmitque_cur++;
         break;
      } else {
         cmd->ulpLe = 0;
      }

      /* get another iocb entry buffer */
      if (binfo->fc_flag & FC_SLI2) {
         /* Allocate buffer for Buffer ptr list */
         if ((bmp = (MATCHMAP * )fc_mem_get(binfo, MEM_BPL)) == 0) {
            goto out;
         }
         /* Fill in continuation entry to next bpl */
         bpl->addrHigh = (uint32)putPaddrHigh(bmp->phys);
         bpl->addrHigh = PCIMEM_LONG(bpl->addrHigh);
         bpl->addrLow = (uint32)putPaddrLow(bmp->phys);
         bpl->addrLow = PCIMEM_LONG(bpl->addrLow);
         bpl->tus.f.bdeFlags = BPL64_SIZE_WORD;
         numble++;
         if (num_iocbs == 1) {
            cmd->un.xseq64.bdl.bdeSize = (numble * sizeof(ULP_BDE64));
         } else {
            topbpl->tus.f.bdeSize = (numble * sizeof(ULP_BDE64));
            topbpl->tus.w = PCIMEM_LONG(topbpl->tus.w);
         }
         topbpl = bpl;
         bpl = (ULP_BDE64 * )bmp->virt;
         leftover = 0;
      } else {
         if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == 0) {
out:
            /* no more available, so toss mbuf by freeing
              * resources associated with qhead 
              */
            if (binfo->fc_flag & FC_SLI2) {
               num_bdes = ((FCELSSIZE / sizeof(ULP_BDE64)) - 1);
               bmp = bmphead;
               while (bmp) {
                  i = 0;
                  bpl = (ULP_BDE64 * )bmp->virt;
                  while (bpl && (i < num_bdes)) {
                     bpl++;
                     i++;
                     fc_bufunmap(p_dev_ctl,
                        (uchar *)getPaddr(bpl->addrHigh, bpl->addrLow), 0, bpl->tus.f.bdeSize);
                  }
                  savebmp = (MATCHMAP * )bmp->fc_mptr;
                  if (bmp) {
                     fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
                  }
                  bmp = savebmp;
               }

               fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
            } else {
               while (qhead) {
                  temp = qhead;
                  cmd = &temp->iocb;
                  for (i = 0; i < (int)cmd->ulpBdeCount; i++) {
                     fc_bufunmap(p_dev_ctl, (uchar *)((ulong)cmd->un.cont[i].bdeAddress), 0, (uint32)cmd->un.cont[i].bdeSize);
                  }
                  qhead = (IOCBQ * )temp->q;
                  fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);
               }
            }

            if(lpfc_nethdr == 0) {
               fcnextdata(m_net) = 0;
               fcfreehandle(p_dev_ctl, m_net);
               m_freem(m_net);

               /* Put p_mbuf back the way it was, without NETHDR */
               fcincdatalen(p_mbuf, sizeof(struct fc_hdr ));
               fcdata(p_mbuf) -= sizeof(struct fc_hdr );
            }

            fcfreehandle(p_dev_ctl, p_mbuf);

            if (binfo->fc_flag & FC_SLI2) {
               m_freem(p_mbuf);
               return(0);
            }
            return(EIO);
         }
         fc_bzero((void *)temp, sizeof(IOCBQ));
         cmd = &temp->iocb;
      }
   }

   if (binfo->fc_flag & FC_SLI2) {
      bpl->addrHigh = 0;
      bpl->addrLow = 0;
      bpl->tus.w = 0;
      cmd->ulpBdeCount = 1;
      if (num_iocbs == 1) {
         cmd->un.xseq64.bdl.bdeSize = (numble * sizeof(ULP_BDE64));
      } else {
         topbpl->tus.f.bdeSize = (numble * sizeof(ULP_BDE64));
         topbpl->tus.w = PCIMEM_LONG(topbpl->tus.w);
      }
   }

   if (temp)
      fc_mem_put(binfo, MEM_IOCB, (uchar * )temp);

   return(0);
}   /* End fc_mbuf_to_iocb */



/**********************************************/
/**  handle_xmit_cmpl                        **/
/**                                          **/
/**  Process all transmit completions        **/
/**                                          **/
/**********************************************/
_static_ int
handle_xmit_cmpl(
fc_dev_ctl_t *p_dev_ctl,
RING         *rp,
IOCBQ        *temp)
{
   FC_BRD_INFO * binfo;
   IOCB        * cmd;
   IOCBQ       * xmitiq;
   IOCBQ       * save;
   NODELIST    * nlp;
   fcipbuf_t   * p_mbuf;
   fcipbuf_t   * m_net;
   int       i, cnt;
   ULP_BDE64     * bpl;
   MATCHMAP  * bmp;
   DMATCHMAP     * indmp;

   cmd = &temp->iocb;
   binfo = &BINFO;
   if (++NDDSTAT.ndd_xmitintr_lsw == 0) {
      NDDSTAT.ndd_xmitintr_msw++;
   }

   /* look up xmit compl by IoTag */
   if ((xmitiq = fc_ringtxp_get(rp, cmd->ulpIoTag)) == 0) {
      FCSTATCTR.strayXmitCmpl++;
      /* Stray XmitSequence completion */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0606,                   /* ptr to msg structure */
              fc_mes0606,                      /* ptr to msg */
               fc_msgBlk0606.msgPreambleStr,   /* begin varargs */
                cmd->ulpCommand,
                 cmd->ulpIoTag);               /* end varargs */
      /* completion with missing xmit command */
      return(EIO);
   }

   if (rp->fc_ringno == FC_ELS_RING) {
      indmp = (DMATCHMAP * )xmitiq->bp;
      if (cmd->ulpStatus) {
         indmp->dfc_flag = -1;
      }
      else {
         indmp->dfc_flag = xmitiq->iocb.un.xseq64.bdl.bdeSize;
      }
      fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
      return(0);
   }


   NDDSTAT.ndd_xmitque_cur--;

   /* get mbuf ptr for completed xmit */
   m_net = (fcipbuf_t * )xmitiq->bp;

   /* check for first xmit completion in sequence */
   nlp = (NODELIST * ) xmitiq->info;

   if (cmd->ulpStatus) {
      uint32 did = 0;

      NDDSTAT.ndd_oerrors++;

      if (nlp)
         did = nlp->nlp_DID;
      /* Xmit Sequence completion error */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0607,                   /* ptr to msg structure */
              fc_mes0607,                      /* ptr to msg */
               fc_msgBlk0607.msgPreambleStr,   /* begin varargs */
                cmd->ulpStatus,
                 cmd->ulpIoTag,
                  cmd->un.ulpWord[4],
                   did);                       /* end varargs */
      if (nlp && (nlp->nlp_state >= NLP_LOGIN)) {
         /* If XRI in xmit sequence with status error matches XRI
          * in nlplist entry, we need to create a new one.
          */
         if ((nlp->nlp_Xri == cmd->ulpContext) && 
             !(nlp->nlp_flag & NLP_RPI_XRI)) {
            /* on xmit error, exchange is aborted */
            nlp->nlp_Xri = 0;  /* xri */
            /* establish a new exchange */
            if ((nlp->nlp_Rpi) && 
                (binfo->fc_ffstate == FC_READY)) {
               nlp->nlp_flag |= NLP_RPI_XRI;
               fc_create_xri(binfo, &binfo->fc_ring[FC_ELS_RING], nlp);
            }
         }
      }
   } else {
      if (++NDDSTAT.ndd_opackets_lsw == 0)
         NDDSTAT.ndd_opackets_msw++;

      if (m_net && 
          ((nlp && ((nlp->nlp_DID & CT_DID_MASK) != CT_DID_MASK)) ||
          (xmitiq->iocb.ulpCommand == CMD_XMIT_BCAST_CX))) {

         if(lpfc_nethdr == 0) {
            p_mbuf = fcnextdata(m_net);
            cnt = fcpktlen(p_mbuf);
         }
         else {
            p_mbuf = m_net;
            cnt = fcpktlen(p_mbuf) - sizeof(NETHDR); /* total data in mbuf */
         }

         NDDSTAT.ndd_obytes_lsw += cnt;
         if ((int)NDDSTAT.ndd_obytes_lsw < cnt)
            NDDSTAT.ndd_obytes_msw++;
      }
   }

   if (nlp && (nlp->nlp_DID == NameServer_DID)) {
      MATCHMAP   * mp;

      mp = (MATCHMAP * )m_net;
      if (binfo->fc_flag & FC_SLI2) {
         fc_mem_put(binfo, MEM_BPL, (uchar * )xmitiq->bpl);
      }
      fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
      fc_mem_put(binfo, MEM_BUF, (uchar * )mp);
      return(0);
   }

   /* Loop through iocb chain and unmap memory pages associated with mbuf */
   if (binfo->fc_flag & FC_SLI2) {
      MATCHMAP * savebmp;
      int   cnt;

      bmp = (MATCHMAP * )xmitiq->bpl;
      cnt = xmitiq->iocb.un.xseq64.bdl.bdeSize;
      while (bmp) {
         bpl = (ULP_BDE64 * )bmp->virt;
         while (bpl && cnt) {
            bpl->addrHigh = PCIMEM_LONG(bpl->addrHigh);
            bpl->addrLow = PCIMEM_LONG(bpl->addrLow);
            bpl->tus.w = PCIMEM_LONG(bpl->tus.w);
            switch (bpl->tus.f.bdeFlags) {
            case BPL64_SIZE_WORD:
               cnt = bpl->tus.f.bdeSize;
               bpl = 0;
               break;
            case BDE64_SIZE_WORD:
               fc_bufunmap(p_dev_ctl, (uchar *)getPaddr(bpl->addrHigh, bpl->addrLow), 0, bpl->tus.f.bdeSize);
               bpl++;
               cnt -= sizeof(ULP_BDE64);
               break;
            default:
               bpl = 0;
               cnt = 0;
               break;
            }
         }
         savebmp = (MATCHMAP * )bmp->fc_mptr;
         fc_mem_put(binfo, MEM_BPL, (uchar * )bmp);
         bmp = savebmp;
      }
      fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
   } else {
      while (xmitiq) {
         for (i = 0; i < (int)xmitiq->iocb.ulpBdeCount; i++) {
            fc_bufunmap(p_dev_ctl, (uchar *)((ulong)xmitiq->iocb.un.cont[i].bdeAddress), 0, (uint32)xmitiq->iocb.un.cont[i].bdeSize); 
         }
         save = (IOCBQ * )xmitiq->q;
         fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);
         xmitiq = save;
      }
   }

   /* free mbuf */
   if (m_net) {
      if(lpfc_nethdr == 0) {
         p_mbuf = fcnextdata(m_net);
         fcnextdata(m_net) = 0;
         fcfreehandle(p_dev_ctl, m_net);
         m_freem(m_net);

         /* Put p_mbuf back the way it was, without NETHDR */
         fcincdatalen(p_mbuf, sizeof(struct fc_hdr ));

         fcdata(p_mbuf) -= sizeof(struct fc_hdr );
      }
      else {
         p_mbuf = m_net;
      }

      fcfreehandle(p_dev_ctl, p_mbuf);
      m_freem(p_mbuf);
   }

   fc_restartio(p_dev_ctl, nlp);

   return(0);
}   /* End handle_xmit_cmpl */


/*
 * Issue an iocb command to create an exchange with the remote
 * specified by the NODELIST entry.
 */
_static_ int
fc_create_xri(
FC_BRD_INFO *binfo,
RING        *rp,
NODELIST    *nlp)
{
   IOCB  * icmd;
   IOCBQ * temp;

   /* While there are buffers to post */
   if ((temp = (IOCBQ * )fc_mem_get(binfo, MEM_IOCB)) == 0) {
      return(1);
   }
   fc_bzero((void *)temp, sizeof(IOCBQ));
   icmd = &temp->iocb;

   /* set up an iotag so we can match the completion to an iocb/mbuf */
   icmd->ulpIoTag = rp->fc_iotag++;
   if (rp->fc_iotag == 0) {
      rp->fc_iotag = 1;
   }
   icmd->ulpContext = nlp->nlp_Rpi;
   icmd->ulpLe = 1;

   icmd->ulpCommand = CMD_CREATE_XRI_CR;
   icmd->ulpOwner = OWN_CHIP;

   temp->bp = (uchar * )nlp;    /* used for delimiter between commands */

   FCSTATCTR.cmdCreateXri++;

   issue_iocb_cmd(binfo, rp, temp);
   return(0);
}   /* End fc_create_xri */


/*
 * Process a create_xri command completion.
 */
_static_ int
handle_create_xri(
fc_dev_ctl_t *p_dev_ctl,
RING         *rp,
IOCBQ        *temp)
{
   FC_BRD_INFO * binfo;
   IOCB        * cmd;
   NODELIST    * nlp;
   IOCBQ       * xmitiq;

   cmd = &temp->iocb;
   binfo = &BINFO;
   /* look up xmit compl by IoTag */
   if ((xmitiq = fc_ringtxp_get(rp, cmd->ulpIoTag)) == 0) {
      FCSTATCTR.strayXmitCmpl++;
      /* Stray CreateXRI completion */
      fc_log_printf_msg_vargs( binfo->fc_brd_no,
             &fc_msgBlk0608,                   /* ptr to msg structure */
              fc_mes0608,                      /* ptr to msg */
               fc_msgBlk0608.msgPreambleStr,   /* begin varargs */
                cmd->ulpCommand,
                 cmd->ulpIoTag);               /* end varargs */
      /* completion with missing xmit command */
      return(EIO);
   }

   /* check for first xmit completion in sequence */
   nlp = (NODELIST * ) xmitiq->bp;

   if (cmd->ulpStatus) {
      fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);

      nlp->nlp_flag &= ~NLP_RPI_XRI;

      fc_freenode_did(binfo, nlp->nlp_DID, 0);

      FCSTATCTR.xriStatErr++;
      return(EIO);
   }

   FCSTATCTR.xriCmdCmpl++;

   nlp->nlp_Xri = cmd->ulpContext;
   nlp->nlp_flag &= ~NLP_RPI_XRI;

   fc_mem_put(binfo, MEM_IOCB, (uchar * )xmitiq);

   fc_restartio(p_dev_ctl, nlp);
   return(0);
}   /* End handle_create_xri */


_static_ void
fc_restartio(
fc_dev_ctl_t *p_dev_ctl,
NODELIST     *nlp)
{
   FC_BRD_INFO  * binfo;
   RING         * rp;
   fcipbuf_t     * p_cur_mbuf;
   fcipbuf_t  * buf_tofree;

   binfo = &BINFO;
   rp = &binfo->fc_ring[FC_IP_RING];

   if (nlp) {
      if ((nlp->nlp_bp) && (nlp->nlp_Xri)) {
         p_cur_mbuf = (fcipbuf_t * )nlp->nlp_bp;
         nlp->nlp_bp = 0;
         buf_tofree = fc_txq_put(p_dev_ctl, rp, p_cur_mbuf);
         while ((p_cur_mbuf = buf_tofree) != 0) {
            NDDSTAT.ndd_opackets_drop++;
            buf_tofree = fcnextpkt(buf_tofree);
            fcnextpkt(p_cur_mbuf) = NULL;
            m_freem(p_cur_mbuf);
         }
      }
   }

   /* Is there a xmit waiting to be started */
   if (rp->fc_tx.q_first) {
      /* If so, start it */
      issue_iocb_cmd(binfo, rp, 0);
   }

   /* If needed */
}   /* End fc_restartio */


