/* LVCool feature.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This feature has been developped using the code of the 'lvcool'
 * utility provided by Martin Peters <mpet@bigfoot.de>.
 * Volker Schmidt wrote a small patch to integrate the LVCool code into
 * the Linux kernel. This code worked, but lacked some functionalities,
 * and caused problems with some of the softwares I used (sound players
 * for example).
 * So, I decided to improve it, and make it cleaner, and better integrated
 * in the kernel sources.
 *
 * Notes:
 * - The original Linux LVCool sources, as well as versions for other OS,
 * and a ton of good explanations on how this feature is implemented
 * can be found at: http://www.vcool.de
 * - The system idle percentage evaluation code is largely inspired by
 * similar code from 'apm.c' (code from Andreas Steinmetz <ast@domdv.de>
 * I think).
 * 
 *
 * History:
 * 0.1  (because I fear to start it at 1.0): first version.
 *      Compared to the initial patch this code is derived from, this
 *      version adds:
 *      - Integrated as a the power mgt idle function,
 *      - Kernel build option,
 *      - Kernel boot parameter to disable or tune the feature,
 *      - Automatic disabling of the feature, depending on the current
 *        system load (idle percentage).
 */

#include <linux/config.h>
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/pci_ids.h>
#include <linux/pm.h>

#include <asm/io.h>

/* #define LVCOOL_DEBUG */

#define DEFAULT_IDLE_THRESHOLD	98
#define DEFAULT_IDLE_PERIOD	(10 * HZ)
#define MAX_IDLE_PERIOD 	100

static int idle_threshold = DEFAULT_IDLE_THRESHOLD;
static int idle_period = DEFAULT_IDLE_PERIOD;

static void (*sys_idle)(void);
    
#define IDLE_CALC_LIMIT   (MAX_IDLE_PERIOD * HZ)
 
void lvcool_idle(void);
extern void default_idle(void);


static int Reg_PL2;
static int lvcool_disabled = 0;
 
static __u32 __init PCIRead(int reg, int fn, int dev,int bus)
{
        __u32 r = 0x80000000;
        __u32 ret,org;
 
        r |= (( bus & 0xff) <<16);
        r |= (( dev & 0x1f) <<11);
        r |= (( fn &  0x7 ) << 8);
        r |= (( reg & 0xfc) );
 
        org = inl(0xcf8);
        outl(r, 0xcf8);
        ret = inl(0xcf8+4);
        outl(org, 0xcf8);
 
        return ret;
}
 
static void __init PCIWrite(__u32 val, int reg, int fn, int dev,int bus)
{
        __u32 r = 0x80000000;
        __u32 org;
 
        r|=(( bus & 0xff) <<16);
        r|=(( dev & 0x1f) <<11);
        r|=(( fn &  0x7 ) << 8);
        r|=(( reg & 0xfc) );
 
        org = inl(0xcf8);
        outl(r,0xcf8);
        outl(val,0xcf8+4);
        outl(org,0xcf8);
}

static int __init lvcool_setup(char *str)
{       
	while ((str != NULL) && (*str != '\0')) {
                if (strncmp(str, "off", 3) == 0) {
			lvcool_disabled = 1;
                } else if (strncmp(str, "on", 2) == 0) {
			lvcool_disabled = 0;
		} else if ((strncmp(str, "idle-threshold=", 15) == 0) ||
                           (strncmp(str, "idle_threshold=", 15) == 0)) {
                        int v = simple_strtol(str + 15, NULL, 0);
                        if ((unsigned int) v > 100) {
                                printk(KERN_NOTICE "lvcool: Invalid treshold parameter value.\n");
                        } else {
                                idle_threshold = v;
                        }
		} else if ((strncmp(str, "idle-period=", 12) == 0) ||
                           (strncmp(str, "idle_period=", 12) == 0)) {
                        int v = simple_strtol(str + 12, NULL, 0);
                        if (v < 0 || v > MAX_IDLE_PERIOD) {
                                printk(KERN_NOTICE "lvcool: Invalid period parameter value.\n");
                        } else {
                                idle_period = v * HZ;
                        }
                }

                /* Search for any other parameter separator. */
                str = strchr(str, ',');

                /* If we found one, skip it, as well as leading spaces. */
		if (str != NULL)
			str += strspn(str, ", \t");

        }
        return 1;
}
  
__setup("lvcool=", lvcool_setup);

int __init lvcool_pci_init(void)
{
        if (lvcool_disabled || (idle_threshold == 100)) {
                printk(KERN_NOTICE "lvcool: Disabled on user request\n");
                return 0;
        } else {
                int nb_b=0, nb_d=0, nb_f=0; /* north bridge */
                int sb_b=0, sb_d=7, sb_f=4; /* south bridge */
 
                __u32 res;
 
                int found = 1;
 
                /* First check processor */
                /* must be an AMD Athlon or Duron */
    
                if (!(boot_cpu_data.x86_vendor == X86_VENDOR_AMD &&
                      boot_cpu_data.x86 == 6)) {
                        printk(KERN_NOTICE "lvcool: Disabled because processor is not an Athlon/Duron.\n");
                        return 0;
                }
 
                /* quick search */

/* Check for VIA 8371 northbridge */                
                res = PCIRead(0,0,0,0);
                if ((res!=0x03051106) &&
                    (res!=0x03911106))
                { /* VIA northbridge? */
                        res = PCIRead(0,0,7,0);
                        if (res != 0x06861106)
                        {
/* Gigabyte GA-7DXR (AMD761+VIA686)? - contributed by Dorian <d@netsurf.de> */
                                found=0;
                        }
                }
                
/* Check for VIA 686 southbridge */ 
                if (found)
                {
                        res = PCIRead(0,4,7,0); /* VIA southbridge? */
                        if (res != 0x30571106)
                        {
                                res = PCIRead(0,4,4,0);
                                if (res == 0x30571106) /* Asus A7V133 */
                                        sb_d=4;
                                else
                                        found=0;
                        }
                }
 
                /* if not found (shouldn't happen I think) - scan bus */
 
                if (!found) {
                        int bus,dev,fun,i=0;
 
                        for (bus = 0;bus < 255;bus++) {
                                for (dev = 0;dev < 32;dev++) {
                                        for (fun = 0;fun < 7;fun++) {
                                                res = PCIRead(0,fun,dev,bus);
                                                if (res == 0xffffffff)
                                                        continue;
 
                                                if ((res == 0x03051106) ||
                                                    (res == 0x03911106)) {
                                                        nb_b = bus;
                                                        nb_d = dev;
                                                        nb_f = fun;
                                                        i += 1;
                                                }
                                                
                                                if (res == 0x30571106) {
                                                        sb_b=bus;
                                                        sb_d=dev;
                                                        sb_f=fun;
                                                        i+=2;
                                                }
                                                if (i >= 3) break;
                                        }
                                        if (i >= 3) break;
                                }
                                if (i >= 3) break;
                        }
                        if (i != 3)
                                found=0;
                }

                if (!found) {
                        printk(KERN_NOTICE "lvcool: Disabled because no adequate chipset detected.\n");
                        return 0;
                }
                
                /* enable ACPI and find I/O-Space */
 
                res = PCIRead(0x41,sb_f,sb_d,sb_b);
 
                if ((res & 0x8000) == 0)
                {
                        res |= 0x8000;
                        PCIWrite(res,0x41,sb_f,sb_d,sb_b);
                }
 
                res = PCIRead(0x48,sb_f,sb_d,sb_b);
                res &= 0xff80;
 
                Reg_PL2 = res + 0x14;
 
                /* now set "enable bus Disconnect when STPGNT detected" bit" */
 
                res = PCIRead(0x52,nb_f,nb_d,nb_b);
                res |= 0x00800000;
                PCIWrite(res,0x52,nb_f,nb_d,nb_b);
                
                printk(KERN_NOTICE "lvcool: Enabling special AMD Athlon/VIA KT133(A) idle loop (ACPI C2 mode)\n");
/*                  lvcool_init2(); */
                
                return 1;
        }
}


int __init lvcool_init(void)
{
        if (lvcool_pci_init()) {
                sys_idle = pm_idle;
                pm_idle = lvcool_idle;
                return 1;
        }
        return 0;
}

void lvcool_idle(void)
{
	static int use_lvcool_idle = 0;
	static unsigned int last_jiffies = 0;
	static unsigned int last_stime = 0;

        /* Compute the number of jiffies since our last check. */
	unsigned int jiffies_since_last_check = jiffies - last_jiffies;

#ifdef LVCOOL_DEBUG
        int last_use_lvcool_idle = use_lvcool_idle;
#endif
        
        while (!current->need_resched) {  
                if (jiffies_since_last_check > IDLE_CALC_LIMIT) {
                        /* If we didn't run during a period longer than
                           IDLE_CALC_LIMIT, that means the system is far
                           from idle, so disable the lvcool feature.
                        */
                        use_lvcool_idle = 0;
                        last_jiffies = jiffies;
                        last_stime = current->times.tms_stime;
                } else if (jiffies_since_last_check > idle_period) {
                        /* Recompute the new idle percentage at each
                           idle_period only
                        */            
                        unsigned int idle_percentage;

                        /* Compute the nb of jiffies spent executing
                           this task.
                        */
                        idle_percentage = current->times.tms_stime - last_stime;

                        /* Divided by the total of elapsed jiffies, we get
                           the idle percentage.
                        */
                        idle_percentage *= 100;
                        idle_percentage /= jiffies_since_last_check;

                        /* Our lvcool feature is eanbled only if the idle
                           percentage is upper than the threshold.
                        */
                        use_lvcool_idle = (idle_percentage > idle_threshold);
                        last_jiffies = jiffies;
                        last_stime = current->times.tms_stime;
                }
#ifdef LVCOOL_DEBUG
                if (last_use_lvcool_idle != use_lvcool_idle) {
                    if (use_lvcool_idle) {
                         printk(KERN_NOTICE "lvcool: Enabling\n");
                    } else {
                         printk(KERN_NOTICE "lvcool: Disabling\n");
                    }
                    last_use_lvcool_idle = use_lvcool_idle;
                }
#endif
                do {            
                        if (use_lvcool_idle) {
                                inb(Reg_PL2);
                        } else if (sys_idle) {
                                sys_idle();
                        } else {
                                default_idle();
                        }
                        jiffies_since_last_check = jiffies - last_jiffies;
                } while (!current->need_resched &&
                         (jiffies_since_last_check <= idle_period));
	}
}
