/* EntryConfiguration.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.menus;

import java.util.Set;

/**
 * Provides configuration information for the menu builder.
 * Every type of entry in the XML configuration file corresponds
 * to an object of this type, except the root elements. 
 */
abstract class EntryConfiguration {
    
    //
    private String predicateKey;
    
    
    //
    private boolean predicatePositive;
    
    
    /**
     * Predicate key associated with this entry. If non-null, this key can 
     * be used to indicate to the menu builder
     * whether this entry should be present or not.
     */
    public String getPredicateKey () {
        return this.predicateKey;
    }
    
    /**
     * If a predicate is associated with this entry, then this flag indicates
     * whether the predicate acts positively or negatively. A positive predicate
     * makes the entry appear when present. A negative predicate makes the entry
     * disappear.
     */
     public boolean isPredicatePositive() {
         return this.predicatePositive;
     }
     
     /**
      * Set a predicate key for this configuration and indicate whether
      * it is positive or negative.
      */
     public void setPredicate (String key, boolean positive) {
         this.predicateKey = key;
         this.predicatePositive = positive;
     }
     
     /**
      * Given the set of predicates, should this configuration be
      * instantiated?
      */
     public boolean satisfies (Set<String> predicates) {
         return predicateKey == null ||
                 predicatePositive == predicates.contains(predicateKey);
     }
     
    /**
     * Accept a visitor on this object.
     */
    public abstract void accept (ConfigurationVisitor visitor) ;
    
}
