/*
 * Copyright (C) The MX4J Contributors.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.connector.rmi;

import java.io.Serializable;
import java.lang.reflect.UndeclaredThrowableException;
import java.rmi.NoSuchObjectException;
import java.rmi.RemoteException;
import javax.management.Notification;
import javax.management.NotificationListener;

/**
 * This class is serialized from client to the server, and forwards calls to remote listener. <p>
 * It shadows the fact that the listener is an RMI remote object.
 *
 * @version $Revision: 1.3 $
 */
public abstract class RemoteNotificationListenerSupport implements NotificationListener, Serializable
{
   /**
    * When this class is serialized to the server, the stub of this remote object
    * will be sent to the server, enabling it to call back the client.
    * This must be the ONLY data member.
    */
   private RemoteNotificationListener m_clientSideListener;

   public RemoteNotificationListenerSupport(NotificationListener clientListener)
   {
      m_clientSideListener = new RemoteNotificationListenerImpl(clientListener);
   }

   protected abstract void export() throws RemoteException;

   protected abstract void unexport() throws NoSuchObjectException;

   protected RemoteNotificationListener getRemoteNotificationListener()
   {
      return m_clientSideListener;
   }

   public final void handleNotification(Notification notification, Object handback)
   {
      // We are on the server
      // This is a remote call from the server to the client
      try
      {
         m_clientSideListener.handleNotification(notification, handback);
      }
      catch (RemoteException x)
      {
         // Wrap this remote exception in a RuntimeException
         throw new UndeclaredThrowableException(x);
      }
   }

   public int hashCode()
   {
      return m_clientSideListener.hashCode();
   }

   public boolean equals(Object obj)
   {
      if (obj == null) return false;
      if (obj == this) return true;

      try
      {
         RemoteNotificationListenerSupport other = (RemoteNotificationListenerSupport)obj;
         return m_clientSideListener.equals(other.m_clientSideListener);
      }
      catch (ClassCastException x)
      {
         return false;
      }
   }
}
