/*
 * Copyright (c) 2007, intarsys consulting GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of intarsys nor the names of its contributors may be used
 *   to endorse or promote products derived from this software without specific
 *   prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.intarsys.tools.reporter;

/**
 * An {@link IReporter} that may be used to intercept the reporting information
 * and "replay" them later on on another {@link IReporter} instance.
 * <p>
 * This is useful if an object receiving reporting information may be
 * dynamically multiplexed on many reporting events emitting instances, for
 * example a window showing a currently active object.
 */
public class ReplayReporter implements IReporter, IReporterSupport {

	private ReportDispatcher dispatcher = new ReportDispatcher(this);

	private String reportStatusMessage;

	private int reportStatusStyle;

	private String reportActivityMessage;

	private int reportActivityStyle;

	private String reportProgressMessage;

	private int reportProgressPercent;

	private int reportProgressStyle;

	private String reportErrorTitle;

	private String reportErrorMessage;

	private Throwable reportErrorThrowable;

	private int reportErrorStyle;

	private String reportMessageTitle;

	private String reportMessageMessage;

	private int reportMessageStyle;

	public void addReporter(IReporter reporter) {
		dispatcher.addReporter(reporter);
		replay(reporter);
	}

	public void removeReporter(IReporter reporter) {
		dispatcher.removeReporter(reporter);
	}

	protected void replay(IReporter reporter) {
		if (reportActivityMessage != null) {
			reporter.reportActivityStart(reportActivityMessage,
					reportActivityStyle);
		}
		if (reportProgressMessage != null) {
			reporter.reportProgress(reportProgressMessage,
					reportProgressPercent, reportProgressStyle);
		}
		if (reportStatusMessage != null) {
			reporter.reportStatus(reportStatusMessage, reportStatusStyle);
		}
		if (reportMessageMessage != null) {
			reporter.reportMessage(reportMessageTitle, reportMessageMessage,
					reportMessageStyle);
		}
		if (reportErrorMessage != null) {
			reporter.reportError(reportErrorTitle, reportErrorMessage,
					reportErrorThrowable, reportErrorStyle);
		}
	}

	public void reportActivityEnd() {
		reportActivityMessage = null;
		dispatcher.reportActivityEnd();
	}

	public void reportActivityStart(String activity, int style) {
		reportActivityMessage = activity;
		reportActivityStyle = style;
		dispatcher.reportActivityStart(activity, style);
	}

	public void reportError(String title, String message, Throwable t, int style) {
		reportErrorTitle = title;
		reportErrorMessage = message;
		reportErrorThrowable = t;
		reportErrorStyle = style;
		dispatcher.reportError(title, message, t, style);
	}

	public void reportMessage(String title, String message, int style) {
		reportMessageTitle = title;
		reportMessageMessage = message;
		reportMessageStyle = style;
		dispatcher.reportMessage(title, message, style);
	}

	public void reportProgress(String message, int percent, int style) {
		if (percent < 0) {
			reportProgressMessage = null;
			reportProgressPercent = -1;
			reportProgressStyle = style;
		} else {
			reportProgressMessage = message;
			reportProgressPercent = percent;
			reportProgressStyle = style;
		}
		dispatcher.reportProgress(message, percent, style);
	}

	public void reportStatus(String message, int style) {
		reportStatusMessage = message;
		reportStatusStyle = style;
		dispatcher.reportStatus(message, style);
	}

}
