/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: UndoableToggleCommand.java,v 1.7 2006/02/26 00:59:10 pietschy Exp $
 */
package org.pietschy.command.undo;

import org.pietschy.command.CommandManager;
import org.pietschy.command.ToggleCommand;
import org.pietschy.command.ToggleVetoException;

import javax.swing.event.UndoableEditListener;
import javax.swing.undo.UndoableEdit;
import javax.swing.undo.UndoableEditSupport;

/**
 * UndoableToggleCommands provide a default implementation for toggle commands that can be undone.
 * UndoableToggleCommands implement {@link UndoableEventSource} and can be monitored by an
 * {@link UndoController}.
 * <p>
 * Subclasses must implement {@link #performEdit(boolean)} to handle the selection process.
 *
 * @see #performEdit(boolean)
 * @see #applySelection(boolean)
 */
public abstract class
UndoableToggleCommand
extends ToggleCommand
implements UndoableEventSource
{
   private UndoableEditSupport undoSupport;

   /**
    * Creates a new anonymous UndoableToggleCommand.  Anonymous commands must be fully programatically
    * generated and can only be added to groups manually by calling
    * <code>groupCommand.installFace(myAnonymousCommand)</code>.
    */
   public UndoableToggleCommand()
   {
      init();
   }

   /**
    * Creates a new command with the speicifed Id that is bound to the
    * {@link CommandManager#defaultInstance()}.
    */
   public UndoableToggleCommand(String commandId)
   {
      super(commandId);
      init();
   }

   /**
    * Creates a new command with the specified id that is bound to the
    * specified {@link CommandManager}.
    */
   protected
   UndoableToggleCommand(CommandManager commandManager, String commandId)
   {
      super(commandManager, commandId);
      init();
   }

   private void
   init()
   {
      undoSupport = new UndoableEditSupport(this);
   }


   /**
    * Delegates to {@link #performEdit(boolean)}
    *
    * @param selected the new desired state of the toggle
    * @throws ToggleVetoException if the new state can't be fullfilled.
    */
   protected final void
   handleSelection(boolean selected)
   throws ToggleVetoException
   {
      UndoableEdit edit = performEdit(selected);
      if (edit != null)
         postEdit(edit);
   }

   /**
    * Subclasses must override to perform the actual edit.  This method behaves similar to {@link org.pietschy.command.ToggleCommand#handleSelection(boolean)}
    * in that it can throw a {@link org.pietschy.command.ToggleVetoException} to cancel the edit.
    * <p>
    * Please note that the edit object should call {@link #applySelection(boolean)} to update the state of this command
    * without causing a new UndoableEdit to be created.  For example
    * <pre>
    * class MyToggle.ToggleUndoableEdit extends UndoableEdit
    * {
    *    public void undo()
    *    {
    *       // undo any command specific stuff...
    *       ...
    *       // and reset our state without causeing any side effects..
    *       <b>applySelection(oldState);</b>
    *    }
    * }
    * </pre>
    *
    * @param selected the new selected state of the command.
    * @return an UndoableEdit object that can undo and redo the commands action.
    * @throws ToggleVetoException if the toggle action can't proceed.
    * @see #applySelection(boolean)
    */
   protected abstract UndoableEdit
   performEdit(boolean selected)
   throws ToggleVetoException;


   /**
    * Notifies all the {@link javax.swing.event.UndoableEditListener}s of the undoable event.
    * @param e the undoable event.
    */
   protected void
   postEdit(UndoableEdit e)
   {
      undoSupport.postEdit(e);
   }


   /**
    * Adds a new {@link javax.swing.event.UndoableEditListener} to this command.
    *
    * @param l the listener to installFace.
    * @see UndoController
    */
   public void
   addUndoableEditListener(UndoableEditListener l)
   {
      undoSupport.addUndoableEditListener(l);
   }

   /**
    * Removes an {@link javax.swing.event.UndoableEditListener} from this command.
    *
    * @param l the listener to remove.
    * @see UndoController
    */
   public void
   removeUndoableEditListener(UndoableEditListener l)
   {
      undoSupport.removeUndoableEditListener(l);
   }

   /**
    * Gets the {@link javax.swing.event.UndoableEditListener}s regstered with this command.
    *
    * @see UndoController
    */
   public UndoableEditListener[]
   getUndoableEditListeners()
   {
      return undoSupport.getUndoableEditListeners();
   }

}
