//LabPlot : WaveletListDialog.cc

#include <qinputdialog.h>
#include "WaveletListDialog.h"
#include "defs.h"

#ifdef HAVE_GSL16
#include <gsl/gsl_wavelet.h>
#include <gsl/gsl_wavelet2d.h>
#endif

#include "waveletitems.h"

WaveletListDialog::WaveletListDialog(MainWin *m, const char *name)
	: ListDialog(m, name)
{
	kdDebug()<<"WaveListDialog()"<<endl;
	setCaption(i18n("Wavelet Dialog"));
	KConfig *config = mw->Config();
	config->setGroup( "Wavelet" );

	QTabWidget *tw = new QTabWidget(vbox);
	QVBox *tab1 = new QVBox(tw);

	QHBox *hb = new QHBox(tab1);
	new QLabel(i18n("Transform : "),hb);
	transform = new KComboBox(hb);
	QStringList tlist;
	tlist << i18n("forward") << i18n("backward");
	transform->insertStringList(tlist);
	transform->setCurrentItem(config->readNumEntry("Transform",0));

	form = new KComboBox(hb);
	QStringList flist;
	flist << i18n("standard") << i18n("non-standard");
	form->insertStringList(flist);
	form->setCurrentItem(config->readNumEntry("Form",0));

	hb = new QHBox(tab1);
	QLabel *typelabel = new QLabel(i18n("Wavelet Type : "),hb);
	typecb = new KComboBox(hb);
	int i=0;
	while(wavelet_typeitems[i] != 0) typecb->insertItem(i18n(wavelet_typeitems[i++]));
	typecb->setCurrentItem(config->readNumEntry("Type",0));

	hb = new QHBox(tab1);
	QLabel *klabel = new QLabel(i18n(" k = "),hb);
	kni = new KIntNumInput(config->readNumEntry("K",4),hb);
	kni->setRange(0,INF,1,false);

	Style *style=0;
	Symbol *symbol=0;
	QVBox *styletab;
	if(p && p->getPlot(p->API())->Type() == PSURFACE) {
		styletab = surfaceStyle(tw,true);
		kni->hide();
		klabel->hide();
		typecb->hide();
		typelabel->hide();
	}
	else {
		form->hide();
		styletab = simpleStyle(tw, style, symbol);
	}

	tw->addTab(tab1,i18n("Parameter"));
	tw->addTab(styletab,i18n("Style"));

	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
	QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));

	setMinimumWidth(vbox->minimumSizeHint().width());
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void WaveletListDialog::saveSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "Wavelet" );

	config->writeEntry("Transform", transform->currentItem());
	config->writeEntry("Form", form->currentItem());
	config->writeEntry("Type", typecb->currentItem());
	config->writeEntry("K", kni->value());
}

int WaveletListDialog::apply_clicked() {
	kdDebug()<<"WavletListDialog::apply_clicked()"<<endl;
#ifdef HAVE_GSL16

	// get Graph object
	Graph2D *g=0;
	Graph3D *g3=0;
	Graph4D *g4=0;
	GraphM *gm=0;
	GraphIMAGE *gi=0;
	if(s) {
		// choose how to read data from spreadsheet
		QStringList lst;
		lst << i18n("2D") << i18n("Matrix");
		bool ok;
		QString res = QInputDialog::getItem(i18n("Wavelet transform"), i18n("Read spredsheet as"), lst, 1, true, &ok,this );
		if ( ok ) {
			if(res==lst[0])
				g=s->getGraph2D();
			else if(res == lst[1])
				gm=s->getGraphM();
		}
		else
			return 0;
	}
	else if(p) {
		GraphList *gl = p->getPlot(p->API())->getGraphList();
		int item = (int) (lv->itemPos(lv->currentItem())/lv->currentItem()->height());
		GRAPHType st = gl->getType(item);

		if (st == GRAPH2D)
			g = gl->getGraph2D(item);
		else if (st==GRAPH3D)
			g3 = gl->getGraph3D(item);
		else if (st==GRAPH4D)
			g4 = gl->getGraph4D(item);
		else if (st==GRAPHM)
			gm = gl->getGraphM(item);
		else if (st==GRAPHIMAGE)
			gi = gl->getGraphIMAGE(item);
	}

	// direction
	gsl_wavelet_direction dir;
#ifdef HAVE_GSL17
	if(transform->currentItem()==0)
		dir = gsl_wavelet_forward;
	else
		dir = gsl_wavelet_backward;
#else
	if(transform->currentItem()==0)
		dir = forward;
	else
		dir = backward;
#endif

	// wavelets
	const gsl_wavelet_type *wtype=0;
	switch(typecb->currentItem()) {
	case DAUBECHIES: wtype = gsl_wavelet_daubechies; break;
	case DAUBECHIES_CENTERED: wtype = gsl_wavelet_daubechies_centered; break;
	case HAAR: wtype = gsl_wavelet_haar; break;
	case HAAR_CENTERED: wtype = gsl_wavelet_haar_centered; break;
	case BSLINE: wtype = gsl_wavelet_bspline; break;
	case BSLINE_CENTERED: wtype = gsl_wavelet_bspline_centered; break;
	}

	// check k values
	int k=kni->value();
	bool k_ok=true;
	switch(typecb->currentItem()) {
	case DAUBECHIES:
	case DAUBECHIES_CENTERED:
		if (k/2 <2 || k/2 >10 )
			k_ok = false;
		break;
	case HAAR:
	case HAAR_CENTERED:
		if(k != 2)
			k_ok = false;
		break;
	case BSLINE:
	case BSLINE_CENTERED:
		if(k!=103 && k!=105 && k!=202 && k!=204 && k!=206 && k!=208 && k!=301 && k!=303 &&
			k!=305 && k!=307 && k!=309)
			k_ok = false;
		break;
	}
	if(!k_ok) {
		KMessageBox::warningContinueCancel(this,
			i18n("Sorry. k value not supported!\n Daubechies : k=4,6,8,10,12,14,16,18,20\nHaar : k=2\nBspline : k=103,105,202,204,206,208,301,303,305,307,309"));
		return -1;
	}

	gsl_wavelet *w = gsl_wavelet_alloc(wtype,k);

	if(g!=0 || g3 != 0 || g4 != 0) {		// 1-dim
		kdDebug()<<"	1 dim transformation"<<endl;
		int n, newn=1;
		Point *data;
		Point3D *data3;
		Point4D *data4;

		if(g)	 {
		 	n = g->Number();
			data = g->Data();
		}
		else if(g3) {
		 	n = g3->Number();
			data3 = g3->Data();
		}
		else if(g4) {
		 	n = g4->Number();
			data4 = g4->Data();
		}

		// power of 2
		while (newn<n)
			newn *=2;
		if(newn>n) {
			KMessageBox::warningContinueCancel(this,i18n("Number of points not power of 2! Filling with zeros."));
			kdDebug()<<"Setting n to "<<newn<<endl;
		}
		double *tmp = new double[newn];
		for(int i=0;i<newn;i++) {
			if(i>=n)
				tmp[i]=0.0;
			else {
				if(g)
					tmp[i]=data[i].Y();
				else if (g3)
					tmp[i]=data3[i].Y();
				else if (g4)
					tmp[i]=data4[i].Y();
			}
		}

		gsl_wavelet_workspace *work = gsl_wavelet_workspace_alloc(newn);

		gsl_wavelet_transform(w,tmp,1,newn,dir,work);

		Point *newdata = new Point[newn];
		for (int i = 0;i<newn;i++) {
			double x;
			if(i>=n)
				x=(double)i;
			else {
				if(g)
					x=data[i].X();
				else if (g3)
					x=data3[i].X();
				else if (g4)
					x=data4[i].X();
			}
			double y = tmp[i];

			newdata[i].setPoint(x,y);
		}

		double xmin=0, xmax=1, ymin=0, ymax=1;
		mw->calculateRanges2D(newdata,newn,&xmin,&xmax,&ymin,&ymax);

		// create the new Graph
		LRange range[2];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);

		QString fun = QString(i18n("wavelet transform of ")+g->getLabel()->simpleTitle());

		Style *style = 0;
		Symbol *symbol = 0;
		if(cb2 !=0) {
			style = new Style((StylesType)cb2->currentItem(),color->color(),filled->isChecked(),fcolor->color(),
				width->value(),pencb->currentItem(),brushcb->currentItem());
			style->setBoxWidth(boxwidth->value());
			style->setAutoBoxWidth(autobox->isChecked());
			style->setPointsSorting(sortpointscb->isChecked());
			symbol = new Symbol((SType)symbolcb->currentItem(),scolor->color(),ssize->value(),
				(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
		}

		Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,newdata,newn);
		int item=-1;
		if(p)
			item=sheetcb->currentItem();
		else
			item = s->Destination();
		mw->addGraph2D(ng,item,P2D);
	}
	else if(gm!=0) {	//2-dim
		//TODO : gi (IMAGE)
		kdDebug()<<"	2 dim transformation"<<endl;
		int nx=gm->NX(), ny=gm->NY(), newn=1;
		if(nx!=ny) {
			KMessageBox::warningContinueCancel(this,i18n("Sorry. Matrix must be square!"));
			return -3;
		}
		int n=nx*ny;

		// power of 2
		while (newn<nx)
			newn *=2;
		if(newn>nx) {
			KMessageBox::warningContinueCancel(this, i18n("Number of points not power of 2! Giving up."));
			return -4;
		}

		double *data = gm->Data();
		double *newdata = new double[n];

		for(int i=0;i<nx;i++)
			for(int j=0;j<ny;j++)
				newdata[j+nx*i]=data[j+nx*i];

		gsl_wavelet_workspace *work = gsl_wavelet_workspace_alloc(n);

		if(form->currentItem()==0)
			gsl_wavelet2d_transform(w,newdata,nx,nx,nx,dir,work);
		else
			gsl_wavelet2d_nstransform(w,newdata,nx,nx,nx,dir,work);

		double zmin=0, zmax=1;
		for (int i = 0;i<nx;i++) {
			for (int j = 0;j<ny;j++) {
				double z = newdata[j+nx*i];

				if (i==0) {
					zmin=zmax=z;
				}
				else {
					z<zmin?zmin=z:0;
					z>zmax?zmax=z:0;
				}
			}
		}

		// create the new Graph
		LRange range[3];
		range[0] = LRange(1,nx);
		range[1] = LRange(1,ny);
		range[2] = LRange(zmin,zmax);

		QString fun = QString(i18n("2d wavelet transform of ")+gm->getLabel()->simpleTitle());

		Style *style = new Style(LINESTYLE);
		Symbol *symbol = new Symbol(SNONE);

		GraphM *ng = new GraphM(fun,fun,range,SSPREADSHEET,type,style,symbol,newdata,nx,ny);
		int item=-1;
		if(p)
			item = sheetcb->currentItem();
		else
			item = s->Destination();
		//TODO : type ??
		mw->addGraphM(ng,item);
	}

	if(p) updateList();
#endif

	return 0;
}
