//LabPlot : ImportDialog.cc

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <iostream>
#include <qlabel.h>
#include <qhbox.h>
#include <qfiledialog.h>
#include <qprogressdialog.h>
#include <qwizard.h>

#include <klocale.h>
#include <kdebug.h>
#include <kfilterdev.h>
#include <kmessagebox.h>
#include "ImportDialog.h"
#include "LTableItem.h"
#include "FilterCDF.h"
#include "FilterNETCDF.h"
#include "FilterAUDIOFILE.h"
#include "FilterMAGICK.h"
#include "FilterHDF5.h"
#include "defs.h"
#include "import.h"
#include "inputfilter.h"

using namespace std;

/* Dialog for importing data into spreadsheet */
ImportDialog::ImportDialog(MainWin *m, QString filename, InputFilter filter, const char *name)
	: Dialog(m, name)
{
	kdDebug()<<"ImportDialog::ImportDialog()"<<endl;
	setCaption(i18n("Import Data"));

	KConfig *config = mw->Config();
	config->setGroup( "Import" );

	QVBox *vb = new QVBox(vbox);
	if(filename.isEmpty())
		filename = config->readEntry("Filename","1.dat");
	importWidget(vb, filename, filter);

	QHBox *hb = new QHBox(vb);
	newspread = new QCheckBox(i18n("create new spreadsheet"),hb);
	newspread->setChecked(config->readBoolEntry("CreateNewSpreadsheet",true));
	namespread = new QCheckBox(i18n("use filename for spreadsheet title"),hb);
	namespread->setChecked(config->readBoolEntry("UseFilenameAsTitle",false));

	QObject::connect(ok,SIGNAL(clicked()),SLOT(Apply()));
	QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));
	QObject::connect(cancel,SIGNAL(clicked()),SLOT(accept()));

	// gbox > vbox here !
	setMinimumWidth((int)(1.5*vbox->minimumSizeHint().width()));
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void ImportDialog::saveSettings() {
	saveImportSettings();

	KConfig *config = mw->Config();
	config->setGroup( "Import" );
	config->writeEntry("CreateNewSpreadsheet",newspread->isChecked());
	config->writeEntry("UseFilenameAsTitle",namespread->isChecked());

}

void ImportDialog::wizardKexiDB(const QString &name) {
	kdDebug()<<"ImportDialog::wizardKexiDB()"<<endl;

	int res;
	if(name == i18n("Driver")) {
		kdDebug()<<"	TAB Driver"<<endl;
	}
	else if(name == i18n("Connection")) {
		kdDebug()<<"	TAB Connection : connecting to "<<driver->currentText()<<endl;
		connectionlabel->setText(i18n("Connecting to Driver ")+driver->currentText()+i18n(" ... "));
		res = kexi->connectDriver(driver->currentText());
		if (res) {
			KMessageBox::error(this, i18n("Sorry. Could not connect to driver!"));
			return;
		}
	}
	else if(name == i18n("Database")) {
		kdDebug()<<"	TAB Database"<<endl;
		res = kexi->connect(host->text(),user->text(),password->text());
		if (res) {
			KMessageBox::error(this, i18n("Sorry. Could not connect to database!"));
			return;
		}
		databases->clear();
		kdDebug()<<"	calling Databases()"<<endl;
		if((kexi->Databases()).empty()) {
			KMessageBox::error(this, i18n("Sorry. No databases found!"));
			return;
		}
		databases->insertStringList(kexi->Databases());
	}
	else if(name == i18n("Table")) {
		kdDebug()<<"	TAB Database"<<endl;
		tablelabel->setText(i18n("Using Database ")+databases->currentText()+i18n(" ... "));
		res = kexi->connectDatabase(databases->currentText());
		if (res) {
			KMessageBox::error(this, i18n("Sorry. Could not connect to database!"));
			return;
		}
		tables->clear();
		tables->insertStringList(kexi->Tables());
	}

	kdDebug()<<"ImportDialog::wizardKexiDB() DONE"<<endl;
}

void ImportDialog::finishKexiDB() {
	QString tablename = tables->currentText();

	QTable *table = mw->activeSpreadsheet()->Table();

	int res = kexi->initialize(tablename);
	if(res) {
		KMessageBox::error(this, i18n("Sorry. Could not read table from database!"));
		return;
	}

	int cols = kexi->Fields(), rows = kexi->Rows();
	table->setNumCols(cols);
	table->setNumRows(rows);

	QString *data = kexi->Data();
	for (int i=0;i<rows;i++) {
		for (int j=0;j<cols;j++) {
			table->setText(i,j,data[cols*i+j]);
		}
	}

	//set column names
	for (int i=0;i<table->numCols();i++) {
		QString fieldname = kexi->fieldName(i);
		if(i==0)
			table->horizontalHeader()->setLabel( i, fieldname+' '+i18n("{double}")+" [X]");
		else
			table->horizontalHeader()->setLabel( i, fieldname+' '+i18n("{double}")+" [Y]");
	}
	table->setUpdatesEnabled(true);
	table->repaintContents();
}

//! import HP41XX data files
int ImportDialog::importHP41XX(QIODevice *file) {
	kdDebug()<<"Reading HP41XX file"<<endl;
	QTextStream t(file);
	QString line=0;

	kdDebug()<<" Title : "<<t.readLine()<<endl;
	int sets=0, lines[255];
	do {
		QString tmp;
		double min,max,step;
		t>>tmp>>min>>tmp>>max>>tmp>>step;
		t.readLine();
		line = t.readLine();
		lines[sets] = (int) (rint((max-min)/step)+1);
		kdDebug()<<"RANGE : "<<min<<'/'<<max<<" in steps : "<<step<<"	-> nr="<<lines[sets]<<endl;
		sets++;
	} while (line.contains(" Append "));
	kdDebug()<<"found "<<sets<<" sets"<<endl;

	double x,y,z;
	int linenr;
	for(int i=0;i<sets;i++) {
		Spreadsheet *ss=0;
		QTable *table=0;
		if(i>0) {
			t.readLine();	// /* Append X */
			line = t.readLine();	// No. VF IF L
			ss = mw->newSpreadsheet();
			table = ss->Table();
		}
		if(table == 0) return -1;

		table->setNumRows(lines[i]);
		table->setNumCols(3);
					// column names
		QStringList names = QStringList::split(' ', line);
		ss->setColumnTitle(0,names[1]);
		ss->setColumnTitle(1,names[2]);
		table->horizontalHeader()->setLabel( 2, names[3]+' '+i18n("{double}")+" [Y]");

		line=t.readLine();
		kdDebug()<<" Set "<<i+1<<" : "<<line<<endl;
		int j=0;
		while(!line.isEmpty()) {
			if(j<lines[i]) {
				t>>linenr>>x>>y>>z;
				t.readLine();		// until new line
							//kdDebug()<<" DATA  : "<<linenr<<" / "<<x<<' '<<y<<' '<<z<<endl;
				table->setText(j,0,QString::number(x));
				table->setText(j,1,QString::number(y));
				table->setText(j,2,QString::number(z));
				j++;
			}
			else
				line=t.readLine();	// empty line
		}

		table->setUpdatesEnabled(true);
		table->repaintContents();	// resizing remains
	}
	return 0;
}

//! import HDF5 data files
int ImportDialog::importHDF5(QString filename, Spreadsheet *ss, int startRow, int endRow) {
#ifdef HAVE_HDF5
	FilterHDF5 hdf5 = FilterHDF5(filename);
	if(!hdf5.fileOK()) return -1;
	hdf5.importFile();

	kdDebug()<<"Reading HDF5 data"<<endl;
	kdDebug()<<"number of attributes = "<<hdf5.numAttributes()<<endl;
	kdDebug()<<"number of datasets = "<<hdf5.numSets()<<endl;

	// add attributes as project comments
	Project*project;
	QString notes;
	if(hdf5.numAttributes()>0) {
		project = mw->getProject();
		notes = project->Notes();
	}
	for (int i=0;i<hdf5.numAttributes();i++) {
		notes.append(hdf5.getAttribute(i)+"\n");
		// kdDebug()<<"	ATTRIBUTE "<<i+1<<" : "<<hdf5.getAttribute(i)<<endl;
	}
	if(hdf5.numAttributes()>0)
		project->setNotes(notes);

	// read data
	headercb->setChecked(true);		// avoid resetting column names
	for (int i=0;i<hdf5.numSets();i++) {
		if(i>0) ss = mw->newSpreadsheet();
		// ss = mw->newSpreadsheet();
		QTable *table = ss->Table();

		int rows = hdf5.Rows(i);
		int cols = hdf5.Cols(i);
		if(rows==0) cols=0;
		kdDebug()<<"Dataset "<<i+1<<" ("<<hdf5.datasetName(i)<<") has "<< rows<<" rows and "<<cols<<" cols"<<endl;
		ss->setTitle(hdf5.datasetName(i));

		// set spreadsheet notes
		QString setnotes;
		int nrsetattr = hdf5.numSetAttributes(i);
		if(nrsetattr>0) {
			setnotes = ss->Notes();
			for(int j=0;j<nrsetattr;j++)
				setnotes.append(hdf5.getSetAttribute(i,j)+"\n");
			ss->setNotes(setnotes);
		}

		table->setNumRows(rows);
		table->setNumCols(cols);
		for (int k=0;k<cols;k++) {
			for (int j=0;j<rows;j++) {
				LTableItem *item = new LTableItem( table, QTableItem::OnTyping,QString::number(hdf5.Data(i,j,k),'g',15));
				table->setItem(j, k, item);
			}

			QString colname = hdf5.columnName(i,k);
			// TODO : column datatype
			if(colname.length()<1)
				colname = QChar(k+65);
			if(k==0)
				table->horizontalHeader()->setLabel(k, colname + " " + i18n("{double}") + " [X]");
			else
				table->horizontalHeader()->setLabel(k, colname + " " + i18n("{double}") + " [Y]");
		}
	}
	return 0;
#else
	kdDebug()<<"hdf5 file format not supported!"<<endl;
#endif
}

//! import netCDF data files
int ImportDialog::importNETCDF(QString filename, Spreadsheet *ss, int startRow, int endRow) {
#ifdef HAVE_NETCDF
	FilterNETCDF ncf = FilterNETCDF(filename);
	if(!ncf.fileOK()) return -1;
	kdDebug()<<"Reading NETCDF data"<<endl;
	QTable *table = ss->Table();
	table->setNumCols(ncf.NVars());

	//QProgressDialog progress( i18n("Reading data ..."), i18n("Cancel"), ncf.VarLen(xname), this, "progress", true );
	kdDebug()<<" nvars = "<<ncf.NVars()<<endl;
	for (int j=startRow;j<ncf.NVars();j++) {
		QString name = ncf.VarName(j);
		int rows = ncf.VarLen(name);
		kdDebug()<<" var / varid / len = "<<name<<' '<<j<<' '<<rows<<endl;

		// TODO : BUG : set column label to name
		ss->setColumnTitle( j, name );

		if (table->numRows()<rows)
			table->setNumRows(rows);
		for (int i=0;i<rows;i++) {
			//if(i%1000 == 0) progress.setProgress(i);
			table->setText(i,j,QString::number(ncf.Data(name,i)));
		}
		if (j>endRow)
			break;
	}
	return 0;
#else
	kdDebug()<<"Format not supported!"<<endl;
	return 1;
#endif
}

//! import CDF data files
int ImportDialog::importCDF(QString filename, Spreadsheet *ss, int startRow, int endRow) {
#ifdef HAVE_CDF
	FilterCDF cdf = FilterCDF(filename);
	if(!cdf.fileOK()) return -1;
	kdDebug()<<"Reading CDF data"<<endl;

	QTable *table = ss->Table();
	table->setNumCols(cdf.NVars());
	table->setNumRows(cdf.MaxRec());

	QProgressDialog progress( i18n("Reading data ..."), i18n("Cancel"), cdf.MaxRec(), this, "progress", true );
	kdDebug()<<" nvars = "<<cdf.NVars()<<endl;
	for (int j=startRow;j<cdf.NVars();j++) {
		QString name = cdf.VarName(j);
		int rows = cdf.VarLen(name);
		kdDebug()<<" var / varid / len = "<<name<<' '<<j<<' '<<rows<<endl;

		// TODO : BUG : set column label to name
		ss->setColumnTitle( j, name );

		for (int i=0;i<rows;i++) {
			if(i%1000 == 0) progress.setProgress(i);
			table->setText(i,j,QString::number(cdf.Data(name,i)));
		}
		if (j>endRow)
			break;
	}
	return 0;
#else
	kdDebug()<<"Format not supported!"<<endl;
	return 1;
#endif
}

//! import audio files
int ImportDialog::importAUDIOFILE(QString filename, Spreadsheet *ss, int startRow, int endRow) {
	kdDebug()<<"importAUDIOFILE()"<<endl;
	FilterAUDIOFILE auf = FilterAUDIOFILE(filename);
	if(!auf.fileOK()) return -1;
	kdDebug()<<"reading Audio DATA"<<endl;

	double *data = auf.Data();
	QTable *table = ss->Table();
	table->setNumRows(auf.frameCount());
	table->setNumCols(auf.channelCount()+1);

	QProgressDialog progress( i18n("Reading data ..."), i18n("Cancel"), auf.frameCount(),
				  this, "progress", true );
	for (int i=startRow;i<auf.frameCount();i++) {
		if(i%1000 == 0) progress.setProgress(i);
		qApp->processEvents();
		table->setText(i,0,QString::number(i/auf.sampleRate()));
		if(auf.channelCount() == 1)
			table->setText(i,1,QString::number(data[i]));
		else if (auf.channelCount() == 2) {
			table->setText(i,1,QString::number(data[2*i]));
			table->setText(i,2,QString::number(data[2*i+1]));
		}
		if (i>endRow)
			break;
	}
	return 0;
}

//! import binary files
int ImportDialog::importBINARY(QIODevice *file, QTable *table, int startRow, int endRow) {
	kdDebug()<<"reading BINARY DATA"<<endl;
	QDataStream ds(file);
	ds.setByteOrder(byteordercb->currentItem());

	// offset
	int nvars = varle->text().toInt();
	for (int j=0;j<nvars*startRow;j++)
		getBinaryValue(&ds,(BinaryType) binarytypecb->currentItem());
	int row = startRow;

	if(nvars > table->numCols())
		table->setNumCols(nvars);

	QProgressDialog progress( i18n("Reading binary data ..."), i18n("Cancel"), file->size(),this, "progress", true );
	while(!ds.eof()) {
		if (row%1000==0) {
			progress.setProgress(file->at());
			table->setNumRows(row+1000);
		}
		qApp->processEvents();

		for (int j=0;j<nvars;j++) {
			LTableItem *item = new LTableItem(table, QTableItem::OnTyping,
				QString::number(getBinaryValue(&ds,(BinaryType) binarytypecb->currentItem())));
			table->setItem( row-startRow,j, item);
		}

		row++;
		if ( progress.wasCancelled() ) {
			table->setUpdatesEnabled(true);
			return 1;
		}
		if (row>endRow)
			break;
	}

	return row;
}

//! import ASCII files
int ImportDialog::importASCII(QIODevice *file, QTable *table, int startRow, int endRow, int column, QStringList::iterator it) {
	QTextStream ts(file);
	kdDebug()<<"	reading ASCII DATA"<<endl;
	QProgressDialog progress( i18n("Reading ASCII data ..."), i18n("Cancel"), file->size(), this, "progress", true );

	int header=0;
	if (headercb->isChecked()) header=1;

	int row=0;
	while (!ts.eof()) {
		//	kdDebug()<<"	reading row "<<row<<endl;
		if (row%1000==0) {
			progress.setProgress(file->at());
			table->setNumRows(row+1000);
		}
		qApp->processEvents();

		while(row<startRow) {
			ts.readLine();
			row++;
		}

		QString line = ts.readLine();

		if(simplifycb->isChecked())
			line = line.simplifyWhiteSpace();
		// ignore comment lines
		if(line.startsWith(commcb->currentText()) == true)
			continue;

		QString sep = sccb->currentText();
		QStringList oneline = splitLine(line,sep,emptycb->isChecked());

		// handle empty lines correct
		if(oneline.count() == 0)
			continue;

		if(oneline.count()+column > (unsigned int) table->numCols())
			table->setNumCols(oneline.count()+column);

		int j=0;
		if(row==startRow && headercb->isChecked()) {
			for ( QStringList::Iterator lit = oneline.begin();lit != oneline.end(); ++lit ) {
				if (j==0)
					table->horizontalHeader()->setLabel(
							column+j, *lit+' '+i18n("{double}")+" [X]");
				else
					table->horizontalHeader()->setLabel(
							column+j, *lit+' '+i18n("{double}")+" [Y]");
				j++;
			}
		}
		else {
			for ( QStringList::Iterator lit = oneline.begin(); lit != oneline.end(); ++lit ) {
				// skip first (X) column if not first file
				QStringList fns = QStringList::split(";",filele->text());
				if(samexcb->isChecked() && it != fns.begin() && lit == oneline.begin() ) {
					j++;
					continue;
				}

				// needed for alignment
				LTableItem *item = new LTableItem(table, QTableItem::OnTyping,(*lit));
				table->setItem( row-startRow-header,j+column, item);
				j++;
			}
		}

		row++;
		if ( progress.wasCancelled() ) {
			table->setUpdatesEnabled(true);
			table->setNumRows(row-startRow-header);
			// set label for all columns (see below)
			if(!headercb->isChecked()) {
				for (int i=1;i<table->numCols();i++) {
					if(i<26)
						table->horizontalHeader()->setLabel( i,
					QChar(i+65)+' '+i18n("{double}")+" [Y]");
					else
						table->horizontalHeader()->setLabel( i,
					QString(QChar(65+i/26-1)) +
							QString(QChar(65+i%26)) + ' ' +
							i18n("{double}")+" [Y]");
				}
			}
			return 1;
		}
		if (row>endRow)
			break;
	}

	return row;
}

//! import data from databases using KEXI
int ImportDialog::importKEXIDB() {
#ifdef HAVE_KEXIDB
	kexi = new FilterKexiDB(0);

	QWizard *wizard = new QWizard(this);
	QObject::connect(wizard,SIGNAL(selected(const QString &)),this,SLOT(wizardKexiDB(const QString&)));
	QObject::connect(wizard->finishButton(),SIGNAL(clicked()),this,SLOT(finishKexiDB()));

	QVBox *page1 = new QVBox(wizard);
	new QLabel(i18n("Please select driver : "),page1);
	driver= new KComboBox(page1);
	driver->insertStringList(kexi->Driver());

	wizard->addPage(page1,i18n("Driver"));
	wizard->setHelpEnabled(page1,false);

	QVBox *page2 = new QVBox(wizard);

	connectionlabel = new QLabel("",page2);
	new QLabel(i18n("Please provide connection informations  : "),page2);
	QHBox *hb = new QHBox(page2);
	new QLabel(i18n("Hostname : "),hb);
	host = new KLineEdit(i18n("localhost"),hb);
	hb = new QHBox(page2);
	new QLabel(i18n("User : "),hb);
	user = new KLineEdit(i18n("root"),hb);
	hb = new QHBox(page2);
	new QLabel(i18n("Password : "),hb);
	password = new KLineEdit(i18n("mysql"),hb);
	password->setEchoMode(QLineEdit::Password);

	wizard->addPage(page2,i18n("Connection"));
	wizard->setHelpEnabled(page2,false);

	QVBox *page3 = new QVBox(wizard);
	new QLabel(i18n("Please select database  : "),page3);
	databases = new KComboBox(page3);
	wizard->addPage(page3,i18n("Database"));
	wizard->setHelpEnabled(page3,false);

	QVBox *page4 = new QVBox(wizard);
	tablelabel = new QLabel("",page4);
	new QLabel(i18n("Please select table : "),page4);
	tables = new KComboBox(page4);
	wizard->addPage(page4,i18n("Table"));
	wizard->setHelpEnabled(page4,false);
	wizard->setFinishEnabled(page4,true);

	wizard->show();
	return 0;
#else
	kdDebug()<<"Format not supported!"<<endl;
	return 1;
#endif
}

//! APPLY : read data from file
int ImportDialog::apply_clicked() {
	kdDebug()<<"ImportDialog::apply_clicked()"<<endl;
	int column=0;	// used for "same x column"

	QStringList fns = QStringList::split(";",filele->text());
	for ( QStringList::Iterator it = fns.begin(); it != fns.end(); ++it ) {
		QString filename = *it;
		kdDebug()<<"Opening "<<filename<<endl;

		Spreadsheet *ss=0;
		if(!newspread->isChecked() && it == fns.begin())
			ss = mw->activeSpreadsheet();

		if(samexcb->isChecked() && it != fns.begin()) {
			ss = mw->activeSpreadsheet();
			column++;
		}

		if(!ss) {
//			kdDebug()<<"No active spreadsheet. Adding a new one."<<endl;
			ss = mw->newSpreadsheet();
		}

		// set title of ss to filename if selected
		if(namespread->isChecked())
			ss->setTitle(filename);

		QTable *table = ss->Table();
		table->setUpdatesEnabled(false);	// for fast processing

		// individual filters
		if(filtercb->currentItem() == FKEXIDB) {
			importKEXIDB();
			return 0;
		}

		if (!filename.isEmpty()) {
			QIODevice *file = KFilterDev::deviceForFile(filename,QString::null,true);
			if(file==0) file = new QFile(filename);

			// HP41XX filter
			if(filtercb->currentItem() == FHP41XX) {
				if ( ! file->open( IO_ReadOnly )) {
					KMessageBox::error(this, i18n("Sorry. Could not open specified data file for reading!"));
					return 2;
				}
				return importHP41XX(file);
			}

			// reading image (dont handle binary data with ImageMagick)
			QString format;

			if ((filtercb->currentItem() == FUSER) && (format = QString(QImageIO::imageFormat(filename))) != 0) {
				QPixmap pm;
#ifdef HAVE_MAGICK
				FilterMAGICK mf = FilterMAGICK(filename);
				if(!mf.fileOK()) break;
				pm = mf.Pixmap();
#endif
				if(pm.isNull()) // loading error
					pm.load(filename);

				kdDebug()<<"reading IMAGE DATA format = "<<format<<endl;

				QImage image = pm.convertToImage();
				int w = image.width();
				int h = image.height();
				kdDebug() <<" Width/Height : "<<w<<' '<<h<<endl;
				kdDebug() <<" Depth : "<<image.depth()<<endl;

				table->setNumCols(w);
				table->setNumRows(h);

				for (int i=0;i<w;i++) {
					for (int j=0;j<h;j++) {
						table->setText(j,i,QString::number(qGray(image.pixel(i,j))));
					}
				}
				table->horizontalHeader()->setLabel( 0, QString( "A ")+i18n("{double}")+" [Y]");
			}
			else if ( file->open( IO_ReadOnly )) {		// reading normal data (not image)
				int startRow = startle->text().toInt()-1;
				int endRow;
				if(endle->text() == i18n("END"))
					endRow = INF;
				else
					endRow = endle->text().toInt()-1;
				kdDebug()<<"START/END ROW = "<<startRow<<" .. "<<endRow<<endl;

				if (filename.lower().endsWith(".h5") || filename.lower().endsWith(".hdf")) {	// hdf5 file
					importHDF5(filename,ss,startRow,endRow);
				}
				else if (filename.lower().endsWith(".nc")) {				// netcdf file
					importNETCDF(filename,ss,startRow,endRow);
				}
				else if (filename.lower().endsWith(".cdf")) {				// cdf file
					importCDF(filename,ss,startRow,endRow);
				}
				else {
					// check for audio file
					if ( !( filename.lower().contains(".dat") || filename.lower().contains(".txt") ||
						filename.lower().contains(".bin") || filename.lower().contains(".csv")) ) {	// audiofile file
						importAUDIOFILE(filename, ss, startRow, endRow);
						break;
					}

					QProgressDialog progress( i18n("Reading data ..."), i18n("Cancel"), file->size(),this,
						"progress", true );
					table->setNumCols(1+column);	// set initial number of cols

					int row=0;
					if(filtercb->currentItem() == FBINARY)
						row = importBINARY(file,table,startRow,endRow);
					else 	// ASCII data
						row = importASCII(file,table,startRow,endRow,column,it);

					if(headercb->isChecked())
						row--;	//header
					table->setNumRows(row-startRow);
				}
				file->close();
			}
			else {
				KMessageBox::error(this, i18n("Sorry. Could not open specified data file for reading!"));
				return 2;
			}

			// set label for all columns
			if(!headercb->isChecked()) {
				for (int i=1;i<table->numCols();i++) {
					if(i<26)
						table->horizontalHeader()->setLabel( i, QChar(i+65)+' '+i18n("{double}")+" [Y]");
					else
						table->horizontalHeader()->setLabel( i, QString(QChar(65+i/26-1)) +
							QString(QChar(65+i%26)) + ' ' + i18n("{double}")+" [Y]");
				}
			}
		}

		table->setUpdatesEnabled(true);
		table->repaintContents();	// resizing remains
	}

	return 0;
}
