//LabPlot : Dialog.cc

#include <iostream>
#include <qlabel.h>
#include <qfiledialog.h>
#include <qprocess.h>

#include <klocale.h>
#include <kdebug.h>
#include <kfilterdev.h>
#include <kfilterbase.h>
#include <kconfig.h>
#include <kmessagebox.h>

#include "Dialog.h"
#include "Plot2DSurface.h"
#include "PlotQWT3D.h"
#include "FileInfoDialog.h"
#include "ColormapReader.h"
#include "importitems.h"
#include "filteritems.h"
#include "defs.h"

using namespace std;

Dialog::Dialog(MainWin *mw, const char *name)
	: KDialog((QWidget *)mw, name), mw(mw), p(mw->activeWorksheet()), s(mw->activeSpreadsheet())
{
	vbox = new QGroupBox(1,QGroupBox::Horizontal,QString(""),this);
	gbox = new QGroupBox(1, QGroupBox::Vertical, QString(""), this);

	ok = new KPushButton(i18n("OK"),gbox);
	apply = new KPushButton(i18n("Apply"),gbox);
	save = new KPushButton(i18n("Save"),gbox);
	cancel = new KPushButton(i18n("Cancel"),gbox);
	QObject::connect(cancel,SIGNAL(clicked()),SLOT(reject()));
}

void Dialog::resizeEvent(QResizeEvent *e) {
//	kdDebug()<<"Dialog::resizeEvent()\n";

	QSize size = e->size();
	gbox->setGeometry(0,size.height()-50,size.width(),50);
	vbox->setGeometry(0,0,size.width(),size.height()-50);
}

void Dialog::selectColormap() {
	ColorMapPreview *colormap = new ColorMapPreview;
	// modal. otherwise problems occur in configure dialog
	QFileDialog *datacolor = new QFileDialog(this,0,true);

	QString cmdir = locate("data","LabPlot/colormaps/");
	datacolor->setDir(cmdir);
	datacolor->setFilter("Colormap files (*.map;*.MAP)");
	datacolor->setContentsPreviewEnabled( true );
	datacolor->setContentsPreview( colormap, colormap);
	datacolor->setPreviewMode( QFileDialog::Contents );

	connect(datacolor, SIGNAL(fileHighlighted(const QString&)), this, SLOT(adaptDataColors(const QString&)));
	datacolor->show();
}

void Dialog::adaptDataColors(const QString& fn) {
#ifdef HAVE_GL
	if (!openColorMap(fn))
		return;

	KConfig *config = mw->Config();
	config->setGroup( "Plot Surface Style" );
	config->writeEntry("Colorscale",fn);

	if(p == 0)
		return;
	Plot *plot = p->getPlot(p->API());
	if(plot == 0)
		return;
	switch(plot->Type()) {
	case PQWT3D: {
		Qwt3D::StandardColor *col = new Qwt3D::StandardColor((PlotQWT3D *)plot);
		col->setColorVector(cv);

		((PlotQWT3D *) plot)->setColorVector(cv);
		}; break;
	case PSURFACE:
		((Plot2DSurface *) plot)->setColorVector(cv);
		break;
	default:
		kdDebug()<<" ERROR setting colormap"<<endl;
		break;
	}
#endif
}

bool Dialog::openColorMap(QString fn) {
#ifdef HAVE_GL
	kdDebug()<<"openColorMap("<<fn<<")"<<endl;
	ifstream file((const char*)fn.local8Bit());

	if (!file) return false;

	Qwt3D::RGBA rgb;
	cv.clear();

	while ( file ) {
		file >> rgb.r >> rgb.g >> rgb.b;
		file.ignore(1000,'\n');
		if (!file.good())
			break;
		else {
			rgb.a = 1;
			rgb.r /= 255;
			rgb.g /= 255;
			rgb.b /= 255;
			cv.push_back(rgb);
		}
	}

	return true;
#else
	return false;
#endif
}

//! used from dialogs for surface plots
QVBox* Dialog::surfaceStyle(QTabWidget *tw, bool fresh) {
	kdDebug()<<"Dialog::surfaceStyle(): fresh="<<fresh<<endl;
	QVBox *styletab = new QVBox(tw);
	KConfig *config = mw->Config();
	config->setGroup( "Plot Surface Style" );

	//TODO : bug?
	Plot *tmpplot=0;
	if(p) {
		tmpplot = p->getPlot(p->API());
		kdDebug()<<"	Getting plot "<<p->API()<<" from worksheet"<<endl;
	}
	Plot2DSurface *plot=0;
	if (tmpplot && tmpplot->Type() == PSURFACE)
		plot = (Plot2DSurface *)tmpplot;
	else {
		kdDebug()<<"Creating new Surface Plot"<<endl;
		if(p) plot = new Plot2DSurface(p);
	}
	if(plot) kdDebug()<<"	gl->Number()="<<plot->getGraphList()->Number()<<endl;

	QHBox *hb = new QHBox(styletab);
	dcb = new QCheckBox(i18n("density"),hb);
	dcb->setChecked(fresh?config->readBoolEntry("Density Enabled",true):plot->densityEnabled());
	ccb = new QCheckBox(i18n("contour"),hb);
	ccb->setChecked(fresh?config->readBoolEntry("Contour Enabled",true):plot->contourEnabled());

	hb = new QHBox(styletab);
	new QLabel(i18n("number of level : "),hb);
	numberle = new KLineEdit(QString::number(fresh?config->readNumEntry("Contour Level",10):
		plot->Number()),hb);
	numberle->setValidator(new QIntValidator(numberle));
	numberle->setMaximumWidth(100);
	hb = new QHBox(styletab);
	new QLabel(i18n("contour line color : "),hb);
	contourcolor = new KColorButton(fresh?config->readColorEntry("Contour Color",&Qt::black):
		plot->ContourColor(),hb);
	new QLabel(i18n("contour line width : "),hb);
	contourwidthle = new KLineEdit(QString::number(fresh?config->readNumEntry("Contour Width",1):
		plot->ContourWidth()),hb);

	hb = new QHBox(styletab);
	new QLabel(i18n("Brush : "),hb);
	dbrushcb = new KComboBox(hb);
	fillBrushBox(dbrushcb,SRECT,Qt::blue,FFULL,Qt::red);
	dbrushcb->setCurrentItem(fresh?config->readNumEntry("Density Brush",1):plot->Brush());

	hb = new QHBox(styletab);
	coloredcb = new QCheckBox(i18n("colored contour lines"),hb);
	coloredcb->setChecked(fresh?config->readBoolEntry("Colored Contour",false):plot->ColoredContour());
	meshcb = new QCheckBox(i18n("show mesh"),hb);
	meshcb->setChecked(fresh?config->readBoolEntry("Show Mesh",false):plot->Mesh());
	relativecb = new QCheckBox(i18n("relative color scale"),hb);
	relativecb->setChecked(fresh?config->readBoolEntry("Relative Colorscale",true):plot->Relative());

	hb = new QHBox(styletab);
	new QLabel(i18n(" Threshold : "),hb);
	thresholdle = new KLineEdit(QString::number(fresh?config->readDoubleNumEntry("Threshold",-INF):
		plot->Threshold()),hb);
	thresholdle->setValidator(new QDoubleValidator(thresholdle));
	thresholdle->setMaximumWidth(100);

	hb = new QHBox(styletab);
	new QLabel(i18n(" Color Palette : "),hb);
	//palabel->setAlignment(Qt::AlignHCenter);

	// use QWT colorscale
	KPushButton *colormap = new KPushButton(i18n("Colormap"),hb);
	QObject::connect(colormap,SIGNAL(clicked()),SLOT(selectColormap()));

	return styletab;
}

void Dialog::setStyle(Style *style) {
	cb2->setCurrentItem(style->Type());
	sortpointscb->setChecked(style->PointsSortingEnabled());
	boxwidth->setValue(style->BoxWidth());
	autobox->setChecked(style->AutoBoxWidth());
	color->setColor(style->Color());
	width->setValue(style->Width());
	pencb->setCurrentItem(style->PenStyle());
	filled->setChecked(style->isFilled());
	fcolor->setColor(style->FillColor());
	brushcb->setCurrentItem(style->Brush());
}

void Dialog::setSymbol(Symbol *symbol) {
	symbolcb->setCurrentItem(symbol->Type());
	scolor->setColor(symbol->Color());
	ssize->setValue(symbol->Size());
	symbolfillcb->setCurrentItem(symbol->Fill());
	sfcolor->setColor(symbol->FillColor());
	sbrushcb->setCurrentItem(symbol->Brush());
}

//! used from dialogs for simple plots
QVBox* Dialog::simpleStyle(QTabWidget *tw, Style *style, Symbol *symbol) {
	QVBox *styletab = new QVBox(tw);
	KConfig *config = mw->Config();
	config->setGroup( "Plot Simple Style" );

	QGroupBox *stylegb = new QGroupBox(1,QGroupBox::Horizontal,i18n("Style"),styletab);
	QHBox *hb = new QHBox(stylegb);
	new QLabel(i18n(" Type : "),hb);
	cb2 = new KComboBox(hb);
	QStringList stylelist;
	stylelist<<i18n("Lines")<<i18n("NoCurve")<<i18n("Steps")<<i18n("Boxes")<<i18n("Impulses")<<i18n("Y Boxes");
	cb2->insertStringList(stylelist);
	cb2->setCurrentItem(style==0?config->readNumEntry("Graph Style",0):style->Type());
	sortpointscb = new QCheckBox(i18n("sort points"),hb);
	sortpointscb->setChecked(style==0?config->readBoolEntry("Sort Points",true):style->PointsSortingEnabled());

	hb = new QHBox(stylegb);
	new QLabel(i18n(" Box width : "),hb);
	boxwidth = new KIntNumInput(style==0?config->readNumEntry("Box Width",10):style->BoxWidth(),hb);
	boxwidth->setRange(1,1000,1,false);
	autobox = new QCheckBox(i18n("auto box width"),hb);
	autobox->setChecked(style==0?config->readBoolEntry("Auto Box Width",false):style->AutoBoxWidth());

	hb = new QHBox(stylegb);
	new QLabel(i18n("   Color : "),hb);
	color = new KColorButton(style==0?config->readColorEntry("Style Color",&Qt::blue):style->Color(),hb);
	QObject::connect(color,SIGNAL(changed(const QColor &)),this,SLOT(styleChanged()));
	hb = new QHBox(stylegb);
	new QLabel(i18n(" Line Width : "),hb);
	width = new KIntNumInput(style==0?config->readNumEntry("Style Width",1):style->Width(),hb);
	width->setMinValue(0);
	new QLabel(i18n(" Style : "),hb);
	pencb = new KComboBox(hb);
	pencb->clear();
	for (int i=0;i<6;i++) {
		QPainter pa;
		QPixmap pm( 50, 30 );
		pm.fill(Qt::white);
		pa.begin( &pm );

		pa.setPen((PenStyle)i);
		pa.drawLine(5,15,45,15);
		pa.end();

		pencb->insertItem(pm);
	}
	pencb->setCurrentItem(style==0?config->readNumEntry("Pen Style",1):style->PenStyle());

	hb = new QHBox(stylegb);
	filled = new QCheckBox(i18n("Filled "),hb);
	filled->setChecked(style==0?config->readBoolEntry("Filled",false):style->isFilled());
	fcolor = new KColorButton(style==0?config->readColorEntry("Fill Color",&Qt::green):style->FillColor(),hb);
	QObject::connect(fcolor,SIGNAL(changed(const QColor &)),this,SLOT(styleChanged()));

	hb = new QHBox(stylegb);
	new QLabel(i18n("    Brush : "),hb);
	brushcb = new KComboBox(hb);
	fillBrushBox(brushcb,SRECT,Qt::blue,FFULL,fcolor->color());
	brushcb->setCurrentItem(style==0?config->readNumEntry("Brush",0):style->Brush());

	// read Symbol values from config or from symbol if defined
	SType stype = (SType) (symbol==0?config->readNumEntry("Symbol Type",SNONE):symbol->Type());
	QColor sycolor = (symbol==0?config->readColorEntry("Symbol Color",&Qt::blue):symbol->Color());
	FType sfill = (FType) (symbol==0?config->readNumEntry("Symbol Fill",FNONE):symbol->Fill());
	QColor syfillcolor = (symbol==0?config->readColorEntry("Symbol Fill Color",&Qt::red):symbol->FillColor());
	int sysize = (symbol==0?config->readNumEntry("Symbol Size",5):symbol->Size());
	int sbrush = (symbol==0?config->readNumEntry("Symbol Brush",1):symbol->Brush());

	QGroupBox *symbolgb = new QGroupBox(1,QGroupBox::Horizontal,i18n("Symbol"),styletab);
	hb = new QHBox(symbolgb);
	new QLabel(i18n(" Type : "),hb);
	symbolcb = new KComboBox(hb);
	fillSymbolBox(sycolor,sfill,syfillcolor,sbrush);
	symbolcb->setCurrentItem(stype);
	QObject::connect(symbolcb,SIGNAL(activated(int)),this,SLOT(symbolChanged()));
	scolor = new KColorButton(sycolor,hb);
	QObject::connect(scolor,SIGNAL(changed(const QColor &)),this,SLOT(symbolChanged()));

	hb = new QHBox(symbolgb);
	new QLabel(i18n("    Size : "),hb);
	ssize = new KIntNumInput(sysize,hb);
	ssize->setRange(1,30);

	hb = new QHBox(symbolgb);
	new QLabel(i18n("    Fill : "),hb);
	symbolfillcb = new KComboBox(hb);
	fillSymbolFillBox(stype,sycolor,syfillcolor,sbrush);
	symbolfillcb->setCurrentItem(sfill);
	QObject::connect(symbolfillcb,SIGNAL(activated(int)),this,SLOT(symbolChanged()));

	// needed ???
	ssize->setValue(sysize);
	sfcolor = new KColorButton(syfillcolor,hb);
	QObject::connect(sfcolor,SIGNAL(changed(const QColor &)),this,SLOT(symbolChanged()));

	hb = new QHBox(symbolgb);
	new QLabel(i18n("    Brush : "),hb);
	sbrushcb = new KComboBox(hb);
	fillBrushBox(sbrushcb,stype,sycolor,sfill,syfillcolor);
	sbrushcb->setCurrentItem(sbrush);
	QObject::connect(sbrushcb,SIGNAL(activated(int)),this,SLOT(symbolChanged()));

	return styletab;
}

// called from configure dialog
void Dialog::saveSimpleStyle() {
	KConfig *config = mw->Config();
	config->setGroup( "Plot Simple Style" );
	config->writeEntry("Graph Style",cb2->currentItem());
	config->writeEntry("Sort Points",sortpointscb->isChecked());
	config->writeEntry("Box Width",boxwidth->value());
	config->writeEntry("Auto Box Width",autobox->isChecked());
	config->writeEntry("Style Color",color->color());
	config->writeEntry("Style Width",width->value());
	config->writeEntry("Pen Style",pencb->currentItem());
	config->writeEntry("Filled",filled->isChecked());
	config->writeEntry("Fill Color",fcolor->color());
	config->writeEntry("Brush",brushcb->currentItem());

	config->writeEntry("Symbol Type",symbolcb->currentItem());
	config->writeEntry("Symbol Size",ssize->value());
	config->writeEntry("Symbol Color",scolor->color());
	config->writeEntry("Symbol Fill",symbolfillcb->currentItem());
	config->writeEntry("Symbol Fill Color",sfcolor->color());
	config->writeEntry("Symbol Brush",sbrushcb->currentItem());
}

// called from configure/function dialog
void Dialog::saveSurfaceStyle() {
	KConfig *config = mw->Config();
	config->setGroup( "Plot Surface Style" );

	config->writeEntry("Density Enabled",dcb->isChecked());
	config->writeEntry("Contour Enabled",ccb->isChecked());
	config->writeEntry("Contour Level",numberle->text().toInt());
	config->writeEntry("Contour Color",contourcolor->color());
	config->writeEntry("Contour Width",contourwidthle->text().toInt());
	config->writeEntry("Density Brush",dbrushcb->currentItem());
	config->writeEntry("Colored Contour",coloredcb->isChecked());
	config->writeEntry("Show Mesh",meshcb->isChecked());
	config->writeEntry("Relative Colorscale",relativecb->isChecked());
	config->writeEntry("Threshold",thresholdle->text().toDouble());
}

// called from configure/function dialog
void Dialog::saveErrorbarSettings() {
	if(mw == 0) {
		kdDebug()<<"mw == 0! errorbar settings not saved."<<endl;
		return;
	}

	KConfig *config = mw->Config();
	if(config == 0) {
		kdDebug()<<"config == 0! errorbar settings not saved."<<endl;
		return;
	}
	config->setGroup( "Errorbar" );

	if(ebarcolor != 0)
		config->writeEntry("Color",ebarcolor->color());
	if(ebarstylecb != 0)
		config->writeEntry("Style",ebarstylecb->currentItem());
	if(ebarwidth != 0)
		config->writeEntry("Width",ebarwidth->value());
	if(ebarxsize != 0)
		config->writeEntry("XSize",ebarxsize->value());
	if(ebarysize != 0)
		config->writeEntry("YSize",ebarysize->value());
	if(ebarxtypecb != 0)
		config->writeEntry("XType",ebarxtypecb->currentItem());
	if(ebarytypecb != 0)
		config->writeEntry("YType",ebarytypecb->currentItem());
	if(ebarbcolor != 0)
		config->writeEntry("BaseColor",ebarbcolor->color());
	if(ebarbstylecb != 0)
		config->writeEntry("BaseStyle",ebarbstylecb->currentItem());
	if(ebarbwidth != 0)
		config->writeEntry("BaseWidth",ebarbwidth->value());
}

//! any style properties changed
void Dialog::styleChanged() {
	fillBrushBox(brushcb,SRECT,color->color(),FFULL,fcolor->color());
}
//! any symbol properties changed
void Dialog::symbolChanged() {
	fillSymbolBox(scolor->color(),(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
	fillSymbolFillBox((SType)symbolcb->currentItem(),scolor->color(),sfcolor->color(),sbrushcb->currentItem());
	fillBrushBox(sbrushcb,(SType)symbolcb->currentItem(),scolor->color(),
		(FType)symbolfillcb->currentItem(),sfcolor->color());
}

QVBox* Dialog::annotateValuesTab(QTabWidget *tw, Graph *graph) {
	QVBox *annotatetab = new QVBox(tw);
	QStringList list;

	QHBox *hb = new QHBox(annotatetab);
	new QLabel(i18n("Type : "),hb);
	typecb = new KComboBox(hb);
	list<<i18n("None")<<i18n("x")<<i18n("y")<<i18n("z")<<i18n("x,y")<<i18n("x,y,z");
	typecb->insertStringList(list);
	if (graph)
		typecb->setCurrentItem(graph->getAnnotateValues().Type());

	hb = new QHBox(annotatetab);
	new QLabel(i18n("Position : "),hb);
	positioncb = new KComboBox(hb);
	list.clear();
	list<<i18n("top")<<i18n("bottom")<<i18n("left")<<i18n("right");
	positioncb->insertStringList(list);
	if (graph != 0)
		positioncb->setCurrentItem(graph->getAnnotateValues().Position());

	hb = new QHBox(annotatetab);
	new QLabel(i18n("Distance : "),hb);
	distance = new KIntNumInput(graph!=0?graph->getAnnotateValues().Distance():10,hb);
	distance->setRange(0,100);

	return annotatetab;
}

QVBox* Dialog::errorbarTab(QTabWidget *tw, Symbol *symbol) {
	kdDebug()<<"errorbarTab()"<<endl;
	QVBox *errorbartab = new QVBox(tw);

	KConfig *config = mw->Config();
	config->setGroup( "Errorbar" );

	QGroupBox *bargb = new QGroupBox(1,QGroupBox::Horizontal,i18n("Errorbar"),errorbartab);
	QHBox *hb = new QHBox(bargb);
	new QLabel(i18n(" Color : "),hb);
	ebarcolor = new KColorButton(symbol!=0?symbol->errorBar()->Color():config->readColorEntry("Color",&Qt::blue),hb);
	hb = new QHBox(bargb);
	new QLabel(i18n(" Style : "),hb);
	ebarstylecb = new KComboBox(hb);
	ebarstylecb->clear();
	for (int i=0;i<6;i++) {
		QPainter pa;
		QPixmap pm( 50, 30 );
		pm.fill(Qt::white);
		pa.begin( &pm );

		pa.setPen((PenStyle)i);
		pa.drawLine(5,15,45,15);
		pa.end();

		ebarstylecb->insertItem(pm);
	}
	ebarstylecb->setCurrentItem(symbol!=0?symbol->errorBar()->Style():config->readNumEntry("Style",1));
	hb = new QHBox(bargb);
	new QLabel(i18n(" Width : "),hb);
	ebarwidth = new KIntNumInput(symbol!=0?symbol->errorBar()->Width():config->readNumEntry("Width",1),hb);
	ebarwidth->setRange(0,10);

	QGroupBox *stylegb = new QGroupBox(1,QGroupBox::Horizontal,i18n("Style"),errorbartab);
	hb = new QHBox(stylegb);
	new QLabel(i18n(" XSize : "),hb);
	ebarxsize = new KIntNumInput(symbol!=0?symbol->errorBar()->XSize():config->readNumEntry("XSize",2),hb);
	ebarxsize->setRange(0,20);
	hb = new QHBox(stylegb);
	new QLabel(i18n(" YSize : "),hb);
	ebarysize = new KIntNumInput(symbol!=0?symbol->errorBar()->YSize():config->readNumEntry("YSize",2),hb);
	ebarysize->setRange(0,20);
	hb = new QHBox(stylegb);
	new QLabel(i18n(" XType : "),hb);
	ebarxtypecb = new KComboBox(hb);
	ebarxtypecb->clear();
	for (int i=0;i<ERRORBARNR;i++) {
		QPainter pa;
		QPixmap pm( 50, 30 );
		pm.fill(Qt::white);
		pa.begin( &pm );

		switch(i) {
		case EFLAT:
			pa.drawLine(5,15,45,15);
			pa.drawLine(5,5,5,25);
			pa.drawLine(45,5,45,25);
			break;
		case EARROW:
			pa.drawLine(5,15,45,15);
			pa.drawLine(10,5,5,15);
			pa.drawLine(5,15,10,25);
			pa.drawLine(40,5,45,15);
			pa.drawLine(45,15,40,25);
			break;
		case EY:
			pa.drawLine(10,15,40,15);
			pa.drawLine(5,5,10,15);
			pa.drawLine(10,15,5,25);
			pa.drawLine(45,5,40,15);
			pa.drawLine(40,15,45,25);
			break;
		}
		pa.end();

		ebarxtypecb->insertItem(pm);
	}
	ebarxtypecb->setCurrentItem(symbol!=0?symbol->errorBar()->XType():config->readNumEntry("XType",0));
	new QLabel(i18n(" YType : "),hb);
	ebarytypecb = new KComboBox(hb);
	ebarytypecb->clear();
	for (int i=0;i<ERRORBARNR;i++) {
		QPainter pa;
		QPixmap pm( 50, 30 );
		pm.fill(Qt::white);
		pa.begin( &pm );

		switch(i) {
		case EFLAT:
			pa.drawLine(25,5,25,25);
			pa.drawLine(10,5,40,5);
			pa.drawLine(10,25,40,25);
			break;
		case EARROW:
			pa.drawLine(25,5,25,25);
			pa.drawLine(10,10,25,5);
			pa.drawLine(25,5,40,10);
			pa.drawLine(10,20,25,25);
			pa.drawLine(25,25,40,20);
			break;
		case EY:
			pa.drawLine(25,10,25,20);
			pa.drawLine(10,5,25,10);
			pa.drawLine(25,10,40,5);
			pa.drawLine(10,25,25,20);
			pa.drawLine(25,20,40,25);
			break;
		}
		pa.end();

		ebarytypecb->insertItem(pm);
	}
	ebarytypecb->setCurrentItem(symbol!=0?symbol->errorBar()->YType():config->readNumEntry("YType",0));

	QGroupBox *basegb = new QGroupBox(1,QGroupBox::Horizontal,i18n("Base"),errorbartab);
	hb = new QHBox(basegb);
	new QLabel(i18n(" Color : "),hb);
	ebarbcolor = new KColorButton(symbol!=0?symbol->errorBar()->BaseColor():config->readColorEntry("BaseColor",&Qt::blue),hb);
	hb = new QHBox(basegb);
	new QLabel(i18n(" Style : "),hb);
	ebarbstylecb = new KComboBox(hb);
	ebarbstylecb->clear();
	for (int i=0;i<6;i++) {
		QPainter pa;
		QPixmap pm( 50, 30 );
		pm.fill(Qt::white);
		pa.begin( &pm );

		pa.setPen((PenStyle)i);
		pa.drawLine(5,15,45,15);
		pa.end();

		ebarbstylecb->insertItem(pm);
	}
	ebarbstylecb->setCurrentItem(symbol!=0?symbol->errorBar()->BaseStyle():config->readNumEntry("BaseStyle",1));
	hb = new QHBox(basegb);
	new QLabel(i18n(" Width : "),hb);
	ebarbwidth = new KIntNumInput(symbol!=0?symbol->errorBar()->BaseWidth():config->readNumEntry("BaseWidth",5),hb);
	ebarbwidth->setRange(0,10);

	kdDebug()<<"errorbarTab() DONE"<<endl;
	return errorbartab;
}

void Dialog::fillBrushBox(KComboBox *cb, SType t, QColor c,FType f, QColor sc) {
	int item = cb->currentItem();
	cb->clear();
	// Qt : number of different brush styles
	int BRUSHNR=15;
	for (int i=0;i<BRUSHNR;i++) {
		QPainter pa;
		QPixmap pm( 30, 30 );
		pm.fill(Qt::white);
		pa.begin( &pm );

		Symbol symbol((SType)t,c,10,f,sc,i);
		symbol.draw(&pa,15,15);
		pa.end();

		cb->insertItem(pm);
	}
	cb->setCurrentItem(item);
}

//! fill the symbol combo box with all symbols
void Dialog::fillSymbolBox(QColor c, FType f, QColor sc, int b) {
	int item = symbolcb->currentItem();
	symbolcb->clear();
	for (int i=0;i<SYMBOLNR;i++) {
		QPainter pa;
		QPixmap pm( 30, 30 );
		pm.fill(Qt::white);
		pa.begin( &pm );

		Symbol symbol((SType)i,c,10,f,sc,b);
		symbol.draw(&pa,15,15);
		pa.end();

		symbolcb->insertItem(pm);
	}
	symbolcb->setCurrentItem(item);
}

//! fill the symbol combo box with all symbols
void Dialog::fillSymbolFillBox(SType t, QColor c, QColor sc, int b) {
	int item = symbolfillcb->currentItem();
	symbolfillcb->clear();
	for (int i=0;i<SYMBOLFILLNR;i++) {
		QPainter pa;
		QPixmap pm( 30, 30 );
		pm.fill(Qt::white);
		pa.begin( &pm );

		Symbol symbol(t,c,10,(FType)i,sc,b);
		symbol.draw(&pa,15,15);
		pa.end();

		symbolfillcb->insertItem(pm);
	}
	symbolfillcb->setCurrentItem(item);
}

void Dialog::fileInfo() {
	QStringList fns = QStringList::split(";",filele->text());
	 for ( QStringList::Iterator it = fns.begin(); it != fns.end(); ++it ) {
		QString filename = *it;

		QIODevice *file = KFilterDev::deviceForFile(filename,QString::null,true);
		if(file==0) file = new QFile(filename);

		if ( file->open( IO_ReadOnly ))
			(new FileInfoDialog(mw,0,filename))->show();
		else
			KMessageBox::error(this, i18n("Sorry. Could not open file for reading!"));
	}
}

void Dialog::updateFileInfo() {
	QProcess *proc = new QProcess( this );
	proc->addArgument("file");
	proc->addArgument("-b");
	proc->addArgument(filele->text());
	if (!proc->start())
		kdDebug()<<"Could not start \"file -b "<<filele->text()<<"\""<<endl;
	usleep(100);

	fileinfo->setText(proc->readLineStdout().left(60));
}

void Dialog::selectFile() {
	QStringList f = QFileDialog::getOpenFileNames(QString::null,QString::null, this );
	if (! f.isEmpty() )
		filele->setText(f.join(";"));
}

// widget for import and data dialog
void Dialog::importWidget(QVBox *vb, QString filename, InputFilter filter) {
	QGroupBox *gb = new QGroupBox(1,QGroupBox::Horizontal,i18n("Data Import"),vb);

	KConfig *config = mw->Config();
	config->setGroup( "Import" );

	QHBox *hb = new QHBox(gb);

	filele = new KLineEdit(filename,hb);
	QObject::connect(filele,SIGNAL(textChanged(const QString &)),SLOT(updateFileInfo()));
	KPushButton *newFile = new KPushButton(i18n("Browse"),hb);
	QObject::connect(newFile,SIGNAL(clicked()),SLOT(selectFile()));

	hb = new QHBox(gb);
	fileinfo = new QLabel("UNKNOWN",hb);
	updateFileInfo();
	KPushButton *infopb = new KPushButton(i18n("File Info"),hb);
	infopb->setMaximumWidth(100);
	QObject::connect(infopb,SIGNAL(clicked()),SLOT(fileInfo()));

	QGroupBox *gb2 = new QGroupBox(1,QGroupBox::Horizontal,i18n("Options"),vb);
	hb = new QHBox(gb2);
	new QLabel(i18n("Filter : "),hb);
	filtercb = new KComboBox(hb);
	int i=0;
	while(filteritems[i] != 0) filtercb->insertItem(i18n(filteritems[i++]));
#ifndef HAVE_KEXIDB
	filtercb->removeItem(FKEXIDB);
#endif
	filtercb->setMaximumWidth(350);
	QObject::connect(filtercb,SIGNAL(activated (int)),SLOT(updateFilter(int)));
	filtercb->setCurrentItem(filter);

	hb = new QHBox(gb2);
	simplifycb = new QCheckBox(i18n("simplify whitespaces"),hb);
	simplifycb->setChecked(config->readBoolEntry("SimplifyWhitespace",true));
	emptycb = new QCheckBox(i18n("allow empty entries"),hb);
	emptycb->setChecked(config->readBoolEntry("EmptyEntries",false));
	hb = new QHBox(gb2);
	headercb = new QCheckBox(i18n("import header"),hb);
	headercb->setChecked(config->readBoolEntry("ImportHeader",false));
	samexcb = new QCheckBox(i18n("x column from first file only"),hb);
	samexcb->setChecked(config->readBoolEntry("SameXColumn",false));

	hb = new QHBox(gb2);
	seplabel = new QLabel(i18n("separating character : "),hb);
	sccb = new KComboBox(hb);
	sccb->setEditable(true);
	sccb->insertStrList(separatoritems);
	sccb->setMaximumWidth(100);
	sccb->setCurrentText(config->readEntry("SeparatingCharacter","auto"));

	hb = new QHBox(gb2);
	commlabel = new QLabel(i18n("comment character : "),hb);
	commcb = new KComboBox(hb);
	commcb->setEditable(true);
	commcb->insertStrList(commentitems);
	commcb->setMaximumWidth(100);
	commcb->setCurrentText(config->readEntry("CommentCharacter","#"));
	hb = new QHBox(gb2);
	startlabel = new QLabel(i18n("Start Row : "),hb);
	startle = new KLineEdit(QString::number(config->readNumEntry("StartRow",1)),hb);
	startle->setValidator(new QIntValidator(startle));
	endlabel = new QLabel(i18n(" End Row : "),hb);
	QString end = QString::number(config->readNumEntry("EndRow",-1));
	if(end == "-1")
		end = i18n("END");
	endle = new KLineEdit(end,hb);

	// only for binary data
	hb = new QHBox(gb2);
	varlabel = new QLabel(i18n("variables : "),hb);
	varle =  new KLineEdit(QString::number(config->readNumEntry("BinaryVariables",2)),hb);
	varle->setValidator(new QIntValidator(1,INF,varle));
	varlabel->hide();
	varle->hide();
	hb = new QHBox(gb2);
	binarytypelabel = new QLabel(i18n("Format : "),hb);
	binarytypecb = new KComboBox(hb);
	binarytypelabel->hide();
	binarytypecb->hide();
	QStringList blist;
	blist<<QString("double")<<QString("float")<<QString("long");
	blist<<QString("int (8 Bit)")<<QString("int (16 Bit)")<<QString("int (32 Bit)")<<QString("int (64 Bit)");
	binarytypecb->insertStringList(blist);
	binarytypecb->setCurrentItem(config->readNumEntry("BinaryFormat",0));
	hb = new QHBox(gb2);
	byteorderlabel = new QLabel(i18n("Byte Order : "),hb);
	byteordercb = new KComboBox(hb);
	byteorderlabel->hide();
	byteordercb->hide();
	QStringList bytelist;
	bytelist<<i18n("big endian")<<i18n("little endian");
	byteordercb->insertStringList(bytelist);
	byteordercb->setCurrentItem(config->readBoolEntry("ByteOrder",1));
}

void Dialog::saveImportSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "Import" );

	config->writeEntry("Filename",filele->text());
	config->writeEntry("Filter",filtercb->currentItem());
	config->writeEntry("SimplifyWhitespace",simplifycb->isChecked());
	config->writeEntry("EmptyEntries",emptycb->isChecked());
	config->writeEntry("ImportHeader",headercb->isChecked());
	config->writeEntry("SameXColumn",samexcb->isChecked());
	config->writeEntry("SeparatingCharacter",sccb->currentText());
	config->writeEntry("CommentCharacter",commcb->currentText());
	config->writeEntry("StartRow",startle->text().toInt());
	int end=endle->text().toInt();
	if(endle->text() == i18n("END"))
		end=-1;
	config->writeEntry("EndRow",end);

	// binary stuff
	config->writeEntry("BinaryVariables",varle->text().toInt());
	config->writeEntry("BinaryFormat",binarytypecb->currentItem());
	config->writeEntry("ByteOrder",byteordercb->currentItem());
}

// update import settings when changing filter
void Dialog::updateFilter(int item) {
	// standard widgets (show it when previously hidden)
	filele->setReadOnly(false);
	seplabel->show();
	sccb->show();
	commcb->show();
	commlabel->show();
	startle->show();
	startlabel->show();
	endle->show();
	endlabel->show();
	simplifycb->show();
	varlabel->hide();
	varle->hide();
	binarytypelabel->hide();
	binarytypecb->hide();
	byteorderlabel->hide();
	byteordercb->hide();

	headercb->show();
	samexcb->show();
	simplifycb->setChecked(true);
	emptycb->show();
	emptycb->setChecked(false);
	commcb->setCurrentItem(CSHARP);
	endle->setText(i18n("END"));

	switch(item) {
	case FUSER:	// dont change settings
		break;
	case FTDI:
		simplifycb->setChecked(false);
		sccb->setCurrentItem(STAB);
		startle->setText("2");
		emptycb->setChecked(true);
		break;
	case FLECROY:	// from Floppy
		sccb->setCurrentItem(SCOMMA);
		startle->setText("6");
		break;
	case FLECROYXY:	// XYPairs
		sccb->setCurrentItem(SCOMMA);
		startle->setText("4");
		break;
	case FGOLDSTAR:
		sccb->setCurrentItem(SAUTO);
		startle->setText("20");
		break;
	case FBINARY:
		emptycb->hide();
		seplabel->hide();
		commlabel->hide();
		simplifycb->hide();
		sccb->hide();
		commcb->hide();
		headercb->hide();
		samexcb->hide();

		varlabel->show();
		varle->show();
		binarytypelabel->show();
		binarytypecb->show();
		byteorderlabel->show();
		byteordercb->show();
		break;
	case FKEXIDB:
		//TODO : disable file linedit
		filele->setReadOnly(true);
		seplabel->hide();
		sccb->hide();
		commcb->hide();
		commlabel->hide();
		startle->hide();
		startlabel->hide();
		endle->hide();
		endlabel->hide();
		simplifycb->hide();
		emptycb->hide();
		break;
	}
}

// split a line when reading data
QStringList Dialog::splitLine(QString line, QString sep, bool empty) {
//	kdDebug()<<"Dialog::splitLine() : sep="<<sep<<endl;
	QStringList oneline;

	if (sep == QString("auto"))
		oneline = QStringList::split(' ', line, empty );
	else if(sep == QString("TAB"))
		oneline = QStringList::split(QRegExp(QString("\t")), line, empty );
	else
		oneline = QStringList::split(sep.at(0), line, empty );

	return oneline;
}

// read binary value from d of type type
double Dialog::getBinaryValue(QDataStream *ds, BinaryType type) {
	//kdDebug()<<"Dialog::getBinaryValue() : Type = "<<type<<endl;
	double value=0;

	// TEST
	/*for(int i=0;i<20;i++) {
		Q_UINT8 c;
		*ds>>c;
		printf("%x ",c);
	}
	printf("\n");*/

	//printf("SIZEOF double : %d\n",sizeof(double));

	switch(type) {
	case BDOUBLE: {
		double var;
		*ds>>var;
		//kdDebug()<<"VAR = "<<var<<endl;

		/*printf("var = ");
		for(int i=7;i>=0;i--) {
			unsigned char *p = (unsigned char*) &var+i;
			printf("%x ",*p);
		}
		printf("\n");
		*/

		value=var;
		} break;
	case BFLOAT: {
		float var;
		*ds>>var;
		value=var;
		} break;
	case BLONG: {
		Q_LONG var;
		*ds>>var;
		value=var;
		} break;
	case BINT8: {
		Q_INT8 var;
		*ds>>var;
		value=var;
		} break;
	case BINT16: {
		Q_INT16 var;
		*ds>>var;
		value=var;
		} break;
	case BINT32: {
		Q_INT32 var;
		*ds>>var;
		value=var;
		} break;
	case BINT64: {
		Q_INT64 var;
		*ds>>var;
		value=var;
		} break;
	}
	// kdDebug()<<"	value = "<<value<<endl;

	return value;
}
