/*
 * ftp.c -- Implements transparent access to ftp
 *
 * Created by: Troy Heber (troy.heber@hp.com) 
 *
 * Copyright 2004 Hewlett-Packard Development Company, L.P.
 *
 * Adapted from ftplib.cpp part of LUFS,
 * a free userspace filesystem implementation.
 * See http://lufs.sourceforge.net/ for updates.
 * Copyright (C) 2002 Florin Malita <mali@go.ro>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <netdb.h>
#include <fcntl.h>
#include <time.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>

#include "ftp.h"

/* Parse ftp URL according to RCF 1738
 * ftp://user:password@host:port/path
 * Does not validate any of the tokens!
 * Only parses into tokens based on structure.
 */
	int
parseUrl(char *input, char **user, char **pass, char **host, int *port, char **path)
{
	char *tok, *p, *q, *end;
	char buf[MAX_BUF];
	struct url *parsed;
	int i;

	if  ((parsed=malloc(sizeof(struct url))) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	}

	end = strlen(input) + input;

	if (strlen(input) > MAX_BUF -1 ){
		free(parsed);
		return 71; /* To big. */
	}


	if ((tok=strstr(input,"ftp://")) == NULL){
		free(parsed);
		return 72; /* Not a ftp url */
	}


	tok += 6; /* strip off ftp:// */
	p = q = tok;


	/************************************************************
	 *    User Name and password
	 *    user:password@
	 */

	/* Look for the @ */
	for(p = tok; p != end; p++)
		if (*p == '@')
			break;

	if (*p == '@'){
		for(q = tok; q != p; q++)
			if (*q == ':')
				break;

		if (*q == ':'){ /* have username & passwd */
			i = q - tok;

			if  ((*user=(char *)malloc(i + 1)) == NULL){
				perror("FATAL ERROR: Couldn't allocate enough" 
						"memory");
				exit(10);
			}

			memcpy(*user, tok, i);
			(*user)[i] = '\0';

			i = p - (q+1);

			if  ((*pass=(char *)malloc(i + 1)) == NULL){
				perror("FATAL ERROR: Couldn't allocate enough"
						"memory");
				exit(10);
			}

			memcpy(*pass, q+1, i);
			(*pass)[i] = '\0';

			p++;
			q = p;

		} else{ /* user but no pass */
			i = p - tok;

			if  ((*user=(char *)malloc(i + 1)) == NULL){
				perror("FATAL ERROR: Couldn't allocate enough"
						"memory");
				exit(10);
			}

			memcpy(*user, tok, i);
			(*user)[i] = '\0';

			p++;
			q = p;
		}

	}else{ /* NO user or pass, reset p & q */
		p = q = tok;
	}

	/************************************************************
	 *    Host, port & path
	 *    host:port/path
	 */

	/* Look for the : is there a port? */
	for(; p != end; p++)
		if (*p == ':')
			break;

	if (*p == ':'){ /* we have a port */
		i = p - q;

		if  ((*host=(char *)malloc(i + 1)) == NULL){
			perror("FATAL ERROR: Couldn't allocate enough memory");
			exit(10);
		}

		memcpy(*host, q, i);
		(*host)[i]='\0';

		p++;
		q=p;

		/* Look for the / is there a path? */
		for(; p != end; p++)
			if (*p == '/')
				break;

		if (*p == '/'){ /* we have a path & port */
			i = p - q;

			if (i > 0){
				memcpy(buf, q, i);
				buf[i+1]='\0'; 

				*port = atoi(buf);
			}

			i = end - p;

			if  ((*path=(char *)malloc(i + 1)) == NULL){
				perror("FATAL ERROR: Couldn't allocate enough"
						"memory");
				exit(10);

			}

			memcpy(*path, p, i);
			(*path)[i]='\0';

		} else{ /* Port but no path */
			i = end - q;


			if (i > 0){

				memcpy(buf, q, i);
				buf[i+1]='\0'; 

				*port = atoi(buf);
			}

		}

	}else{ /* no port */
		p = q;

		/* Look for the / is there a path? */
		for(; p != end; p++)
			if (*p == '/')
				break;

		if (*p == '/'){ /* we have a path */
			i = p - q;

			if  ((*host=(char *)malloc(i + 1)) == NULL){
				perror("FATAL ERROR: Couldn't allocate enough"
						"memory");
				exit(10);
			}

			memcpy(*host, q, i);
			(*host)[i]='\0';

			i = end - p;

			if  ((*path=(char *)malloc(i + 1)) == NULL){
				perror("FATAL ERROR: Couldn't allocate enough"
						"memory");
				exit(10);

			}

			memcpy(*path, p, i);
			(*path)[i]='\0';


		} else { /* no path, no port, just host */

			i = end - q;

			if  ((*host=(char *)malloc(i + 1)) == NULL){
				perror("FATAL ERROR: Couldn't allocate enough"
						"memory");
				exit(10);
			}

			memcpy(*host, q, i);
			(*host)[i]='\0';
		}

	}

	free(parsed);
	return 0;
}

	static ssize_t
my_read(int fd, char *ptr)
{
	static int	read_cnt = 0, count = 0;
	static char	*read_ptr;
	static char	read_buf[FTP_MAXLINE];
	struct timespec req;
	req.tv_sec = 0;
	req.tv_nsec = 1000000; /* 1 millisecond */

	if (read_cnt <= 0) {
		while (((read_cnt = read(fd, read_buf, sizeof(read_buf))) < 0 
					&& (count < 5000))){
			if (errno != EINTR && errno != EAGAIN){
				perror("FTP: read failed");
				return -1;
			}

			/* try to sleep for ~ 1ms */
			nanosleep(&req, NULL);
			count++;
		}

		if (read_cnt == 0)
			return 0;
		read_ptr = read_buf;

		if (count >= 5000){
			fprintf(stderr, "FTP: timeout waiting for response from server!\n");
			exit(23);
		}
	}

	read_cnt--;
	*ptr = *read_ptr++;
	return 1;
}

	ssize_t
readline(int fd, void *vptr, size_t maxlen)
{
	ssize_t	n, rc;
	char	c, *ptr;

	ptr = vptr;
	for (n = 1; n < maxlen; n++) {
		if ( (rc = my_read(fd, &c)) == 1) {
			*ptr++ = c;
			if (c == '\n')
				break;	/* newline is stored, like fgets() */
		} else if (rc == 0) {
			*ptr = 0;
			return n - 1;	/* EOF, n - 1 bytes were read */
		} else
			return -1;		/* error, errno set by read() */
	}

	*ptr = 0;	/* null terminate like fgets() */
	return n;
}

	int
protect_write(int fd, const char *vstr, int n)
{
	size_t nleft;
	ssize_t nwritten;
	const char *ptr;
	unsigned long start, stop, timeout=0;
	static struct timezone tz;
	static struct timeval  tv;

	ptr = vstr;
	nleft = n;

	gettimeofday(&tv,&tz);
	start = tv.tv_sec;

	/* Time out ~5 seconds */
	while (nleft > 0 && timeout < 5) {
		if ((nwritten = write(fd, ptr, nleft)) <= 0) {
			if (errno == EINTR || errno == EAGAIN)
				nwritten = 0;
			else
				return -1;
		}

		if (!nwritten){
			gettimeofday(&tv,&tz);
			stop = tv.tv_sec;
			timeout = stop - start;
		}else{
			gettimeofday(&tv,&tz);
			start = tv.tv_sec;
		}

		nleft -= nwritten;
		ptr += nwritten;
	}

	if (timeout >= 5){
		fprintf(stderr, "Maximum write block time of 5 seconds"
				" exceeded, please verify connectivity to"
				" the server!\n");
		exit(24);
	}

	return n;
}


	int
do_rename(char *old, char *new)
{
	int res;
	char *nold, *nnew;

	if ((nold = (char *) malloc(strlen(old)+10)) == NULL){
		perror("cannot allocate enough memory for a buffer.");
		exit (10);
	}

	if ((nnew = (char *) malloc(strlen(new)+10)) == NULL){
		free(nold);
		perror("cannot allocate enough memory for a buffer.");
		exit (10);
	}

	(void)snprintf(nold,(strlen(old)+6), "RNFR %s", old);
	(void)snprintf(nnew,(strlen(new)+6), "RNTO %s", new);

	if ((res = execute_retry(nold, 350, 1)) < 0){
		free(nold);
		free(nnew);
		fprintf(stderr, "RNFR failed!\n");
		return res;
	}else
		free(nold);

	if ((res = execute_retry(nnew, 250, 1)) < 0){
		free(nnew);
		fprintf(stderr, "RNFR failed!\n");
		return res;
	}else
		free(nnew);

	return 0;
}


	int
do_unlink(char *file)
{
	int res;
	char *nfile;

	if ((nfile = (char *) malloc(strlen(file)+10)) == NULL){
		perror("cannot allocate enough memory for a buffer.");
		return -1;
	}

	(void)snprintf(nfile, (strlen(file)+6), "DELE %s", file);

	if ((res = execute_retry(nfile, 250, 1)) < 0){
		free(nfile);
		fprintf(stderr, "DELE failed!\n");
		return res;
	}else
		free(nfile);

	return 0;
}

	int
do_mkdir(char *dir)
{
	int res;
	char *ndir;

	if ((ndir = (char *) malloc(strlen(dir)+10)) == NULL){
		perror("cannot allocate enough memory for a buffer.");
		return -1;
	}

	(void)snprintf(ndir, (strlen(dir)+5), "MKD %s", dir);

	if ((res = execute_retry(ndir, 257, 1)) < 0){
		free(ndir);
		fprintf(stderr, "MKDIR failed!\n");
		return res;
	}else
		free(ndir);

	return 0;
}

	int
do_rmdir(char *dir)
{
	int res;
	char *ndir;

	if ((ndir = (char *) malloc(strlen(dir)+10)) == NULL){
		perror("cannot allocate enough memory for a buffer.");
		return -1;
	}

	(void)snprintf(ndir,(strlen(dir)+5), "RMD %s", dir);

	if ((res = execute_retry(ndir, 0, 1)) < 0){
		free(ndir);
		fprintf(stderr, "RMDIR EXECUTE failed!\n");
		return res;
	}else
		free(ndir);

	/* WarFTP Hack */
	if ((get_response() / 100) != 2){
		fprintf(stderr, "RMDIR Failed1\n");
		return -1;
	}

	return 0;
}

/* Read count bytes from first line of *file. */
	int
do_readFirstLine(char * file, char *line, int count)
{
	int res = 0;
	char *nfile;

	if ((nfile = (char *) malloc(strlen(file)+10)) == NULL){
		perror("cannot allocate enough memory for a buffer.");
		exit(10);
	}
	(void)snprintf(nfile, (strlen(file)+6), "RETR %s", file);
	if ((res = execute_open(nfile, "I", 0)) < 0){
#ifdef DEBUG
		fprintf(stderr,"couldn't open data connection!\n");
		fprintf(stderr,"from: %s\n",nfile);
#endif
		free(nfile);
		return -2;
	}

	if ((res = readline(dfd, line, count)) < 0){
		free(nfile);
		return res;
	}

	free(nfile);
	return 0;
}



	int 
do_open(char * file)
{
	char *nfile;
	int res;

	if ((nfile = (char *) malloc(strlen(file)+10)) == NULL){
		perror("cannot allocate enough memory for a buffer.");
		exit(10);
	}
	(void)snprintf(nfile,(strlen(file)+6), "STOR %s", file);

	if ((res = execute_open(nfile, "I", 0)) < 0){
		free(nfile);
		return res;

	}else{
		free(nfile);
		return dfd;
	}
}

	int
execute_open(char *cmd, char *type, long long offset)
{
	struct sockaddr_in addr, ctrl;
	int res, ssock, tries = 0;
	char buf[FTP_MAXLINE];
	char *strtype;
	fd_set lfds;
	struct timeval tv;
	int flags;

	if ((strtype = (char *) malloc(strlen(type)+10)) == NULL){
		perror("cannot allocate enough memory for a buffer.");
		return -1;
	}

	(void)snprintf(strtype, (strlen(type)+6), "TYPE %s", type);

	if ((dfd < 0) || (offset != last_off) || (cmd != last_cmd)){

again:

		if (tries++ > FTP_MAXTRIES){
			fprintf(stderr, "too many failures!\n");
			free(strtype);
			return -1;
		}

		close_data();
		memset(&addr, 0, sizeof(struct sockaddr_in));
		addr.sin_addr.s_addr = INADDR_ANY;
		addr.sin_port = 0;
		addr.sin_family = AF_INET;

		if ((ssock = socket(PF_INET, SOCK_STREAM, 0)) < 0){
			fprintf(stderr, "socket call failed!\n");
			free(strtype);
			return ssock;
		}

		if ((res = bind(ssock, (struct sockaddr*)&addr, 
						sizeof(struct sockaddr_in))) < 0){
			fprintf(stderr, "bind call failed!\n");
			close(ssock);
			free(strtype);
			return res;
		}

		if ((res = listen(ssock, 2)) < 0){
			fprintf(stderr, "listen call failed!\n");
			close(ssock);
			free(strtype);
			return res;
		}

		res = sizeof(struct sockaddr_in);
		if ((res = getsockname(cfd, (struct sockaddr*)&ctrl, 
						(socklen_t*)&res)) < 0){
			fprintf(stderr, "getsockname call failed!\n");
			close(ssock);
			free(strtype);
			return res;
		}

		res = sizeof(struct sockaddr_in);
		if ((res = getsockname(ssock, (struct sockaddr*)&addr, 
						(socklen_t*)&res)) < 0){
			fprintf(stderr, "getsockname call failed!\n");
			close(ssock);
			free(strtype);
			return res;
		}

		sprintf(buf, "PORT %u,%u,%u,%u,%u,%u",
				ctrl.sin_addr.s_addr & 0xff,
				(ctrl.sin_addr.s_addr >> 8) & 0xff,
				(ctrl.sin_addr.s_addr >> 16) & 0xff,
				(ctrl.sin_addr.s_addr >> 24) & 0xff,
				ntohs(addr.sin_port) >> 8,
				ntohs(addr.sin_port) & 0xff);

		if ((res = execute(buf, 200, 1)) < 0){
			close(ssock);
			if (res == -EAGAIN)
				goto again;
			free(strtype);
			return res;
		}

		if ((res = execute(strtype, 200, 1)) < 0){
			close(ssock);
			if (res == -EAGAIN)
				goto again;
			free(strtype);
			return res;
		}


		sprintf(buf, "REST %llu", offset);
		if (offset && ((res = execute(buf, 350,1)) < 0)){
			close(ssock);
			if (res == -EAGAIN)
				goto again;
			free(strtype);
			return res;
		}


		if ((res = execute(cmd, 150, 1)) < 0){
			close(ssock);
			if (res == -EAGAIN)
				goto again;
			free(strtype);
			return res;
		}

		FD_ZERO(&lfds);
		FD_SET(ssock,&lfds);
		tv.tv_sec = FTP_TIMEOUT;
		tv.tv_usec = 0;
		res = select(ssock+1, &lfds, NULL, NULL, &tv);
		if (res == -1) {
			fprintf(stderr, "select call failed!\n");
			close(ssock);
			free(strtype);
			return res;
		} else {
			if (res == 0) {
				fprintf(stderr, "select call timed out (%d" 
						"second(s))!\n",FTP_TIMEOUT);
				close(ssock);
				free(strtype);
				return -1;
			} else {
				/* someone connected to our listener socket */

				res = sizeof(struct sockaddr_in);
				if ((res = accept(ssock, (struct sockaddr*)&addr, 
								(socklen_t*)&res)) < 0){
					fprintf(stderr, "accept call failed!\n");
					close(ssock);
					free(strtype);
					return res;
				}
			}
		}

		close(ssock);

		dfd = res;

		/* Make the sure the fd is non-blocking */	
		if ((flags=fcntl(dfd, F_GETFL, 0)) < 0){
			perror("fcntl read failed");
			return 66;
		}

		if (fcntl(dfd, F_SETFL, flags | O_NONBLOCK) < 0){
			perror("fcntl set failed");
			return 66;
		}

		last_cmd = cmd;
		last_off = offset;

	}else
#ifdef DEBUG
		printf("keepalive...\n");
#endif

	free(strtype);
	return 0;
}



void
disconnect(int err)
{
	int res;
	if (err && dfd>0){
		close(dfd);
		dfd =-1;

		if (cfd>0){
			close(cfd);
			cfd=-1;
		}
	}
	else if (dfd>0){
		close(dfd);
		dfd = -1;
		/* wait for response "Transfer complete" */
		if ((res = get_response()) != 226)
				fprintf(stderr, "Expected FTP response 226 "
				"(closing data connection). received %d\n",
				res);

		if ((res = execute(QUIT, 0, 0)) < 0)
			fprintf(stderr, "QUIT command failed!\n");
		else{
			if ((res = get_response()) != 221)
				fprintf(stderr, "Expected FTP response 221 "
						"(closing control connection)."
						"received %d\n", res);
		}

		if (cfd>0){
			close(cfd);
			cfd = -1;
		}
	}
}

int
FTPConnect(char *hname, char *user, char *pass)
{
	int	res;
	struct	sockaddr_in pin;
	struct	hostent *hp;
	char 	buf[FTP_MAXLINE];
	char 	system[32];
	int 	flags;
	int 	err = 1;

	if (hostname == NULL)
		hostname = hname;

	if (fuser == NULL)
		fuser = strdup(user);

	if (fpass == NULL)
		fpass = strdup(pass);

	if ((hp = gethostbyname(hostname)) == 0) {
		fprintf(stderr, "Can not resolve hostname %s\n", hostname);
		exit(20);
	}
#ifdef DEBUG
	printf("hostname set: %s\n",hostname);
	printf("fuser set: %s\n",fuser);
	printf("fpass set: %s\n",fpass);
#endif

	memset(&pin, 0, sizeof(pin));
	pin.sin_family = AF_INET;
	pin.sin_addr.s_addr = ((struct in_addr *)(hp->h_addr))->s_addr;
	pin.sin_port = htons(PORT);

	if ((cfd = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
		perror("Could not create socket!");
		return 21;
	}

#ifdef DEBUG
	printf("Client has created a socket\n");
#endif


	/* connect to PORT on HOST */
	if (connect(cfd,(struct sockaddr *) &pin, sizeof(pin)) == -1) {
		perror("Can not connect to socket");
		close(cfd);
		return 22;
	}

#ifdef DEBUG
	printf("Client has connected to PORT: %d\n",PORT);
#endif

	/* Make the sure the fd is non-blocking */	
	if ((flags=fcntl(cfd, F_GETFL, 0)) < 0){
		perror("fcntl read failed");
		return 66;
	}

	if (fcntl(cfd, F_SETFL, flags | O_NONBLOCK) < 0){
		perror("fcntl set failed");
		return 66;
	}

	if ((res = get_response()) != 220){
		disconnect(err);
		fprintf(stderr, "protocol error!\n");
		return 32;
	}

	if (execute(fuser, 0, 0) < 0){
		fprintf(stderr, "USER failed!\n");
		disconnect(err);
		return 33;
	}

	res = get_response();

	if ((res < 0) || ((res != 331) && (res != 230))){
		fprintf(stderr, "invalid user!\n");
		disconnect(err);
		return 34;
	}

	if ((res == 331) && (execute(fpass, 230, 0) < 0)){
		fprintf(stderr, "login failed!\n");
		disconnect(err);
		return 35;
	}

	if ((res = execute(SYST, 0, 0)) < 0){
		fprintf(stderr, "SYST command failed!\n");
		disconnect(err);
		return 36;
	}

	if ((res = readline(cfd, buf, FTP_MAXLINE)) < 0){
		fprintf(stderr, "no response to SYST!\n");
		disconnect(err);
		return res;
	}

	if ((sscanf(buf, "%u %32s", &res, system) != 2) || (res != 215)){
		fprintf(stderr, "bad response to SYST!");
		disconnect(err);
		return 37;
	}

#ifdef DEBUG
	printf("Logged in!! System type is %s\n", system);
#endif

	return 0;
}

	int 
get_response()
{
	int res = 0;
	unsigned multiline = 0;
	char buf[FTP_MAXLINE];

	if (!cfd > 0)
		return -1;

	if ((res = readline(cfd, buf, FTP_MAXLINE)) < 0)
		return res;

#ifdef DEBUG
	printf("line: %s\n", buf);
#endif


	if (buf[3] == '-'){
		if (!sscanf(buf, "%u-", &multiline)){
			fprintf(stderr, "bad response!\n");
			return -1;
		}
#ifdef DEBUG
		printf("multiline: %d\n", multiline);
#endif

	}

	if (multiline)
		do {
			if ((res = readline(cfd, buf, FTP_MAXLINE)) < 0)
				return res;

#ifdef DEBUG
			printf("line: %s\n", buf);
#endif


			if (buf[3] == ' ')
				sscanf(buf, "%u ", &res);
		}while((unsigned)res != multiline);

	else if (!sscanf(buf, "%u", &res)){
		fprintf(stderr, "bad response!\n");
		return -1;
	}

	return res;    
}

	int
close_data()
{

	if (dfd>0){
		close(dfd);
		dfd = -1;
		return get_response();
	}

	return 0;
}

	int
execute(char *cmd, int resp, int reconnect)
{
	int res;
	char *ncmd;

	if ((ncmd = (char *) malloc(strlen(cmd)+10)) == NULL){
		perror("cannot allocate enough memory for a buffer.");
		exit(10);
	}

	close_data();

	if (cfd < 0){
		if ((!reconnect) || (FTPConnect(hostname, fuser, fpass) != 0)){
			fprintf(stderr, "could not connect!\n");
			free(ncmd);
			return -1;
		}
	}

	(void)snprintf(ncmd, (strlen(cmd)+3), "%s\r\n", cmd);

#ifdef DEBUG
	printf("executing: %s\n", ncmd);
#endif

	if ((res = protect_write(cfd, ncmd, strlen(ncmd))) != strlen(ncmd)){
		fprintf(stderr, "write error!: %d of %d\n", (int)strlen(ncmd), res);
		free(ncmd);
		if ((!reconnect) || ((res = FTPConnect(hostname, fuser, fpass)) != 0)){
			fprintf(stderr, "could not reconnect!\n");
			free(ncmd);
			return res;
		}
	}else
		free(ncmd);

	if (resp){
		if ((res = get_response()) != resp){
#ifdef DEBUG
			fprintf(stderr, "command failed!\n");
#endif
			if ((reconnect) && ((res < 0) || (res == 421))){
				if ((res = FTPConnect(hostname, fuser, fpass)) != 0)
					return res;
				else
					return -EAGAIN;
			}else
				return -1;
		}
	}

	return 0;
}

	int
execute_retry(char *cmd, int r, int reconnect)
{
	int res, tries = 0;

	do {
		res = execute(cmd, r, reconnect);
	}while((res == -EAGAIN) && (tries++ < FTP_MAXTRIES));

	return res;
}
