/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <qtimer.h>
#include <qdict.h>
#include <qfile.h>

#include <kconfig.h>
#include <kstddirs.h>
#include <khtml_part.h>

#include <contentswidget.h>
#include <mailobj.h>
#include <clientvfs.h>
#include <vfolder.h>
#include <vfolderattribute.h>
#include <noteview.h>
#include <cardview.h>
#include <groupview.h>
#include <mainwindow.h>
#include <maillistview.h>
#include <mailview.h>
#include <indexmailitem.h>
#include <brokeriface.h>
#include <clientnotifier.h>
#include <esplash.h>
#include <containerwidget.h>
#include <cardtilewidget.h>
#include <grouptilewidget.h>
#include <notewidget.h>
#include <clientrmi.h>
#include <uiservice.h>
#include <korganizerservice.h>
#include <pluginmanager.h>

extern KConfig *GlobalConfig;
extern ESplash *s;

ContentsWidget *ContentsWidget::inst;

// The folder status class
FolderStatus::FolderStatus(const QString &name, const QString &type, const QString &viewType, const unsigned state)
{
  _name=name;
  _type=type;
  _viewType=viewType;
  _state=state;
}

void FolderStatus::changeStatus(const QString &name, const QString &type, const QString &viewType, const unsigned state)
{
  _name=name;
  _type=type;
  _viewType=viewType;
  _state=state;
}

void FolderStatus::reset()
{
  _name="/Magellan";
  _type=QString::null;
  _viewType=QString::null;
  _state=FolderStatus::NO_STATE;
}


// The contents widget class
ContentsWidget::ContentsWidget(QWidget *parent):QWidget(parent)
{
	// init instance pointer, as this is a unique object
	inst=this;

	lastID=0;
	lastMailID=0;
		
	// init the views
	initViews();
	
	// set current folder
	currentFolder = new FolderStatus("/Magellan", QString::null, QString::null, FolderStatus::NO_STATE);
	
	// set current displayed message
	currentMessage=QString::null;
	
	// set current view
	currentView=(QWidget *)(welcomeView->view());
	
	connect(ClientNotifier::thisInstance(), SIGNAL(createVInstance(QString)), this, SLOT(createVInstanceView(QString)));
	connect(ClientNotifier::thisInstance(), SIGNAL(createVInstances(QStringList&)), this, SLOT(createVInstanceView(QStringList&)));
}

ContentsWidget::~ContentsWidget()
{
  delete currentFolder;
}

ContentsWidget *ContentsWidget::ref()
{
	return inst;
}

QString ContentsWidget::getCurrentFolder()
{
	return currentFolder->_name;
}

void ContentsWidget::deleteCurrentFolder()
{
  currentFolder->reset();
}

void ContentsWidget::showMessage(QString address)
{
	currentMessage=address;
	
	MailObject *item=(MailObject *)ClientVFS::thisInstance()->objectRef(address);
	if(item) item->sync();
	
	mailView->setAddress(address);
	
	// mark message as read
	GlobalConfig->setGroup("Read Options");
	int timeout=GlobalConfig->readNumEntry("Read timeout");
	
	if(timeout<0) timeout=3;
	
	if(timeout==0)
	{
		markAsRead(address);
	}
	else
	{
		lastReadMessage=address;
		QTimer::singleShot(timeout*1000, this, SLOT(markAsRead()));
	}
}

void ContentsWidget::showMessage(IndexMailItem *item)
{
	QString address=QString::null;
	
	if(item)
		address=item->getVItem()->path();
	
	showMessage(address);
}

void ContentsWidget::showMessage(QListViewItem *item)
{
	showMessage((IndexMailItem *)item);
}

void ContentsWidget::initViews()
{
	// make the widget stack
	viewStack=new QWidgetStack(this);

	// mail splitter
	mailSplitter=new QSplitter(QSplitter::Vertical, viewStack);
	viewStack->addWidget(mailSplitter, ++lastID);
	mailSplitter->show();
		
	// mail widget stack
	mailviewStack=new QWidgetStack(mailSplitter);
	mailviewStack->show();
	
	// mail view
	mailView=new MailView(mailSplitter);
	mailView->show();
	
	// make the main layout
	mainLayout=new QVBoxLayout(this);
	
	// add the widget stack
	mainLayout->addWidget(viewStack);
	mainLayout->activate();
	
	// get the folders from ClientVFS
	QDict<VFolder> folderDict(ClientVFS::thisInstance()->folderDict());
	
	// scan folders, building the main view for each one
	QDictIterator<VFolder> it(folderDict);
	QDictIterator<VFolder> splashIt(folderDict);
	VFolder *folder;
	int ticks=10; // already have 10 ticks from "mainwidget.cpp"
	while((folder=splashIt.current()))
	{
		ticks ++;
		++ splashIt;
	}
	
	s->setTicks(ticks+1,PROGRESS_WINDOWS);
	s->setActive(PROGRESS_LOGO, false);
	s->setActive(PROGRESS_WINDOWS, true);
	s->tick("Building views...");
	while((folder=it.current()))
	{
		createFolderView(folder);
		++it;
	}
	
	// add plug ins views
	// add KOrganizer view
  UIService *korganizer = dynamic_cast<UIService*> ( PlugInManager::thisInstance()->serviceManager->registeredProvider(KORGANIZERSERVICE_SERVICE) );
  if( korganizer )
  {
    korganizerView=(QWidget *)korganizer->mainWindow(viewStack, "KOrganizer PlugIn");
	  viewStack->addWidget(korganizerView, lastID++);
  }
	
	// add a view for "/Magellan", which doesn't exist in the ClientVFS tree
	welcomeView=new KHTMLPart(viewStack);
	updateWelcomeView();
	
	viewStack->addWidget((QWidget *)welcomeView->view(), lastID++);
	
	((QWidget *)welcomeView->view())->show();
	viewStack->raiseWidget((QWidget *)welcomeView->view());
	viewStack->show();
}

void ContentsWidget::createFolderView(VFolder *folder)
{
		QString viewType=folder->viewType();

		// debug
		printf("Creating view for %s...\n", (const char *)folder->path());
		
		if(viewType=="notes")
		{
			// create a note view
			NoteView *view=new NoteView(folder, viewStack);
			viewStack->addWidget(view, ++lastID);
			view->show();
			
			// associate it with the folder
			folder->setView((TemplateView *)(view));
			
			// debug
			printf("Notes view: %p/%p/%p\n", view, folder->getView(), (QWidget *)folder->getView());
			s->tick(); // views
		}
		else if(viewType=="contacts")
		{
			CardView *view=new CardView(folder, viewStack);
			viewStack->addWidget(view, ++lastID);
			view->show();
			
			// associate it with the folder
			folder->setView((TemplateView *)(view));
			s->tick(); // views
		}
		else if(viewType=="groups")
		{
			GroupView *view=new GroupView(folder, viewStack);
			viewStack->addWidget(view, ++lastID);
			view->show();
			
			// associate it with the folder
			folder->setView((TemplateView *)(view));
			s->tick(); // views
		}
		else if( viewType=="mail" ||
		        viewType=="inbox" || viewType=="outbox" ||
		        viewType=="sent" || viewType=="saved" ||
		        viewType=="drafts" || viewType=="deleted" )
		{
			MailListView *view=new MailListView(folder, mailviewStack, 0, enumType(viewType));
			connect(view, SIGNAL(currentChanged(QListViewItem *)), this, SLOT(showMessage(QListViewItem *)));
			mailviewStack->addWidget(view, ++lastMailID);
			
			// associate the view with the folder
			folder->setView((TemplateView *)(view));
			s->tick(); // views
		}
		else if(viewType=="genmail")
		{
			ContainerWidget *view = new ContainerWidget(folder, viewStack);
			viewStack->addWidget((QWidget *)view->view(), ++lastID);
			((QWidget *)(view->view()))->show();
			
			// asssociate it with the folder
			folder->setView((TemplateView *)(view));
			s->tick(); // views
		}
		else
		{
			// create a generic view
			QWidget *view=new QWidget(viewStack);
			viewStack->addWidget(view, ++lastID);
			view->show();
			
			// asssociate it with the folder
			folder->setView((TemplateView *)(void *)view);
			s->tick(); // views
		}
}

void ContentsWidget::showFolder(QString folderPath)
{
  // Mount and unmount the remote folders.
  // Check if the last selected folder was an IMAP one.
  // If so, we have to unmount the folder.
  // Check if selected folder is an IMAP one.
  // If so, we have to mount the folder.
  // If the IMAP folder is a genmail type, we should check for the subfolders.

	VFolder *folder=ClientVFS::thisInstance()->folderRef(folderPath);
	if( currentFolder->_state == FolderStatus::MOUNT )
	{
	  printf("\n-------------- UNMOUNT -------------------\n");
	  fflush(stdout);
	  currentFolder->_state=FolderStatus::UMOUNT;
	  ClientRMI::ref()->unmount(currentFolder->_name);
	}
	if( currentFolder->_name!=folderPath && folder && folder->type()=="imap" )
	{
  	if( folder->viewType()=="mail" )
  	{
  	  printf("\n-------------- MOUNT -------------------\n");
//  	  printf("\nFolder=%s||ServerFolder=%s\n", folderPath.latin1(), ClientVFS::thisInstance()->map(folderPath).latin1());
//  	  fflush(stdout);
      currentFolder->changeStatus(folderPath, folder->type(), folder->viewType(), FolderStatus::MOUNT);
  	  ClientRMI::ref()->mount(folderPath);
  	}
  	else
  	{
  	  // Create the subfolders
  	  printf("\n-------------- SUBFOLDERS -------------------\n");
  	  fflush(stdout);
      currentFolder->changeStatus(folderPath, folder->type(), folder->viewType(), FolderStatus::NO_STATE);
  	  ClientRMI::ref()->subfolders(folderPath);
  	}
	}
	else
    currentFolder->changeStatus(folderPath, "default", "default", FolderStatus::NO_STATE);

  // debug
  printf("ContentsWidget: showing %s\n", (const char *)folderPath);

	if(folderPath=="/Magellan")
	{
		MainWindow::ref()->setMode("welcome");
		updateWelcomeView();
		currentView=(QWidget *)welcomeView->view();
		viewStack->raiseWidget(currentView);
	}
	else if(folderPath=="/Magellan/Jabber")
	{
		MainWindow::ref()->setMode("jabber");
		updateWelcomeView();
		currentView=(QWidget *)welcomeView->view();
		viewStack->raiseWidget(currentView);
	}
	else if(folderPath=="/Magellan/KOrganizer")
	{
		MainWindow::ref()->setMode("korganizer");
    currentView=korganizerView;
		viewStack->raiseWidget(currentView);
	}
	else
	{
		if(folder)
		{
			if(folder->getView())
			{
				QString viewType=folder->viewType();

				MainWindow::ref()->setMode(viewType);	

		    if( viewType=="mail" ||
		        viewType=="inbox" || viewType=="outbox" ||
		        viewType=="sent" || viewType=="saved" ||
		        viewType=="drafts" || viewType=="deleted" )
				{
					viewStack->raiseWidget(mailSplitter);
					MailListView *view=(MailListView *)(folder->getView());
					
					// debug
//					printf("maillistview: %p\n", view);
					
					mailviewStack->raiseWidget(view);
					currentView=view;
					updateMailListView(view);
				}
				else if(viewType=="genmail")
				{
					printf(" SHOWING GENMAIL\n");
					fflush(stdout);
					ContainerWidget *container=(ContainerWidget *)folder->getView();
					container->buildWidget();
					currentView=(QWidget *)container->view();
					currentView->show();
					viewStack->raiseWidget(currentView);
					printf("container: %p\n", container);
					fflush(stdout);
				}
				else
				{
					// debug
					printf("contents: raising widget...\n");
		
					TemplateView *view=folder->getView();

					if(viewType=="notes")
						currentView=(NoteView *)view;
					else if(viewType=="contacts")
						currentView=(CardView *)view;
					else if(viewType=="groups")
						currentView=(GroupView *)view;
						
					viewStack->raiseWidget(currentView);
				}

				printf("--view: %p\n", folder->getView());
			}
		}
	}
	
	// signal the view change
	emit viewChanged(currentView);
}

void ContentsWidget::updateCurrentView()
{
  showNext("last",true);
}

QWidget *ContentsWidget::getCurrentView()
{
	return currentView;
}

void ContentsWidget::createVInstanceView(QStringList &paths)
{
  for(QStringList::Iterator it=paths.begin(); it!=paths.end(); it++)
    createVInstanceView((*it));
}

void ContentsWidget::createVInstanceView(QString path)
{
	QString itemPath=ClientVFS::thisInstance()->reverseMap(path);

	if(itemPath.isNull())
	{
		printf("Warning: no reverse mapping was found for %s, can't display\n", (const char *)path);
		return;
	}
	
	if(itemPath.contains("."))
	{
		printf("Warning: no views can be run-time created for attributes or part objects, discarding\n");
		return;
	}
	
	// the only case handled here is for newly created objects that don't have views
	
	VObject *item=ClientVFS::thisInstance()->objectRef(itemPath);

	if(item)
	{
	// Eugen C. - I comment the sync line !!!
//		item->sync();
		QString type=item->type();
		
		if(type=="mail" || type=="imap")
		{
			MailListView *folderView=(MailListView *)item->parentFolder()->getView();
			if(folderView)
			{
				item->setView(new IndexMailItem(folderView, (MailObject *)item));
			}
		}
		else if(type=="contacts")
		{
			CardView *cardView=(CardView *)item->parentFolder()->getView();
			if(cardView)
			{
				CardTileView *cardTileView=cardView->getTileView();
				CardTileWidget *w=new CardTileWidget(cardTileView, (ContactObject *)item);
				cardTileView->addChild(w);
				item->setView(w);
				w->show();
			}
		}
		else if(type=="notes")
		{
			NoteView *noteView=(NoteView *)item->parentFolder()->getView();
			if(noteView)
			{
				NoteWidget *w=new NoteWidget(noteView, (NoteObject *)item);
				noteView->addChild(w);
				item->setView(w);
				w->show();
			}
		}
		else if(type=="groups")
		{
			GroupView *groupView=(GroupView *)item->parentFolder()->getView();
			if(groupView)
			{
				GroupTileView *groupTileView=groupView->getTileView();
				GroupTileWidget *w=new GroupTileWidget(groupTileView, (GroupObject *)item);
				groupTileView->addChild(w);
				item->setView(w);
				w->show();
			}
		}
	}
}

int ContentsWidget::enumType(QString type)
{
	if(type=="outbox")
		return outgoing;
	if(type=="sent")
		return sent;
	if(type=="drafts")
		return drafts;
	return mail;
}

QString ContentsWidget::strType(int type)
{
	if(type==outgoing)
		return "outbox";
	if(type==sent)
		return "sent";
	if(type==drafts)
		return "drafts";
	return "mail";
}

void ContentsWidget::updateMailListView(MailListView *view)
{
	if(!view) return;
	
	if(view->isSelectionMultiple())
	{
		// view is in multi selection mode
		
		IndexMailItem *item=(IndexMailItem *)view->currentItem();
		if(!item) item=view->firstSelected();
		showMessage(item);
	}
	else
	{
		// view has at most one selected item
		
		GlobalConfig->setGroup("Misc Mail");
		QString mode=GlobalConfig->readEntry("Goto");
		if(mode.isEmpty()) mode="last";
		showNext(mode, true);
	}
}

void ContentsWidget::showNext(const QString &mode, bool direction)
{
	VFolder *folder=ClientVFS::thisInstance()->folderRef(currentFolder->_name);

	if(!folder)
	{
		printf("Warning: ContentsWidget::showNext() was called with an invalid current folder, bailing out\n");
		return;
	}
	
	if( folder->viewType()!="mail" )
	  return;
	MailListView *view=(MailListView *)folder->getView();
	
	if(!view)
	{
		printf("Warning: ContentsWidget::showNext() was called with a current folder not holding a valid mail list view, bailing out\n");
		return;
	}
	
	IndexMailItem *item=0;
		
	if(mode=="last")
	{
		// keep the last current item

		// get current item
		item=(IndexMailItem *)view->currentItem();
	}
	else if(mode=="top")
	{
		// set the current item to be the first item
		
		item=(IndexMailItem *)view->firstChild();
	}
	else if(mode=="bottom")
	{
		// set the current item to be the last item
		
		item=(IndexMailItem *)view->firstChild();
		if(item)
		{
			while(item->nextSibling()) item=(IndexMailItem *)item->itemBelow();
		}
	}
	else if(mode=="next")
	{
		// jump one item above or below
		
		item=(IndexMailItem *)view->currentItem();
		if(item) item=direction?(IndexMailItem *)item->itemBelow():(IndexMailItem *)item->itemAbove();
	}
	else if(mode=="priority")
	{
		// jump to the next item with "High" or "Highest" priority, above or below
		
		IndexMailItem *temp=(IndexMailItem *)view->currentItem();
		if(!temp) temp=(IndexMailItem *)view->firstChild();
		
		while(!item && temp)
		{
			temp=direction?(IndexMailItem *)temp->itemBelow():(IndexMailItem *)temp->itemAbove();
			
			if(temp)
			{
				MailObject *obj=(MailObject *)temp->getVItem();
				if(obj->priority().left(4)=="High") item=temp;
			}
		}
	}
	else if(mode=="unread")
	{
		// jump to the next item with "New" status, above or below
		
		IndexMailItem *temp=(IndexMailItem *)view->currentItem();
		if(!temp) temp=(IndexMailItem *)view->firstChild();
		
		while(!item && temp)
		{
			temp=direction?(IndexMailItem *)temp->itemBelow():(IndexMailItem *)temp->itemAbove();
			
			if(temp)
			{
				MailObject *obj=(MailObject *)temp->getVItem();
				if(obj->status()=="New") item=temp;
			}
		}
	}
	else if(mode=="flag")
	{
		// jump to the next item that has a little flag, above or below
		
		IndexMailItem *temp=(IndexMailItem *)view->currentItem();
		if(!temp) temp=(IndexMailItem *)view->firstChild();
		
		while(!item && temp)
		{
			temp=direction?(IndexMailItem *)temp->itemBelow():(IndexMailItem *)temp->itemAbove();

			if(temp)
			{			
				MailObject *obj=(MailObject *)temp->getVItem();
				if(obj->flag()) item=temp;
			}
		}
	}
	
	// we should display something even if the previous selection methods didn't find anything useful
	if(!item) item=(IndexMailItem *)view->currentItem();
	if(!item) item=(IndexMailItem *)view->firstChild();
	
	if(item)
	{
		view->clearSelection();
		view->setCurrentItem(item);
		view->setSelected(item, true);
		view->ensureItemVisible(item);
		
		showMessage(item);
	}
	else
	{
		showMessage(QString::null);
	}
}

void ContentsWidget::updateWelcomeView()
{
	QFile fp(locate("appdata","frontpage.html"));
  if(!fp.open(IO_ReadOnly))
	{
		return;
	}
	
	QString htmlSource;
	QString buffer;
	QString mailTable;
  QTextStream txt(&fp);
	
  while(!txt.atEnd())
    buffer+=txt.readLine()+"\n";
  fp.close();
	
	QString temp="<!-- Date -->";
	int pos=buffer.find(temp, 0, false);
	if(pos>=0)
		buffer.replace(pos, temp.length(), getDate());
	temp="<!-- Body -->";
	pos=buffer.find(temp, 0, false);
	if(pos>=0)
	{
		htmlSource = buildUnreadMessagesTable("","");
		buffer.replace(pos, temp.length(), htmlSource);
	}
  welcomeView->autoloadImages(true);
  welcomeView->begin();
  welcomeView->write(buffer);
  welcomeView->end();
}

// This method is designed for recursive building of the mail unread messages
// table for presentation page
QString ContentsWidget::buildUnreadMessagesTable(QString /*parentFolder*/, QString /*parent*/)
{
	QString result;
	QString tmp;
	VFolder *folder;
	QDict<VFolder> dict=ClientVFS::thisInstance()->folderDict();
	QDictIterator<VFolder> it(dict);
	int newMessages=0;
	while((folder=it.current()))
	{
		if(folder->type()=="mail" || folder->type()=="imap")
		{
			VFolderAttribute *attr;
			attr=(VFolderAttribute *)ClientVFS::thisInstance()->_fact_ref(folder->path()+"/default.fattr", "vfolderattribute");
			if(attr) newMessages+=attr->recent;
		}
		++it;
	}
	//rescan to display folders only if necessary
	if(newMessages==0)
	{
		result="<br><br>&nbsp;&nbsp;&nbsp;&nbsp;There are no new messages.";
	}
	else
	{
		result="<br><br>&nbsp;&nbsp;&nbsp;&nbsp;You have unread messages:<br><br>";
		result += "<table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">";
		QDictIterator<VFolder> it2(dict);
		while((folder=it2.current()))
		{
			if(folder->type()=="mail" || folder->type()=="imap")
			{
				VFolderAttribute *attr;
				attr=(VFolderAttribute *)ClientVFS::thisInstance()->_fact_ref(folder->path()+"/default.fattr", "vfolderattribute");
				if(attr && attr->recent)
				{
					result += "<tr><td>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;</td>";
					result += "<td align=\"center\"><font color=\"blue\">";
					tmp.sprintf("%d&nbsp;", attr->recent); result += tmp + "</font></td>";
					result += "<td>&nbsp;<b>unread</b> messages in <a href=\"" + folder->path() + "\"><b>";
					if(folder->path().find("/Magellan/Mail/")!=-1)
						result += folder->name();
					else
						result += folder->path().mid(10, folder->path().length()-10);
					result += "</b></a></td></tr>";
				}
			}
			++it2;
		}
		result+="</table>";
	}
	return result;
}

QString ContentsWidget::getDate()
{
	QString result;
	QDate today;
	result="<font face=\"Arial, Helvetica, sans-serif\" size=\"-1\"> \
		<b><font color=\"#660000\">";
	switch(today.currentDate().dayOfWeek())
	{
		case 1:
			result.append("Monday");
			break;
		case 2:
			result.append("Tuesday");
			break;
		case 3:
			result.append("Wednesday");
			break;
		case 4:
			result.append("Thursday");
			break;
		case 5:
			result.append("Friday");
			break;
		case 6:
			result.append("Saturday");
			break;
		case 7:
			result.append("Sunday");
			break;
		default:
			// This should *NEVER* occur!
			result.append("<Incorrect date set>");
	} // switch
	result.append("</font>,&nbsp;");
	QString number;
	result.append(number.setNum(today.currentDate().day()));
	result.append(" ");
	switch(today.currentDate().month())
	{
		case 1:
			result.append("January");
			break;
		case 2:
			result.append("February");
			break;
		case 3:
			result.append("March");
			break;
		case 4:
			result.append("April");
			break;
		case 5:
			result.append("May");
			break;
		case 6:
			result.append("June");
			break;
		case 7:
			result.append("July");
			break;
		case 8:
			result.append("August");
			break;
		case 9:
			result.append("September");
			break;
		case 10:
			result.append("October");
			break;
		case 11:
			result.append("November");
			break;
		case 12:
			result.append("December");
			break;
	}
	result.append(" ");
	result.append(number.setNum(today.currentDate().year()));
	result.append(" &nbsp;</b></font>");
	return result;
}

void ContentsWidget::markAsRead()
{
	if(currentMessage==lastReadMessage) markAsRead(currentMessage);
}

void ContentsWidget::markAsRead(const QString &address)
{
	MailObject *obj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(address, "mail");
	if(obj && obj->status()=="New")
	{
		obj->sync();
		obj->setStatus("Read");
		obj->save();
		obj->updateViews();
	}
}

void ContentsWidget::markAsUnread(const QString &address)
{
	MailObject *obj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(address, "mail");
	if(obj && obj->status()=="Read")
	{
		obj->sync();
		obj->setStatus("New");
		obj->save();
		obj->updateViews();
	}
}

void ContentsWidget::markAsRead(const QStringList &addresses)
{
	for (unsigned int i = 0; i < addresses.count(); i++) {
		markAsRead(addresses[i]);
	}
}

void ContentsWidget::markAsUnread(const QStringList &addresses)
{
	for (unsigned int i = 0; i < addresses.count(); i++) {
		markAsUnread(addresses[i]);
	}
}

/*
ContentsWidget::ContentsWidget(QWidget *parent, const char
		*name):QWidget(parent, name)
{
	mode=welcome;
  folder=0;
  lastID=0;
  lastCardViewID=0;
	ContentsWid=this;
  frontpage=new KHTMLWidget(this);
  showFrontPage();
	l=new QVBoxLayout(this);
	l->addWidget(frontpage);
	l->activate();
	update=TRUE;
	autosync=TRUE;

	s=new QSplitter(QSplitter::Vertical, this);
	s->hide();
  wstack=new QWidgetStack(s);
	view=new MailView(s);
	QValueList<int> m_size;
	GlobalConfig->setGroup("Appearance");
	int lv_size=GlobalConfig->readNumEntry("List view");
	if(lv_size==0 || lv_size>100)
		lv_size=50;
	m_size<<lv_size*height()/100<<(100-lv_size)*height()/100;
	s->setSizes(m_size);

	printf("Reading mail folders..\n");	
  initMailFolders((LocalMailFolder *)MDispatcher->Folders->Son);

	noteview=new NoteView(this);
	noteview->setChildWidth(90);
	noteview->setChildHeight(70);
	noteview->setChildType(notes);
	noteview->viewport()->setBackgroundColor(QColor("white"));
	
	printf("Reading notes...\n");
	initNotes();
	noteview->hide();

  cardstack=new QWidgetStack(this);
  cardstack->hide();
	
	printf("Reading contacts...\n");
  initContacts();
}

ContentsWidget::~ContentsWidget()
{
 	delete(noteview);
 	delete(view);
  delete(wstack);
 	delete(s);
  delete(frontpage);
}

int ContentsWidget::getMode()
{
	return mode;
}


void ContentsWidget::showFolder(TemplateView<VFolder> *f)
{

  int newmode=f->getType();
	
	if(f==0||(newmode!=mail && newmode!=sent && newmode!=notes
			&& newmode!=drafts && newmode!=outgoing && newmode!=welcome && newmode!=contacts)) return;
	folder=f;

  printk("Switching..\n");
	switch(newmode)
	{
		case mail:
    case sent:
    case drafts:
    case outgoing:
		{
      printf("Mail mode selected.\n");
      if(mode!=mail && mode!=sent && mode!=drafts && mode!=outgoing)
      {
        cardstack->hide();
        noteview->hide();
        delete(l);
        l=new QVBoxLayout(this);
        l->addWidget(s);
        s->show();
        l->activate();
      }

			MWindow->setMode(mail);
      printf("Main window mode was set to mail.\n");
      //mailview=f->Folder->listview;
			mailview=0;
      if(mailview==0)
      {
//        printf("Error: the contents widget was switched to a mail folder (%p) with a null listview. Bailing out.\n", f->Folder);
        break;
      }


      printf("Showing mailview...\n");
      wstack->raiseWidget(mailview);
      printf("Checking if the mailview is empty..\n");
      if(mailview->currentItem()==0)
      {
        printf("No current item, trying to get one...\n");
        showNext(MWindow->arrow_goto, true);
      }
      else
      {
        printf("Current item is not empty, showing current item...\n");
        printf("Current item: %p\n", mailview->currentItem());
        view->setIndex((IndexMailItem *)mailview->currentItem());
      }
			break;
		}
		case notes:
      printf("Notes mode selected.\n");

      s->hide();
      cardstack->hide();
      delete(l);
      l=new QVBoxLayout(this);
      l->addWidget(noteview);
      noteview->show();
      l->activate();

			MWindow->setMode(notes);
			break;
    case contacts:
      printf("Contacts mode selected.\n");
      s->hide();
      noteview->hide();
      delete(l);
      l=new QVBoxLayout(this);
      l->addWidget(cardstack);
      cardstack->show();
//      cardstack->raiseWidget(f->cFolder->cView);
      l->activate();

      MWindow->setMode(contacts);
      break;
    case welcome:
      printf("Welcome mode selected.\n");
      s->hide();
      noteview->hide();
      cardstack->hide();
      delete(l);
      l=new QVBoxLayout(this);
      l->addWidget(frontpage);
      l->activate();

      MWindow->setMode(welcome);
      break;
	}
  mode=newmode;
}

void ContentsWidget::initMailFolders(LocalMailFolder *folder)
{
  while(folder)
  {
    printf("Initializing %s...\n", folder->Name.c_str());
    // create a new maillistview
		mailview=new MailListView(wstack, 0, folder->FolderItem->folder_type);
    wstack->addWidget(mailview, lastID++);
		folder->listview=mailview;

    // populate the maillistview
		IndexClass *idx=folder->IndexList;
		while(idx)
		{
      addIndexMailItem(idx, folder);
			idx=idx->Next;
		}

    // connect it to the appropriate slots
    connect(mailview, SIGNAL(currentChanged(IndexMailItem *)), view,
	 		SLOT(setIndex(IndexMailItem *)));
		connect(mailview, SIGNAL(currentChanged(IndexMailItem *)), this,
			SLOT(markAsRead(IndexMailItem *)));
    connect(mailview, SIGNAL(returnPressed(IndexMailItem *)), this,
      SLOT(openMailWindow(IndexMailItem *)));
    connect(mailview, SIGNAL(doubleClicked(IndexMailItem *)), this,
      SLOT(openMailWindow(IndexMailItem *)));

    if(folder->Son)
      initMailFolders((LocalMailFolder *)folder->Son);

    folder=(LocalMailFolder *)folder->Next;
  }
}

void ContentsWidget::markAllUnread()
{
	if(getFolder()==0||mode!=mail) return;
	pushFlags();
	update=FALSE;
	autosync=FALSE;
	IndexClass *idx=getFolder()->Folder->IndexList;
	while(idx)
	{
		markAsUnread(idx->IndexItem);
		idx=idx->Next;
	}
	updateMailView();
	MDispatcher->updateFolders(getFolder()->Folder);
	syncIndex();
	popFlags();
}

IndexMailItem* ContentsWidget::addIndexMailItem(IndexClass *idx)
{
	if(idx==0||mailview==0) return 0;
//  printf("Creating new indexmailitem..\n");
	IndexMailItem *t=new IndexMailItem((QListView *)mailview, idx);	
	return t;
}

IndexMailItem* ContentsWidget::addIndexMailItem(IndexClass *idx, LocalMailFolder *f)
{
	if(idx==0)
  {
    printf("Fatal error: cannot create an IndexMailItem from a null index.\n");
    return 0;
  }
  if(f==0)
  {
    printf("Fatal error: cannot create an IndexMailItem in a null folder.\n");
    return 0;
  }
  if(f->listview==0)
  {
    printf("Fatal error: cannot create an IndexMailItem in a null MailListView.\n");
    return 0;
  }
//  printf("Creating new indexmailitem..\n");
	IndexMailItem *t=new IndexMailItem((QListView *)f->listview, idx);	
	return t;
}

void ContentsWidget::updateMailView()
{
 	if(mailview)
 		((QListView *)mailview)->triggerUpdate();
}

TemplateView<VFolder> *ContentsWidget::getFolder()
{
	return folder;
}

void ContentsWidget::markAsRead(IndexMailItem *idx)
{
  if(!idx) return;
	if(idx->Index->MStatus.compare("New")!=0||mode!=mail) return;
	idx->Index->MStatus="Read";
	idx->setText(4, "Read");
	if(getFolder()->Folder->Unread>0)
		getFolder()->Folder->Unread--;
	if(update)
	{
		updateMailView();
		MDispatcher->updateFolders(getFolder()->Folder);
	}
	if(autosync)
		getFolder()->Folder->saveIndex();
}

void ContentsWidget::syncIndex()
{
	if(getFolder())
		getFolder()->Folder->saveIndex();
}

void ContentsWidget::markAsUnread(IndexMailItem *idx)
{
	if(idx->Index->MStatus.compare("New")==0||mode!=mail) return;
	idx->Index->MStatus="New";
	idx->setText(4, "New");
	getFolder()->Folder->Unread++;
	if(update)
	{
		MDispatcher->updateFolders(getFolder()->Folder);
		updateMailView();
	}
	if(autosync)
		getFolder()->Folder->saveIndex();
}

void ContentsWidget::setFlag(IndexMailItem *idx)
{
	if(idx->Index->hasFlag||mode!=mail) return;
	idx->Index->hasFlag=1;
	idx->setPixmap(5, QPixmap(flag));
	if(update)
		updateMailView();
	if(autosync)
		getFolder()->Folder->saveIndex();
}

void ContentsWidget::unsetFlag(IndexMailItem *idx)
{
	if(idx->Index->hasFlag==0||mode!=mail) return;
	idx->Index->hasFlag=0;
	idx->setPixmap(5, QPixmap(noflag));
	if(update)
		updateMailView();
	if(autosync)
		getFolder()->Folder->saveIndex();
}

void ContentsWidget::initNotes()
{
	if(!noteview) return;
	NoteWidget *nw;
	GlobalConfig->setGroup("Directory");
	QString dir=GlobalConfig->readEntry("AetheraDir");
	dir+="/Notes/";
	QDir ndir;
	ndir.setPath(dir);
	ndir.setFilter(QDir::Files|QDir::NoSymLinks|QDir::Readable|QDir::Writable);
	for(int i=0;i<ndir.count();i++)
	{
		if(ndir[i]!="."&&ndir[i]!="..")
		{
			nw=new NoteWidget(noteview, 0, dir+ndir[i]);
			noteview->addChild(nw);
			nw->show();
		}
	}
}

void ContentsWidget::initContacts()
{
  GlobalConfig->setGroup("Directory");
  QString dir=GlobalConfig->readEntry("AetheraDir");
  dir+="/Contacts/";
  QDir cdir;
  cdir.setPath(dir);
  CardFolderRoot=new CardFolder(cdir);
  BrowseWid->makeContactFolders(BrowseWid->getRootFolder(), CardFolderRoot);
  BrowseWid->updateTree();
}

void ContentsWidget::showFrontPage()
{
	QFile fp(DATADIR"html/frontpage/frontpage.html");
  if(!fp.open(IO_ReadOnly))
	{
		printf("Warning: could not open "DATADIR"html/frontpage.html\n");
		return;
	}
	QString htmlSource;
	QString buffer;
	QString mailTable;
  QTextStream txt(&fp);
  while(!txt.atEnd())
    buffer+=txt.readLine()+" ";
  fp.close();
	htmlSource.append("<tr valign=\"top\"><td> <table width=\"100%\" border=\"0\">");
	mailTable=buildUnreadMessagesTable(MDispatcher->Folders,"");
	htmlSource.append("<tr valign=\"middle\"><td><table width=\"100%\" border=\"0\">");
  htmlSource.append("<tr><td height=\"33\"><b><font face=\"Arial, Helvetica, sans-serif\"");
	htmlSource.append("size=\"-1\"><img src=\"mail.png\" width=\"60\" height=\"30\"");
	htmlSource.append("align=\"absmiddle\"></font></b></td>");
  htmlSource.append("<td width=\"100%\" height=\"33\"><b>");
	if(mailTable.isEmpty())
	{
		htmlSource.append("<font face=\"Arial, Helvetica, sans-serif\" size=\"-1\">");
		htmlSource.append("You have no new messages.</font></b></td></tr></table>");
	}
		else
	{
		htmlSource.append("<font face=\"Arial, Helvetica, sans-serif\" size=\"-1\">");
		htmlSource.append("You have received new messages.</font></b></td></tr></table></td></tr>");
		htmlSource.append("<tr><td><table width=\"100%\" border=\"0\" cellpadding=\"2\">");
		htmlSource.append(mailTable);
		htmlSource.append("</table>");
	}
	htmlSource.append("<br>");
	if(MDispatcher->getOutbox()->Messages)
	{
		htmlSource.append("<font face=\"Arial, Helvetica, sans-serif\" size=\"-1\">");
		htmlSource.append("There ");
		if(MDispatcher->getOutbox()->Messages>1)
			htmlSource.append("are");
		else
			htmlSource.append("is");
		htmlSource.append(" <font color=\"#FF0000\">");
		QString number;
		htmlSource.append(number.setNum(MDispatcher->getOutbox()->Messages));
		htmlSource.append("</font> unsent message");
		if(MDispatcher->getOutbox()->Messages>1)
			htmlSource.append("s");
		htmlSource.append(" in your ");
		htmlSource.append("<a href=\"magellan_mail://thisInstance/Mail/Outbox\">Outbox</a>.</font>");
	}
	htmlSource.append("<br><br><br><br><br><br></td></tr>");
	int pos=buffer.find("<!-- ******************** -->");
	buffer.replace(pos, strlen("<!-- ******************** -->"), htmlSource);
	pos=buffer.find("<!-- ******************** -->", pos);
	buffer.replace(pos, strlen("<!-- ******************** -->"), getDate());
//	printf(" -------------------------------------------\n");
//	printf("%s\n",(const char *)buffer);
//	printf("--------------------------------------------\n");
  frontpage->enableImages(true);
  frontpage->begin("file:/"DATADIR"html/frontpage/frontpage.html");
  frontpage->write(buffer);
  frontpage->end();
}

void ContentsWidget::openMailWindow(IndexMailItem *idx)
{
  MailWindow *w=new MailWindow(idx->Index);
  w->show();
}

void ContentsWidget::openCurrentMailItem()
{
  MailWindow *w=new MailWindow(mailview->currentItem()->Index);
  w->show();
}


// This method is designed for recursive building of the mail unread messages
// table for presentation page
QString ContentsWidget::buildUnreadMessagesTable(LocalMailFolder *f,
		QString parent)
{
	QString fullname;
	QString table="";
	QString number;
	while(f)
	{
		fullname="";
		if(!parent.isEmpty())
			fullname=parent+"/";
		fullname+=f->Name.c_str();
		if(f->Unread && strcmp(f->Name.c_str(),"Outbox"))
		{
			printf("ContentsWidget : folder \"%s\" : with %d unread messages, %d\n",
				(const char *)fullname, f->Unread, f->Messages);
			table.append("<tr>");
			table.append("<td width=\"10\" height=\"10\" nowrap valign=\"middle\" align=\"center\">");
			table.append("<img src=\"bullet.png\" width=\"7\" height=\"7\">");
			table.append("</td>");
      table.append("<td width=\"100%\">");
			table.append("<font face=\"Arial, Helvetica, sans-serif\" size=\"-1\">");
			table.append("&nbsp;&nbsp;&nbsp;");
			table.append("<a href=\"magellan_mail://Mail/"+fullname+"\">"+fullname+"</a>");
			table.append(" (<font color=\"#FF0000\">"+number.setNum(f->Unread));
			table.append("</font>)</font></td>");
  	  table.append("</tr>");
		}
		if(f->Son)
			table+=buildUnreadMessagesTable(f->Son, fullname);
		f=f->Next;
	}
	return table;
}

QString ContentsWidget::getDate()
{
	QString result;
	QDate today;
	result="<font face=\"Arial, Helvetica, sans-serif\" size=\"-1\"> \
		<b><font color=\"#660000\">";
	switch(today.currentDate().dayOfWeek())
	{
		case 1:
			result.append("Monday");
			break;
		case 2:
			result.append("Tuesday");
			break;
		case 3:
			result.append("Wednesday");
			break;
		case 4:
			result.append("Thursday");
			break;
		case 5:
			result.append("Friday");
			break;
		case 6:
			result.append("Saturday");
			break;
		case 7:
			result.append("Sunday");
			break;
		default:
			// This should *NEVER* occur!
			result.append("<Incorrect date set>");
	} // switch
	result.append("</font>,&nbsp;");
	QString number;
	result.append(number.setNum(today.currentDate().day()));
	result.append(" ");
	switch(today.currentDate().month())
	{
		case 1:
			result.append("January");
			break;
		case 2:
			result.append("February");
			break;
		case 3:
			result.append("March");
			break;
		case 4:
			result.append("April");
			break;
		case 5:
			result.append("May");
			break;
		case 6:
			result.append("June");
			break;
		case 7:
			result.append("July");
			break;
		case 8:
			result.append("August");
			break;
		case 9:
			result.append("September");
			break;
		case 10:
			result.append("October");
			break;
		case 11:
			result.append("November");
			break;
		case 12:
			result.append("December");
			break;
	}
	result.append(" ");
	result.append(number.setNum(today.currentDate().year()));
	result.append(" &nbsp;</b></font>");
	return result;
}

void ContentsWidget::saveAppearance()
{
	GlobalConfig->setGroup("Appearance");
	GlobalConfig->writeEntry("List view", 100*(height()-view->height())/height());
}
*/


















