/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <stdio.h>

#include <qtooltip.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qtimer.h>
#include <qtextcodec.h>

#include <kmainwindow.h>
#include <kiconloader.h>
#include <kmenubar.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kcharsets.h>

#include <composerwindow.h>
#include <ebutton.h>
#include <epopupbutton.h>
#include <toolbarseparator.h>
#include <clientrmi.h>
#include <brokeriface.h>
#include <clientvfs.h>
#include <mailobj.h>
#include <vpartattribute.h>
#include <partobj.h>
#include <vmailattribute.h>
#include <texthtml.h>
#include <textprocessor.h>
#include <mimecodec.h>

extern KConfig *GlobalConfig;
extern TextHtmlConvertor *textHtmlConverter;

QList<ComposerWindow> ComposerWindow::composerList;

void ComposerWindow::openComposer()
{
	ComposerWindow *cWin=new ComposerWindow();
	composerList.append(cWin);
	cWin->show();
}

void ComposerWindow::openComposer(const QString &var, ComposeMode mode)
{
	ComposerWindow *cWin;
	
	// check if the composer for this message is already open
	if(mode==EditMessage)
	{
		for(unsigned int i=0; i < composerList.count(); i++)
		{
			cWin=composerList.at(i);
			if(cWin->messageID==var)
			{
				cWin->show();
				return;
			}
		}
	}

	cWin=new ComposerWindow();
	
	if(mode==EditMessage)
	{	
		cWin->messageID=var;
		cWin->loadMessage();
	}
	else if(mode==Forward)
	{
		cWin->loadFromFwdSource(var);
	}
	else if(mode==ForwardInline)
	{
		cWin->loadFromFwdSourceInline(var);
	}
	else if(mode==ForwardQuoted)
	{
		cWin->loadFromFwdSourceQuoted(var);
	}
	else if(mode==ForwardAsAttachment)
	{
		cWin->loadFromFwdSourceAsAttachment(var);
	}
	else if(mode==Reply)
	{
		cWin->loadFromRepSource(var);
	}
	else if(mode==ReplyAll)
	{
		cWin->loadFromRepSource(var, true);
	}
	else if(mode==SendObject)
	{
		cWin->loadObjectAsAttachment(var);
	}
	else if(mode==UsingRecipient)
	{
		cWin->composerWidget->toField->setAddressList(AddressListClass((const char *)var));
	}
	
	composerList.append(cWin);
	cWin->show();
}

void ComposerWindow::openComposer(const QStringList &var, ComposeMode mode)
{
	ComposerWindow *cWin=new ComposerWindow();
	
	if(mode==Forward)
	{
		cWin->loadFromFwdSource(var);
	}
	else if(mode==SendObject)
	{
		for(unsigned int i=0; i < var.count(); i++)
			cWin->loadObjectAsAttachment(var[i]);
	}
	else if(mode==UsingRecipient)
	{
		cWin->composerWidget->toField->setAddressList(AddressListClass((const char *)var.join(", ")));
	}
	
	composerList.append(cWin);
	cWin->show();
}

ComposerWindow::~ComposerWindow()
{
	GlobalConfig->setGroup("Composer");
	GlobalConfig->writeEntry("Geometry", geometry());
	
	composerList.removeRef(this);
}

ComposerWindow::ComposerWindow():messageDirty(false),messageChanged(false),forcedClose(false)
{
	sendAccounts=new QPopupMenu;
	sendLaterAccounts=new QPopupMenu;
	
	QPopupMenu *file=new QPopupMenu();
	file->insertItem("Send &now", this, SLOT(sendUsingDefault()));
	file->insertItem("Send now using", sendAccounts);
	file->insertItem("Send &later", this, SLOT(sendLaterUsingDefault()));
	file->insertItem("Send later using", sendLaterAccounts);
	file->insertSeparator();
	QPopupMenu *insert=new QPopupMenu();
	insert->insertItem("&Inline");
	insert->insertItem("&Quoted");
	file->insertItem("Insert &file", insert);
	file->insertItem("&Attach file", this, SLOT(addFileAttachment()));
	file->insertItem("Attach personal &vCard");
	file->insertItem("&Save", this, SLOT(saveMessage()), CTRL+Key_S);
	QPopupMenu *save=new QPopupMenu();
	save->insertItem("to a f&ile");
	save->insertItem("to &Drafts", this, SLOT(saveAsDraft()));
	save->insertItem("to &Templates");
	file->insertItem("Save &to", save);
	file->insertSeparator();
	file->insertItem("&Close", this, SLOT(close()));
	QPopupMenu *edit=new QPopupMenu();
	edit->insertItem("Cu&t");
	edit->insertItem("&Copy");
	edit->insertItem("&Paste");
	edit->insertItem("Paste &quoted");
	edit->insertSeparator();
	edit->insertItem("C&lear");
	edit->insertItem("&Select all");
	edit->insertSeparator();
	edit->insertItem("&Find");
	edit->insertItem("F&ind again");
	edit->insertItem("Replace");
	QPopupMenu *format=new QPopupMenu();
	format->insertItem("&Numbers");
	format->insertItem("&Bullets");
	format->insertItem("&Horizontal line");
	format->insertSeparator();
	format->insertItem("&Quote");
	format->insertItem("&Unquote");
	format->insertItem("Shift &left");
	format->insertItem("Shift &right");
	format->insertSeparator();
	format->insertItem("&Word wrap");
	QPopupMenu *tools=new QPopupMenu();
	tools->insertItem("Edit &signature");
	tools->insertItem("Edit personal &vCard");
	tools->insertItem("&Spelling");
	tools->insertItem("&Address Book");
	priority=new QPopupMenu();
	priority->insertItem("&Highest", this, SLOT(setPriority(int)), 0, MENU_HIGHEST_PRIORITY);
	priority->setItemParameter(MENU_HIGHEST_PRIORITY, 2);
	priority->insertItem("Hi&gh", this, SLOT(setPriority(int)), 0, MENU_HIGH_PRIORITY);
	priority->setItemParameter(MENU_HIGH_PRIORITY, 1);
	priority->insertItem("&Normal", this, SLOT(setPriority(int)), 0, MENU_NORMAL_PRIORITY);
	priority->setItemParameter(MENU_NORMAL_PRIORITY, 0);
	priority->setItemChecked(MENU_NORMAL_PRIORITY, true);
	priority->insertItem("L&ow", this, SLOT(setPriority(int)), 0, MENU_LOW_PRIORITY);
	priority->setItemParameter(MENU_LOW_PRIORITY, -1);
	priority->insertItem("&Lowest", this, SLOT(setPriority(int)), 0, MENU_LOWEST_PRIORITY);
	priority->setItemParameter(MENU_LOWEST_PRIORITY, -2);
	QPopupMenu *options=new QPopupMenu();
	options->insertItem("&Set priority", priority);
	options->insertSeparator();
	options->insertItem("&Message preferences");
	options->insertItem("&Composer preferences");
	QPopupMenu *help=new QPopupMenu();
	help->insertItem("About");
	menuBar()->insertItem("&File", file);
	menuBar()->insertItem("&Edit", edit);
	menuBar()->insertItem("&Format", format);
	menuBar()->insertItem("&Tools", tools);
	menuBar()->insertItem("&Preferences", options);
	menuBar()->insertSeparator();
	menuBar()->insertItem("&Help", help);
  statusBar()->message("");
	
	// main toolbar
	sendButton=new EPopupButton(toolBar("main"));
	sendButton->button()->setNormalPixmap(BarIcon("getmail-normal"));
	sendButton->button()->setFocusPixmap(BarIcon("getmail-focus"));
	sendButton->button()->setClickedPixmap(BarIcon("getmail-focus"));
	sendButton->button()->setText("Send message ");
	
	toolSeparator(toolBar("main"));
	
	EButton *signature_button, *spelling_button;
	signature_button=new EButton(BarIcon("signature"), toolBar("main"));
	spelling_button=new EButton(BarIcon("spell-check"), toolBar("main"));

	toolSeparator(toolBar("main"));

	higherPriorityButton=new EButton(BarIcon("high-priority"), toolBar("main"));
	connect(higherPriorityButton, SIGNAL(clicked()), this, SLOT(higherPriority()));
	QToolTip::add(higherPriorityButton, "Set high priority");
	EButton *lowerPriorityButton=new EButton(BarIcon("low-priority"), toolBar("main"));
	connect(lowerPriorityButton, SIGNAL(clicked()), this, SLOT(lowerPriority()));
	QToolTip::add(lowerPriorityButton, "Set low priority");
	EButton *attach_button=new EButton(BarIcon("attach"), toolBar("main"));
	connect(attach_button, SIGNAL(clicked()), this, SLOT(addFileAttachment()));
	QToolTip::add(attach_button, "Attach file");
	
  toolSeparator(toolBar("main"));
	
  EButton *help_button=new EButton(toolBar("main"));
  help_button->setNormalPixmap(BarIcon("help-normal"));
  help_button->setFocusPixmap(BarIcon("help-focus"));
  help_button->setClickedPixmap(BarIcon("help-focus"));
	
	help_button->setFixedWidth(help_button->width());

	// edit toolbar
	EButton *cut_button=new EButton(BarIcon("cut"), toolBar("edit"));
	QToolTip::add(cut_button, "Cut");
	EButton *copy_button=new EButton(BarIcon("copy"), toolBar("edit"));
	QToolTip::add(copy_button, "Copy");
	EButton *paste_button=new EButton(BarIcon("paste"), toolBar("edit"));
	QToolTip::add(paste_button, "Paste");
	
	toolSeparator(toolBar("edit"));
	
	EButton *bullets_button, *numbering_button, *hline_button;
	bullets_button=new EButton(BarIcon("bullets"), toolBar("edit"));
	numbering_button=new EButton(BarIcon("numbering"), toolBar("edit"));
	hline_button=new EButton(BarIcon("hr-line"), toolBar("edit"));
	
	toolSeparator(toolBar("edit"));
	
	EButton *indent_button, *deindent_button;
	indent_button=new EButton(BarIcon("indent-normal"), toolBar("edit"));
	deindent_button=new EButton(BarIcon("deindent-normal"), toolBar("edit"));

	toolSeparator(toolBar("edit"));
	
	EButton *quote_button, *dequote_button;
	quote_button = new EButton(BarIcon("quote"), toolBar("edit"));
	dequote_button = new EButton(BarIcon("dequote"), toolBar("edit"));
	
	dequote_button->setFixedWidth(dequote_button->width());

	// other widgets	
	composerWidget=new ComposerWidget(this);
	setCentralWidget(composerWidget);
	
	toolBar("main")->show();
	toolBar("edit")->show();
	composerWidget->show();

	// get size
	GlobalConfig->setGroup("Composer");
	QRect defaultGeometry(100, 100, 700, 500);
	setGeometry(GlobalConfig->readRectEntry("Geometry", &defaultGeometry));

	int saveTimeout=GlobalConfig->readNumEntry("Save timeout");
	if(saveTimeout==0) saveTimeout=3;

	saveTimer=new QTimer(this);
	connect(saveTimer, SIGNAL(timeout()), this, SLOT(ackAutoSaveTimer()));
	saveTimer->start(saveTimeout*60*1000);
	
	// initial caption
	QWidget::setCaption("Aethera composer: no subject");
		
	// build initial account list
	buildAccountList();
	
	// connect stuff
	connect(sendAccounts, SIGNAL(aboutToShow()), this, SLOT(buildAccountList()));
	connect(sendLaterAccounts, SIGNAL(aboutToShow()), this, SLOT(buildAccountList()));
	connect(sendButton->popup(), SIGNAL(aboutToShow()), this, SLOT(buildAccountList()));
	connect(sendButton->button(), SIGNAL(clicked()), this, SLOT(sendUsingDefault()));
	connect(composerWidget->editor->editBox, SIGNAL(textChanged()), this, SLOT(ackTextChanged()));
	connect(composerWidget->subjectField, SIGNAL(textChanged(const QString &)), this, SLOT(ackSubjectChanged(const QString &)));
	connect(composerWidget->toField, SIGNAL(addressChanged()), this, SLOT(ackAddressChanged()));
	connect(composerWidget->ccField, SIGNAL(addressChanged()), this, SLOT(ackAddressChanged()));
	connect(composerWidget->bccField, SIGNAL(addressChanged()), this, SLOT(ackAddressChanged()));
	
	messagePriority=0;
}

void ComposerWindow::sendUsingDefault()
{
	// todo step 1
	if (!canSend())
		return;

	// find index in the account list
	int index = -1;

	if (sourceAccount.isEmpty())
		 sourceAccount = ClientRMI::ref()->getDefaultAccount();

	for (unsigned int i=0 ; i < accountList.count(); i++)
		if (accountList[i] == sourceAccount)
			index=i;
	
	// call sendUsing(int accountNumber)
	if (index != -1)
		sendUsing(index);
	else
		QMessageBox::warning(this, "Aethera: No accounts defined", 
				"Cannot send message, there are no active accounts", QMessageBox::Ok, 0, 0);
}

void ComposerWindow::sendUsing(int accountNumber)
{
	// todo step 1
	if(!canSend()) return;
			
	// set account
	sourceAccount=accountList[accountNumber];
		
	// save to Outbox
	saveToOutbox();
	
	// set scheduling data and outgoing account
	MailObject *obj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(messageID, "mail");
	if(obj)
	{
		obj->setScheduling("");
		obj->setAccount(sourceAccount);
		obj->save();
	}
		
	// try to send message
	sendMessage();
	
	// forcibly close window
	forcedClose=true;
	close();
}

void ComposerWindow::sendLaterUsingDefault()
{
	// todo step 1
	if(!canSend()) return;
	
	int index=-1;
	
	// find index in the account list
	if(sourceAccount.isEmpty())
		sourceAccount=ClientRMI::ref()->getDefaultAccount();

	for (unsigned int i = 0; i < accountList.count(); i++)
		if (accountList[i] == sourceAccount)
			index = i;
	
	// call sendLaterUsing(int accountNumber)
	if(index!=-1)
		sendLaterUsing(index);
	else
		QMessageBox::warning(this, "Aethera: No accounts defined", 
				"Cannot send message, there are no active accounts", QMessageBox::Ok, 0, 0);
}

bool ComposerWindow::canSend()
{
	if(composerWidget->toField->isEmpty())
	{
		QMessageBox::warning(this, "Aethera: No address",
				"Please specify a recipient in the To: field", QMessageBox::Ok, 0, 0);
		return false;
	}
	
	return true;
}

void ComposerWindow::sendLaterUsing(int accountNumber)
{
	// todo step 1
	if(!canSend()) return;
	
	// set account
	sourceAccount=accountList[accountNumber];
	
	// save to Outbox; the message will be sent at the next Send&Receive activation
	saveToOutbox();
	
	// set scheduling data and outgoing account
	MailObject *obj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(messageID, "mail");
	if(obj)
	{
		obj->setScheduling("");
		obj->setAccount(sourceAccount);
		obj->save();
	}
		
	// forcibly close window
	forcedClose=true;
	close();
}

void ComposerWindow::sendDefault()
{
	if(isOffline())
		sendLaterUsingDefault();
	else
		sendUsingDefault();
}

void ComposerWindow::closeEvent(QCloseEvent *ev)
{
	// todo step 1
	bool shouldSave=false, shouldSend=false, shouldSendLater=false, shouldDelete=false;
	
	// if forced, just close window
	if(forcedClose)
	{
		ev->accept();
		return;
	}
	
	// do we have a copy in Drafts or Outbox?
	bool inDrafts, inOutbox=false;
	
	inDrafts=ClientVFS::thisInstance()->map(messageID).contains("sysmail_drafts");
	if(!inDrafts) 
		inOutbox=ClientVFS::thisInstance()->map(messageID).contains("sysmail_outbox");
	
	if(!messageID.isEmpty() && !BrokerInterface::thisInstance()->exists(messageID))
	{
		// message was deleted by the user from the "Drafts" folder
		inDrafts=false;
		inOutbox=false;
		messageDirty=true;
	}
	
	// we have a copy in Drafts. does the user want to cancel close, keep it, or delete it?
	if(messageChanged && !inDrafts && !inOutbox)
	{
		switch(QMessageBox::information(this, "Aethera: Save message?",
					"You can save this message in the Drafts folder\n"
					"and resume composition later. Save message?",
					"&Save", "&Discard", "&Cancel", 0, 2))
		{
			case 0:
				shouldSave=true;
				break;
			case 1:
				shouldDelete=true;
				break;
			case 2:
				ev->ignore();
				return;
		}
	}
	
	if(inDrafts)
	{
		// we have a copy in Drafts, does the user want to keep it?
		switch(QMessageBox::information(this, "Aethera: Keep message?",
					"This message was saved in the Drafts folder.\n"
					"Do you want to keep this message and resume composition later?",
					"&Keep", "&Discard", "&Cancel", 0, 2))
		{
			case 0:
				shouldSave=true;
				break;
			case 1:
				shouldDelete=true;
				break;
			case 2:
				ev->ignore();
				return;
		}
	}
	
	if(inOutbox)
	{
		// we have a copy in Outbox. does the user want to cancel close, send it now, later or delete it?
		switch(QMessageBox::information(this, "Aethera: Send message?",
					"This message was saved in the Outbox folder.\n"
					"You can send this message or discard it. Send message?",
					"&Send", "&Discard", "&Cancel", 0, 2))
		{
			case 0:
				shouldSend=true;
				shouldSendLater=QMessageBox::information(this, "Aethera: Send now?",
						"Send message now or later?", "&Now", "&Later", QString::null, 0, 1);
				break;
			case 1:
				shouldDelete=true;
				break;
			case 2:
				ev->ignore();
				return;
		}
	}
		
	// delete message, if needed
	if(shouldDelete && !messageID.isEmpty())
		BrokerInterface::thisInstance()->remove(messageID);

	// send message, if needed
	if(shouldSend)
	{
		if(shouldSendLater)
			sendLaterUsingDefault();
		else
			sendUsingDefault();
	}
	
	// save message, if needed
	if(shouldSave)
	{
		saveMessage();
	}
	
	// close window
	ev->accept();
}

void ComposerWindow::buildAccountList()
{
	// todo step 1

 	sendAccounts->QMenuData::clear();
 	sendLaterAccounts->QMenuData::clear();
 	sendButton->popup()->QMenuData::clear();
	
	// set up default account
	if(sourceAccount.isEmpty()) sourceAccount=ClientRMI::ref()->getDefaultAccount();
	
	// get the account list from server using ClientRMI
	QStringList allAccountList=ClientRMI::ref()->getAccountNames();
	accountList.clear();
	
	for (unsigned int i = 0; i < allAccountList.count(); i++)
		if (ClientRMI::ref()->isAccountActive(allAccountList[i]))
			accountList.append(allAccountList[i]);

	// build menus
	for (unsigned int i = 0; i < accountList.count(); i++)
	{
		sendAccounts->insertItem(accountList[i], this, SLOT(sendUsing(int)), 0, i);
		sendLaterAccounts->insertItem(accountList[i], this, SLOT(sendLaterUsing(int)), 0, i);
		
		if(isOffline())
		{
			sendButton->popup()->insertItem(accountList[i], this, SLOT(sendLaterUsing(int)), 0, i);
			if(!sourceAccount.isEmpty())
				QToolTip::add(sendButton, "Send later using account "+sourceAccount);
		}
		else
		{
			sendButton->popup()->insertItem(accountList[i], this, SLOT(sendUsing(int)), 0, i);
			if(!sourceAccount.isEmpty())
				QToolTip::add(sendButton, "Send using account "+sourceAccount);
		}
	}
}

bool ComposerWindow::isOffline()
{
	// get offline mode
	GlobalConfig->setGroup("Send Options");
	return GlobalConfig->readEntry("Work offline")=="Yes";
}

void ComposerWindow::loadMessage()
{
	// todo step 1
	
	// load text, subject, recipients, priority, extended status and scheduling from server
	MailObject *obj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(messageID, "mail");
	
	if(obj)
	{
		QString text;
		
		if(obj->hasText())
		{
			text=obj->text();
		}
		else if(obj->hasHTML())
		{
			text=obj->html();
			textHtmlConverter->html2text(text);
			
		}

		messageDirty=true;
		if(text.length()) messageChanged=true;
				
		composerWidget->setText(text);
		
		composerWidget->setToAddresses(obj->to());
		composerWidget->setCcAddresses(obj->cc());
		composerWidget->setBccAddresses(obj->bcc());
		composerWidget->setSubject(obj->subject());

		messagePriority=priorityFromString(obj->priority());
		scheduling=obj->scheduling();
		extendedStatus=obj->extendedStatus();
		
		composerWidget->attachmentPane->setAddress(messageID);
		if(obj->attachments())
			composerWidget->attachmentPane->show();
		
		// @todo: load the extended status in the status view		
	}
	else
	{
		QMessageBox::warning(this, "Aethera: Message does not exist", "The composer attempted to load a non-existent message.\n"
			"Probably the message you have referenced was dispatched and deleted by the outgoing mail spooler.\n", QMessageBox::Ok, 0, 0);
		messageID="";
	}
}

void ComposerWindow::loadFromFwdSource(const QString &sourcePath)
{
	GlobalConfig->setGroup("Read Options");
	QString fwType=GlobalConfig->readEntry("Forward type");
	if(fwType=="Inline") loadFromFwdSourceInline(sourcePath);
	else if(fwType=="As attachment") loadFromFwdSourceAsAttachment(sourcePath);
	else loadFromFwdSourceQuoted(sourcePath);
}

void ComposerWindow::initBasicMessage(const QString &subject, const QString &text)
{
	// create new message object
	messageID=BrokerInterface::thisInstance()->createObject("/Magellan/Mail/Drafts");
	
	if(!messageID.isNull())
	{
		MailObject *obj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(messageID, "mail");
		
		if(obj)
		{
			// set subject
			obj->setSubject(subject);
			
			// set text
			QString partID;

			partID=BrokerInterface::thisInstance()->createObject(messageID);

			VPartAttribute *attr=(VPartAttribute *)ClientVFS::thisInstance()->_fact_ref(partID+".attr", "vpartattribute");

			if(!attr)
			{
				QMessageBox::warning(this, "Aethera: Could not retrieve text part", "Could not retrieve a text part in this message.\n"
						"This is an unexpected error and the composer cannot allow to further edit this message.", QMessageBox::Ok, 0, 0);
				forcedClose=true;
				close();
			}

			attr->sync();
			attr->mtype="text";
			attr->mimetype="text/plain";
			attr->encoding="8bit";
			attr->charset=getDefaultCharset();
			attr->save();
			
			delete attr;
			
			BrokerInterface::thisInstance()->setObjectData(partID, QCString(text));
		}
	}
	else
	{
		QMessageBox::warning(this, "Aethera: Could not create text part", "Could not create a text part in this message.\n"
				"This is an unexpected error and the composer cannot allow to further edit this message.", QMessageBox::Ok, 0, 0);
		forcedClose=true;
		close();
	}
}

void ComposerWindow::attachObject(const QString &path, const QString &name, const QString &mimeType, const QString &encoding)
{
	QString partID=BrokerInterface::thisInstance()->createObject(messageID);

	PartObject *part=(PartObject *)ClientVFS::thisInstance()->_fact_ref(partID, "part");

	if(part)
	{
		part->setFilename(name);
		part->setPartType("attachment");
		part->setMimeType(mimeType);
		part->setEncoding(encoding);
		
		part->save();

		QByteArray messageData;
		BrokerInterface::thisInstance()->getObjectData(path, messageData);
		BrokerInterface::thisInstance()->setObjectData(partID, messageData);

		delete part;
	}
	else
	{
		QMessageBox::warning(this, "Aethera: Could not create attachment", "Could not create an attachment in this message.\n"
				"This is an unexpected error and the composer cannot allow to further edit this message.", QMessageBox::Ok, 0, 0);
		forcedClose=true;
		close();
	}
}

void ComposerWindow::copyAttachmentsFrom(const QString &sourcePath)
{
	MailObject *srcObj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(sourcePath, "mail");
	
	if(srcObj)
	{
		srcObj->sync();
		
		if(srcObj->attachments())
		{
			VMailAttribute *attr=(VMailAttribute *)ClientVFS::thisInstance()->_fact_ref(sourcePath+".fattr", "vmailattribute");
			
			if(attr)
			{
				attr->sync();
				
				for(int i=0;i<attr->partCount;i++)
				{
					QString partID=sourcePath+"."+QString::number(i);
					
					PartObject *part=(PartObject *)ClientVFS::thisInstance()->_fact_ref(partID, "part");
					
					if(part)
					{
						part->sync();
						
						if(part->partType()=="attachment") attachObject(partID, part->filename(), part->mimeType(), part->encoding());
						
						delete part;
					}
				}
				
				delete attr;
			}
		}
	}
}

void ComposerWindow::loadFromFwdSource(const QStringList &sourcePathList)
{
	initBasicMessage("Fwd: ", "");
				
	// add attachments
	for(unsigned int i=0;i<sourcePathList.count();i++)
	{
		MailObject *srcObj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(sourcePathList[i], "mail");
		
		if(srcObj)
		{
			srcObj->sync();
			attachObject(sourcePathList[i]+"%rfc822message", srcObj->subject()+".msg", "message/rfc822", "8bit");
		}
	}

	// load message
	loadMessage();
}

void ComposerWindow::loadFromFwdSourceInline(const QString &sourcePath)
{
	MailObject *srcObj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(sourcePath, "mail");
	
	if(srcObj)
	{
		srcObj->sync();
		
		initBasicMessage(QString("Fwd: ")+srcObj->subject(), srcObj->text());

		copyAttachmentsFrom(sourcePath);		
	}
	
	loadMessage();
}

void ComposerWindow::loadFromFwdSourceQuoted(const QString &sourcePath)
{
	MailObject *srcObj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(sourcePath, "mail");
	
	if(srcObj)
	{
		srcObj->sync();
		
		QString text=srcObj->text();
		
		// quote text
		GlobalConfig->setGroup("Composer");
		QString quoteSeq=GlobalConfig->readEntry("Quote char");
		if(quoteSeq.isEmpty()) quoteSeq="> ";
		
		text=TextProcessor::quoteText(text, quoteSeq);
		
		initBasicMessage(QString("Fwd: ")+srcObj->subject(), text);

		copyAttachmentsFrom(sourcePath);		
	}
	
	loadMessage();
}

void ComposerWindow::loadFromFwdSourceAsAttachment(const QString &sourcePath)
{
	MailObject *srcObj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(sourcePath, "mail");
	
	if(srcObj)
	{
		srcObj->sync();
		
		initBasicMessage(QString("Fwd: ")+srcObj->subject(), "");

		// add attachment
		attachObject(sourcePath+"%rfc822message", srcObj->subject()+".msg", "message/rfc822", "8bit");
	}

	// load message
	loadMessage();
}

void ComposerWindow::loadFromRepSource(const QString &sourcePath, bool replyAll)
{
	// load text, subject, recipients, priority, extended status and scheduling from server
	MailObject *obj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(sourcePath, "mail");
	
	if(obj)
	{
		// load original text
		QString text;
		
		if(obj->hasText())
		{
			text=obj->text();
		}
		else if(obj->hasHTML())
		{
			text=obj->html();
			textHtmlConverter->html2text(text);
		}
		
		// quote text
		GlobalConfig->setGroup("Composer");
		QString quoteSeq=GlobalConfig->readEntry("Quote char");
		if(quoteSeq.isEmpty()) quoteSeq="> ";
		
		text=TextProcessor::quoteText(text, quoteSeq);
		
		messageDirty=true;
		if(text.length()) messageChanged=true;
				
		composerWidget->setText(text);

		// set subject		
		QString replyPrefix=GlobalConfig->readEntry("Reply prefix");
		if(!replyPrefix.contains("%1")) replyPrefix="Re: %1";
		composerWidget->setSubject(replyPrefix.arg(obj->subject()));

		// set account
		sourceAccount=obj->account();
				
		if(replyAll)
		{
			// get address pool
			QCString myEmail=(const char *)ClientRMI::ref()->getAccountFromAddress(sourceAccount);
			QCString myReplyTo=(const char *)ClientRMI::ref()->getAccountReplyToAddress(sourceAccount);
			
			AddressListClass origTo(obj->to());
			AddressClass origFrom(obj->from());;
			AddressListClass origCc(obj->cc());

			// map from + to => to, cc => cc except for our address
			QCString destTo=origFrom.toQCString() + ", ";
			
			for(int i=0;i<origTo.count();i++)
			{
				QCString toAdr=origTo[i].toQCString();
				if(toAdr!=myEmail && toAdr!=myReplyTo)
				{
					destTo.append(toAdr);
					destTo.append(", ");
				}
			}
			
			// remove trailing ", "
			if (static_cast<unsigned int>(destTo.findRev(", ")) == (destTo.length()-2))
				destTo.truncate(destTo.length()-2);

			QCString destCc;
			
			for (int i=0; i < origCc.count(); i++)
			{
				QCString ccAdr=origCc[i].toQCString();
				if (ccAdr!=myEmail && ccAdr!=myReplyTo)
				{
					destCc.append(ccAdr);
					destCc.append(", ");
				}
			}
			
			// remove trailing ", "
			if (static_cast<unsigned int>(destCc.findRev(", ")) == (destCc.length()-2))
				destCc.truncate(destCc.length()-2);

			// load them in the To: and Cc: composer fields
			composerWidget->setToAddresses(AddressListClass(destTo));
			composerWidget->setCcAddresses(AddressListClass(destCc));
		}
		else
		{
			composerWidget->setToAddresses(AddressListClass(obj->from().toQCString()));
		}
		
		// @todo: load the extended status in the status view
		
		// @todo: trigger a status update on the source object
	}
	else
	{
		QMessageBox::warning(this, "Aethera: Message does not exist", "The composer attempted to reply to a non-existent message.\n"
			"Probably the message you have referenced was deleted meanwhile.\n", QMessageBox::Ok, 0, 0);
	}
}

void ComposerWindow::saveAsDraft()
{
	// todo step 1
	
	// save message
	saveMessage();
	
	// if not in Drafts move to Drafts
	if(messageID.find("/Magellan/Mail/Drafts")!=0)
		messageID=BrokerInterface::thisInstance()->move(messageID, "/Magellan/Mail/Drafts");
}

void ComposerWindow::saveMessage()
{
	// todo step 1

	if( messageDirty )
	{	
		// create object in Drafts if no address was set; set address
		if(messageID.isEmpty() || !BrokerInterface::thisInstance()->exists(messageID))
		{
			messageID=BrokerInterface::thisInstance()->createObject("/Magellan/Mail/Drafts");
			composerWidget->attachmentPane->setAddress(messageID);
		}

		MailObject *obj=(MailObject *)(ClientVFS::thisInstance()->_fact_ref(messageID, "mail"));

		if(obj)
		{
			// set text part, subject, recipients, priority and scheduling information
			obj->setSubject(composerWidget->subject());
			obj->setTo(*composerWidget->toAddresses());
			obj->setCc(*composerWidget->ccAddresses());
			obj->setBcc(*composerWidget->bccAddresses());
			obj->setStatus("Not sent");
			obj->setPriority(priorityToString(messagePriority));
			obj->setScheduling(scheduling);

			int textPart=obj->messageText();
			QString partID;

			if( textPart==-1 )
			{
				// no text part exists, create one
				partID=BrokerInterface::thisInstance()->createObject(messageID);

				VPartAttribute *attr=(VPartAttribute *)ClientVFS::thisInstance()->_fact_ref(partID+".attr", "vpartattribute");

				if(!attr)
				{
					QMessageBox::warning(this, "Aethera: Could not retrieve text part", "Could not retrieve a text part in this message.\n"
							"This is an unexpected error and the composer cannot allow to further edit this message.", QMessageBox::Ok, 0, 0);
					forcedClose=true;
					close();
				}
				
				attr->sync();
				attr->mtype="text";
				attr->mimetype="text/plain";
				attr->encoding="7bit";
				attr->charset=getDefaultCharset();
				attr->save();
			}
			else
			{
				partID=messageID+"."+QString::number(textPart);
			}
			QCString _text=composerWidget->text().isEmpty()?"":(const char *)composerWidget->text();
			BrokerInterface::thisInstance()->setObjectData(partID, _text);

			obj->save();

			obj->updateViews();
			
			int saveTimeout=GlobalConfig->readNumEntry("Save timeout");
			if(saveTimeout==0) saveTimeout=3;
			saveTimer->changeInterval(saveTimeout*60*1000);
		}
		else
		{
			QMessageBox::warning(this, "Aethera: Could not save message", "Could not save message. The server reply was:\n"
					+BrokerInterface::thisInstance()->regetLastError(), QMessageBox::Ok, 0, 0);
		}
		
		messageDirty=false;
		QString subject=composerWidget->subject();
		if(subject.isEmpty()) subject="no subject";
		QWidget::setCaption("Aethera composer: "+subject);
	}
}

void ComposerWindow::saveToOutbox()
{
	// save message
	saveMessage();
	
	// move message to Outbox, if needed
	if(messageID.find("/Magellan/Mail/Outbox")!=0)
		messageID=BrokerInterface::thisInstance()->move(messageID, "/Magellan/Mail/Outbox");
}

void ComposerWindow::sendMessage()
{
	// todo step 1
	
	// save to Outbox
	saveToOutbox();
		
	// ask the broker to send this message
	if(!BrokerInterface::thisInstance()->send(messageID))
		QMessageBox::warning(this, "Aethera: Could not send message", "Could not send message to"
				" the outgoing spooler.\nReason given by server:\n"+
				BrokerInterface::thisInstance()->regetLastError(), QMessageBox::Ok, 0, 0);
}

void ComposerWindow::ackTextChanged()
{
	// todo step 1
	messageDirty=true;
	messageChanged=true;

	QString subject=composerWidget->subject();
	if(subject.isEmpty()) subject="no subject";
	QWidget::setCaption("Aethera composer: "+subject+" (modified)");
}

void ComposerWindow::ackAddressChanged()
{
	messageDirty=true;
}

void ComposerWindow::ackSubjectChanged(const QString &subject)
{
	messageDirty=true;
	
	QWidget::setCaption(QString("Aethera composer: ")+(subject.isEmpty()?QString("no subject"):subject)+" (modified)");
}

void ComposerWindow::addFileAttachment()
{
	messageDirty=true;
	saveMessage();
	composerWidget->attachmentPane->newObject();
}

void ComposerWindow::lowerPriority()
{
	setPriority(-1);
}

void ComposerWindow::higherPriority()
{
	setPriority(1);
}

void ComposerWindow::setPriority(int newPriority)
{
	priority->setItemChecked(MENU_NORMAL_PRIORITY+messagePriority, false);
	priority->setItemChecked(MENU_NORMAL_PRIORITY+newPriority, true);
	messagePriority=newPriority;
	messageDirty=true;
}

QString ComposerWindow::priorityToString(int p)
{
	if(p==-2) return "Lowest";
	else if(p==-1) return "Low";
	else if(p==1) return "High";
	else if(p==2) return "Highest";
	else return "Normal";
}

int ComposerWindow::priorityFromString(const QString &p)
{
	if(p=="Lowest") return -2;
	else if(p=="Low") return -1;
	else if(p=="High") return 1;
	else if(p=="Highest") return 2;
	else return 0;
}

void ComposerWindow::ackAutoSaveTimer()
{
	saveMessage();
	
	int saveTimeout=GlobalConfig->readNumEntry("Save timeout");
	if(saveTimeout==0) saveTimeout=3;
	saveTimer->changeInterval(saveTimeout*60*1000);
}

void ComposerWindow::cut()
{
}

void ComposerWindow::copy()
{
}

void ComposerWindow::paste()
{
}

void ComposerWindow::clear()
{
}

void ComposerWindow::quoteSelection()
{
}

void ComposerWindow::unquoteSelection()
{
}

void ComposerWindow::shiftLeft()
{
}

void ComposerWindow::shiftRight()
{
}

void ComposerWindow::insertHLine()
{
}

void ComposerWindow::toggleBulletMode()
{
}

void ComposerWindow::toggleNumberingMode()
{
}

void ComposerWindow::insertFile()
{
}

void ComposerWindow::saveAsFile()
{
}

void ComposerWindow::importFromFile()
{
}

void ComposerWindow::setScheduling()
{
}

void ComposerWindow::transferSignature()
{
}

void ComposerWindow::messageProperties()
{
}

void ComposerWindow::signatureManager()
{
}

void ComposerWindow::spellChecker()
{
}

void ComposerWindow::setTitle(const QString &/*title*/)
{
}

void ComposerWindow::updatePriorityTips()
{
}

void ComposerWindow::loadObjectAsAttachment(const QString &/*sourcePath*/)
{
}

ToolBarSeparator* ComposerWindow::toolSeparator(QWidget *parent, const char *name)
{
	return new ToolBarSeparator(parent, name);
}

const QString ComposerWindow::getDefaultCharset() const
{
  // the charset name
  QString mainName = QTextCodec::codecForLocale()->name();
  // remove all white spaces
  mainName.replace(QRegExp("\\s"), "");
  // find and return the KDE charset name which is MIME compatible
  return KGlobal::charsets()->name(KGlobal::charsets()->nameToID(mainName));
}

