/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

// QT
#include <qdatastream.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qcstring.h>
#include <qtl.h>

// KDE
//#include <ktmainwindow.h>

// Aethera
#include <config.h>
#include <mailer.h>
#include <indexclass.h>
#include <mailclasses.h>
#include <localmailfolder.h>
//#include <miscfunctions.h>
#include <mimecodec.h>
//#include <journal.h>
#include <texthtml.h>
#include <messagedevice.h>
#include <messagedescriptor.h>
#include <mimepart.h>
#include <servernotifier.h>
extern TextHtmlConvertor *textHtmlConverter;

MessageDevice::MessageDevice(IndexClass *index):_index(index)
{
}

int MessageDevice::getPartCount()
{
	return _index->getPartCount();
}

int MessageDevice::createPart()
{
	// create part
	MimePart *_part=new MimePart();
	_index->setMultipartOnly(true);
	int partIndex=_index->addPart(_part, false);
	
	ServerNotifier::thisInstance()->objectChanged(_index->getParentFolder()->name()+"/"+_index->getID());
	
	// save index
	_index->getParentFolder()->saveIndex(_index);
	
	return partIndex;
}

bool MessageDevice::deletePart(int partIndex)
{
	if(_index->removePart(partIndex))
	{
		_index->setMultipartOnly(true);
		_index->getParentFolder()->saveIndex(_index);
		
		ServerNotifier::thisInstance()->objectDeleted(_index->getParentFolder()->name()+"/"+_index->getID()+"."+QString::number(partIndex));
		ServerNotifier::thisInstance()->objectChanged(_index->getParentFolder()->name()+"/"+_index->getID());
		
		return true;
	}
	else
	{
		return false;
	}
}

bool MessageDevice::getPartData(int partIndex, QByteArray &data)
{
	data.truncate(0);
	if(partIndex<_index->getPartCount() && partIndex>=0)
	{
		MimePart *part=_index->getPartAt(partIndex);
		
		// debug
		// printf("messagedevice: reading part data (part %d, bytes %d, enc %s)\n", partIndex, part->length, (const char *)part->encoding);
		
		if(part->length>0)
		{
			QFile dataFile(_index->getDataFilename());
			data.resize(part->length+1);

			if(dataFile.open(IO_ReadOnly))
			{
				dataFile.at(part->offset);
				dataFile.readBlock(data.data(), part->length);
				dataFile.close();

				data[part->length]=0;
				data=MIMECodec::decode(data, part->encoding);

				return true;
			}
			else
			{
				data.truncate(0);
				
				// debug
				printf("messagedevice: could not open file\n");
				
				return false;
			}
		}
		else
		{
			// debug
			printf("messagedevice: empty part\n");
			
			data.truncate(0);
			return true;
		}
	}
	else
	{
		// debug
		printf("messagedevice: part out of scope (%d parts registered)\n", _index->getPartCount());
		
		return false;
	}
}

bool MessageDevice::setPartData(int partIndex, const QByteArray &data)
{
	if(partIndex<_index->getPartCount() && partIndex>=0)
	{
		MimePart *part=_index->getPartAt(partIndex);
		QFile dataFile(_index->getDataFilename());
		QByteArray newData=MIMECodec::encode(data, part->encoding);
		
		// debug
		printf("messagedevice: saving part data (part %d, bytes %d, enc %s)\n", partIndex, data.size(), (const char *)part->encoding);
		
		// debug
//		printf("messagedevice: encoded data:\n----\n%s\n-----\n", newData.data());
//		printf("messagedevice: decoded data:\n----\n%s\n-----\n", MIMECodec::decode(newData, part->encoding).data());
	
		if(dataFile.open(IO_WriteOnly|IO_Append))
		{
			part->offset=dataFile.at();
			dataFile.writeBlock(newData.data(), newData.size());
			dataFile.close();
			
			part->length=newData.size();
			
			_index->setMultipartOnly(true);
			_index->getParentFolder()->saveIndex(_index);
						
			return true;
		}
		else
		{
			// debug
			printf("messagedevice: warning, could not open data file\n");
			
			return false;
		}
	}
	else
	{
		// debug
		printf("messagedevice: warning, part out of scope (%d parts registered)\n", _index->getPartCount());
		
		return false;
	}
}

MimePart& MessageDevice::operator[](int partIndex)
{
	return *(_index->getPartAt(partIndex));
}

void MessageDevice::loadDescriptor(bool fast)
{
	if(_index->getDescriptorOffset()!=-1)
	{
		QFile dataFile(_index->getDescriptorFileName());
		if(dataFile.open(IO_ReadOnly))
		{
			dataFile.at(_index->getDescriptorOffset());
			QDataStream stream(&dataFile);
			_descriptor.setCoreLoad(fast);
			stream>>_descriptor;
			_descriptor.setCoreLoad(false); // better safe than sorry
			dataFile.close();
		}
	}
}

MessageDescriptor& MessageDevice::getDescriptor()
{
	return _descriptor;
}

void MessageDevice::saveDescriptor()
{
	QFile dataFile(_index->getDescriptorFileName());
	if(dataFile.open(IO_WriteOnly|IO_Append))
	{
		_index->setDescriptorOffset(dataFile.at());
		QDataStream stream(&dataFile);
		stream<<_descriptor;
		_index->setDescriptorLength(dataFile.at()-_index->getDescriptorOffset());
		dataFile.close();
		
		_index->getParentFolder()->saveIndex(_index);
	}
}

bool MessageDevice::hasAttachments()
{
	for(int i=0;i<_index->getPartCount();i++)
		if(_index->getPartAt(i)->type=="attachment")
		  return true;
	return false;
}

bool MessageDevice::hasText()
{
	return textPart();
}

bool MessageDevice::hasHtml()
{
	return htmlPart();
}

bool MessageDevice::textPart(IVList *textPartList)
{
  bool bFound = false;

  for(int i=0;i<_index->getPartCount();i++)
  {
	  MimePart *part=_index->getPartAt(i);

    if( ( part->type=="text" || part->type=="attachment" ) &&
        part->mimetype=="text/plain" )
    {
      // if it is a test or we realy need the parts list
      if( textPartList != NULL )
        textPartList->append(i);
      bFound=true;
    }
  }

  return(bFound);
}

int MessageDevice::messageText()
{
  for(int i=0;i<_index->getPartCount();i++)
  {
	  MimePart *part=_index->getPartAt(i);

    if( part->type=="text" && part->mimetype=="text/plain" )
    {
      return( i );
    }
  }
  return( -1 );
}

bool MessageDevice::htmlPart(IVList *htmlPartList)
{
  bool bFound = false;

  for(int i=0;i<_index->getPartCount();i++)
  {
	  MimePart *part=_index->getPartAt(i);

    if( ( part->type=="text" || part->type=="attachment" ) &&
        part->mimetype=="text/html" )
    {
      // if it is a test or we realy need the parts list
      if( htmlPartList != NULL )
        htmlPartList->append(i);
      bFound=true;
    }
  }

  return(bFound);
}

QCString MessageDevice::text(bool bAllTextParts)
{
  QCString txt;

  IVList textParts;

  if( textPart(&textParts) )
  {
    for(IVList::Iterator it=textParts.begin(); it!=textParts.end(); ++it)
    {
      QCString text_;
   	  MimePart *part=_index->getPartAt(*it);

      if( bAllTextParts==true || part->type!="attachment" )
      {
        getPartData( *it , text_);
        if( part->type=="attachment" )
          txt+="\n-------- Attachment Part ---------\n";
        txt+=(const char *)text_;
      }
    }
  }
  return txt;
}

QCString MessageDevice::html(bool bAllHTMLParts)
{
  QCString txt;
  QCString txtTemp;
  QCString htmlAttachment="\
<br>&nbsp;\
<p>-------- Attachment Part ---------\
<br>&nbsp;\
";
  IVList textParts;
  IVList htmlParts;
  IVList allParts;

  if( htmlPart(&htmlParts) || textPart(&textParts) )
  {
    allParts=textParts+htmlParts;
    qHeapSort( allParts );
    for(IVList::Iterator it=allParts.begin(); it!=allParts.end(); ++it)
    {
      QCString text_;
   	  MimePart *part=_index->getPartAt(*it);

   	  if( bAllHTMLParts==true || part->type!="attachment" )
   	  {
        getPartData( *it , text_);
        txtTemp=(const char*)text_;
        if( part->mimetype=="text/plain" )
        {
          if( part->type=="attachment" )
            txt+=htmlAttachment;
          textHtmlConverter->text2html(txtTemp);
        }
        else //html
        {
          if( part->type=="attachment" )
            txt+=htmlAttachment;
          textHtmlConverter->html2html(txtTemp);
        }
        txt+=txtTemp;
   	  }
    }
  }
  return txt;
}

QCString MessageDevice::rfc822Message()
{
	QCString message, line;
	loadDescriptor(false);
	
	if(_index->isMultipartOnly() && MailFolder::Drafts==_index->getParentFolder()->getFolderType())
	{
	  printf("\nBuilding the message from parts\n");
	  fflush(stdout);
	
		QDateTime currentDateTime=QDateTime::currentDateTime();
		
		// date
		message="Date: "+(QCString)DateClass(currentDateTime)+"\n";
		
		// to
		line="To: ";
		
		for(int i=0;i<_descriptor.to.count();i++)
		{
			line+=_descriptor.to[i];
			if(i<_descriptor.to.count()-1)
			{
				line+=", ";
				if(line.length()+_descriptor.to[i+1].length()>75) line+="\n\t";
			}
		}
		
		line+="\n";
		
		message+=line;
		
		// cc
		if(_descriptor.cc.count())
		{
			line="Cc: ";
			for(int i=0;i<_descriptor.cc.count();i++)
			{
				line+=_descriptor.cc[i];
				if(i<_descriptor.cc.count()-1)
				{
					line+=", ";
					if(line.length()+_descriptor.cc[i+1].length()>75) line+="\n\t";
				}
			}
			
			line+="\n";
			
			message+=line;
		}
		
		// from
		message+="From: "+(QCString)_descriptor.from+"\n";
			
		// reply-to
		if(!_descriptor.replyTo.isEmpty()) message+="Reply-To: "+_descriptor.replyTo+"\n";
		
		// subject
		message+="Subject: "+_descriptor.subject+"\n";
		
		// header-id
		message+="Message-ID: "+_descriptor.messageID+"\n";
		
		// in-reply-to
		if(!_descriptor.inReplyTo.isEmpty()) message+="In-Reply-To: "+_descriptor.inReplyTo+"\n";
		
		// references
		if(_descriptor.references.count()) message+="References: "+_descriptor.references.join("\t\n")+"\n";
		
		// priority
		if(!_descriptor.priority.isEmpty()) message+="Priority: "+_descriptor.priority+"\n";
				
		// x tags
		message+="X-Mailer: "+QCString(RELEASE_VERSION)+"\n";
		message+="X-Aethera-Generated: header (rfc822), reference id "+_descriptor.indexID+"\n";
		
		// MIME version
		message+="MIME-Version: 1.0\n";
		
		// content-type & content-transfer-encoding
		if(_index->getPartCount())
		{
			if(_index->getPartCount()==1)
			{
				MimePart *part=_index->getPartAt(0);
				
				message+="Content-Type: "+part->mimetype;
				if(!part->charset.isEmpty())
					message+="; charset="+part->charset;
				message+="\n";
				message+="Content-Transfer-Encoding: "+part->encoding+"\n\n";
				
				QFile mailFile(_index->getDataFilename());
				mailFile.open(IO_ReadOnly);
				mailFile.at(part->offset);
				
				char *buf=new char[part->length+1];
				mailFile.readBlock(buf, part->length);
				buf[part->length]=0;
				
				message.append(buf);
				delete buf;
				mailFile.close();
			}
			else
			{
				QString time=currentDateTime.toString();
				time=time.replace(QRegExp("\\s"), "_").lower();
				QCString boundary=QCString("-boundary");
				boundary+=(const char *)time;
				boundary+=_descriptor.indexID;
				message+="Content-Type: multipart/mixed;\n\tboundary=\""+boundary+"\"\n";
				
				for(int i=0;i<_index->getPartCount();i++)
				{
					MimePart *part=_index->getPartAt(i);
					
					message+="\n--"+boundary+"\n";
					
					line="Content-Type: ";
					line+=part->mimetype;
					
					if(part->type=="text" && !part->charset.isEmpty()) line+=QCString("; charset=")+part->charset;
					else if(part->type=="attachment")	
					{
						if(part->name.isEmpty()) part->name="attachment.dat";
									
						QCString attachmentName=QCString("; name=")+part->name;
						
						if((line.length()+attachmentName.length())>75) line+="\n\t";
									
						line+=attachmentName;
					}
					line+="\n";
					message+=line;
					
					message+="Content-Transfer-Encoding: "+part->encoding+"\n";
					
					if(part->type=="attachment")
					{
						line="Content-Disposition: attachment; ";
						QCString attachmentFilename="filename="+part->name+"\n";
						
						if((line.length()+attachmentFilename.length())>75) line+="\n\t";
						
						line+=attachmentFilename;
						
						line+="\n";

						message+=line;
					}
					else message+="\n";
					
					QFile mailFile(_index->getDataFilename());
					mailFile.open(IO_ReadOnly);
					mailFile.at(part->offset);

					char *buf=new char[part->length+1];
					mailFile.readBlock(buf, part->length);
					buf[part->length]=0;
					
					// printf("reading block:\n----\n%s\n-----\n", buf);
										
					message.append(buf);
					delete buf;
					mailFile.close();
				}
				
				message+="\n--"+boundary+"--\n";
			}
		}
	}
	else
	{
		QFile mailFile(_index->getDataFilename());
		QCString headerLine;
		int dataLength=_index->getUniblockLength();
		
		mailFile.open(IO_ReadOnly);
		mailFile.at(_index->getUniblockOffset());
		
		char *buf=new char[dataLength+1];
		mailFile.readBlock(buf, dataLength);
		buf[dataLength]=0;
		
		message=QCString(buf);
		
		delete buf;
		mailFile.close();
	}
	
	return message;
}

QCString MessageDevice::rfc822Header()
{
	QCString header;	
	loadDescriptor(false);
	
	if(_index->isMultipartOnly())
	{
		QDateTime currentDateTime=QDateTime::currentDateTime();
		
		// date
		header+="Date: "+(QCString)DateClass(currentDateTime)+"\n";
		
		// to
		header+="To: ";
		for(int i=0;i<_descriptor.to.count();i++)
		{
			header+=_descriptor.to[i];
			if(i==_descriptor.to.count()-1)
				header+="\n";
			else
				header+=",\n\t";
		}
		
		// cc
		if(_descriptor.cc.count())
		{
			header+="Cc: ";
			for(int i=0;i<_descriptor.cc.count();i++)
			{
				header+=_descriptor.cc[i];
				if(i==_descriptor.cc.count()-1)
					header+="\n";
				else
					header+=",\n\t";
			}
		}
		
		// from
		header+="From: "+(QCString)_descriptor.from+"\n";
			
		// reply-to
		if(!_descriptor.replyTo.isEmpty()) header+="Reply-To: "+_descriptor.replyTo+"\n";
		
		// subject
		header+="Subject: "+_descriptor.subject+"\n";
		
		// header-id
		header+="Message-ID: "+_descriptor.messageID+"\n";
		
		// in-reply-to
		if(!_descriptor.inReplyTo.isEmpty()) header+="In-Reply-To: "+_descriptor.inReplyTo+"\n";
		
		// references
		if(_descriptor.references.count()) header+="References: "+_descriptor.references.join("\t\n")+"\n";
		
		// priority
		if(!_descriptor.priority.isEmpty()) header+="Priority: "+_descriptor.priority+"\n";
				
		// x tags
		header+="X-Mailer: "+QCString(RELEASE_VERSION)+"\n";
		header+="X-Aethera-Generated: header (rfc822), reference id "+_descriptor.indexID+"\n";
		
		// MIME version
		header+="MIME-Version: 1.0\n";
		
		// content-type & content-transfer-encoding
		if(_index->getPartCount())
		{
			if(_index->getPartCount()==1)
			{
				MimePart *part=_index->getPartAt(0);
				
				header+="Content-Type: "+part->mimetype;
				if(!part->charset.isEmpty())
					header+="; charset="+part->charset;
				header+="\n";
				header+="Content-Transfer-Encoding: "+part->encoding+"\n";
			}
			else
			{
				QString time=currentDateTime.toString();
				time=time.replace(QRegExp("\\s"), "_").lower();
				QCString boundary=QCString("\"-boundary");
				boundary+=(const char *)time;
				boundary+=_descriptor.indexID+"\"";
				header+="Content-Type: multipart/mixed;\n\tboundary="+boundary+"\n";
			}
		}
	}
	else
	{
		QFile mailFile(_index->getDataFilename());
		QCString headerLine;
		int dataLength=_index->getUniblockLength();
		
		mailFile.open(IO_ReadOnly);
		mailFile.at(_index->getUniblockOffset());
		
		char *buf=new char[dataLength+1];
		mailFile.readBlock(buf, dataLength);
		buf[dataLength]=0;
		
		header=QCString(buf);
		
		delete buf;
		mailFile.close();
	}
	
	return header;
}

QByteArray MessageDevice::getEntryCacheBlock()
{
	QByteArray data;
	QDataStream stream(data, IO_WriteOnly);
	
	loadDescriptor(true);
	
	Q_UINT8 attachments=false;
	for(int i=0;i<_index->getPartCount() && !attachments;i++)
		if(_index->getPartAt(i)->type=="attachment")
			attachments=true;
	
	stream<<_descriptor.status;
	stream<<_descriptor.extendedStatus;
	stream<<_descriptor.flag;
	stream<<attachments;
	stream<<_descriptor.red;
	stream<<_descriptor.green;
	stream<<_descriptor.blue;
	stream<<_descriptor.from;
	stream<<_descriptor.to;
	stream<<_descriptor.subject;
	stream<<_descriptor.sentDate;
	stream<<_descriptor.receivedDate;
	stream<<_descriptor.scheduling;
	stream<<_descriptor.priority;
	stream<<_descriptor.parentIndexID;	
	stream<<_index->getSize();
	stream<<_descriptor.contentType;
	
	return data;
}


