/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <pop3handler.h>
#include <kconfig.h>
#include <qsocketnotifier.h>
#include <unistd.h>
#include <incomingauthstructure.h>
#include <accountmanager.h>
#include <accounts.h>
#include <pthread.h>
#include <pop3uidjar.h>
#include <sysmanager.h>
#include <iosynchandler.h>
#include <servernotifier.h>
#ifdef OS_LINUX
  #include <asm/errno.h>
#else
  #ifdef OS_FREEBSD
    #include <errno.h>
  #endif
#endif
#include <stdio.h>

extern KConfig *GlobalConfig;

POP3Handler *POP3Handler::thisInstance;
int POP3Handler::rcvSync[2];
int POP3Handler::dlSync[2];
int POP3Handler::progressSync[2];
int POP3Handler::filterSync[2];
int POP3Handler::uidSync[2];
sem_t POP3Handler::pop3Lock;
sem_t POP3Handler::threadLock;
sem_t POP3Handler::stopThread;
int POP3Handler::dlResponse;
int POP3Handler::filterResponse;
IncomingAuthStructure *POP3Handler::incomingConnectionData;
struct IncomingMessage POP3Handler::incomingMessage;
struct DlData POP3Handler::dlData;
string POP3Handler::progressData;
string POP3Handler::filterData;

POP3Handler::POP3Handler():QObject()
{
	incomingConnectionData=0;
	
	timer=new QTimer(this);
	connect(timer, SIGNAL(timeout()), this, SLOT(__timerAck()));
	
	sem_init(&pop3Lock, 0, 1);
	sem_init(&threadLock, 0, 0);
	sem_init(&stopThread, 0, 0);

	pipe(rcvSync);
	QSocketNotifier *receiveNotifier=new QSocketNotifier(rcvSync[0], QSocketNotifier::Read, this);
	connect(receiveNotifier, SIGNAL(activated(int)), this, SLOT(__rcvAck(int)));
	
	pipe(dlSync);
	QSocketNotifier *downloadNotifier=new QSocketNotifier(dlSync[0], QSocketNotifier::Read, this);
	connect(downloadNotifier, SIGNAL(activated(int)), this, SLOT(__dlAck(int)));
	
	pipe(progressSync);
	QSocketNotifier *progressNotifier=new QSocketNotifier(progressSync[0], QSocketNotifier::Read, this);
	connect(progressNotifier, SIGNAL(activated(int)), this, SLOT(__progressAck(int)));

	pipe(filterSync);
	QSocketNotifier *filterNotifier=new QSocketNotifier(filterSync[0], QSocketNotifier::Read, this);
	connect(filterNotifier, SIGNAL(activated(int)), this, SLOT(__filterAck(int)));
	
	pipe(uidSync);
	QSocketNotifier *uidNotifier=new QSocketNotifier(uidSync[0], QSocketNotifier::Read, this);
	connect(uidNotifier, SIGNAL(activated(int)), this, SLOT(__uidAck(int)));
		
	readConfiguration();
}

POP3Handler *POP3Handler::ref()
{
	return thisInstance?thisInstance:(thisInstance=new POP3Handler());
}

void POP3Handler::readConfiguration()
{
	GlobalConfig->setGroup("Receive options");
	int interval=GlobalConfig->readNumEntry("Interval");
	if(interval<600) interval=600;
	timer->changeInterval(interval*1000);
	
	// load uids
	if(!POP3_UIDJar::ref())
	{
		printf("pop3handler: fatal: uidjar could not be initialized\n");
		exit(-1);
	}
}

void POP3Handler::getMail(Account *acc)
{
	// check if another session is in progress
	if(sem_trywait(&pop3Lock))
	{
		printf("pop3handler: another session is in progress, try again later\n");
		return;
	}
	
	printf("pop3handler: no session is in progress, proceeding\n");
	
	// get default account if necessary
	if(!acc) acc=AccountManager::ref()->getDefaultAccount();
	
	// stop here if there are no accounts available
	if(!acc)
	{
		printf("pop3handler: cannot proceed, no accounts are available\n");
		return;
	}
	
	// return if get temporary disabled or the type differs from POP3
	if(acc->getdisabled || acc->type!=Account::POP3)
	{
		printf("pop3handler: cannot proceed, account (%s) temporarely disabled or of wrong type\n", (const char *)acc->name);
		return;
	}

	ServerNotifier::thisInstance()->ioProgress(QString("pop3:task:")+acc->accname);
		
	// clear abort signal
	sem_trywait(&stopThread);
	
	// get the incoming connection data
	if(incomingConnectionData) delete incomingConnectionData;
	incomingConnectionData=new IncomingAuthStructure(acc);

	// set account flag
	incomingMessage.account=(const char *)acc->accname;
		
	// start thread
	pthread_t thread;
  pthread_attr_t attr;
  pthread_attr_init(&attr);
  pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
  pthread_create(&thread, &attr, pop3_thread, 0);
}

void POP3Handler::stopDelivery()
{
	sem_post(&stopThread);
}

void POP3Handler::__timerAck()
{
	getMail();
}

void POP3Handler::__rcvAck(int)
{
	// debug
	printf("pop3handler: message received, saving...\n");
	
	// empty the rcv sync pipe
	static char buf;
	::read(rcvSync[0], &buf, 1);
	
	// debug
	printf("pop3handler: sending message to the iosynchandler...\n");
	
	// get message and pass it to the IOSyncHandler
	IOSyncHandler::ref()->dispatchToInbox(incomingMessage);
	
	// debug
	printf("pop3handler: message saved.\n");
	
	// unlock thread
	sem_post(&threadLock);
}

void POP3Handler::__dlAck(int)
{
	// empty the dl sync pipe
	char buf;
	read(dlSync[0], &buf, 1);
	
	// ask user about action to perform

	// set option to be read by thread
		
	// unlock thread
	sem_post(&threadLock);
}

void POP3Handler::__progressAck(int)
{
	// debug
	// printf("pop3handler: thread progress acknowledged\n");
	
	// empty the progress sync pipe
	static char buf;
	::read(progressSync[0], &buf, 1);
	
	// sync the messages
	if( !progressData.compare("pop3:completed") )
	  IOSyncHandler::ref()->pop3Sync();
	
	// send the progress info to the notifier
	ServerNotifier::thisInstance()->ioProgress(QString(progressData.c_str()));
	
	// debug
	// printf("pop3handler: got progress data: %s\n", progressData.c_str());
	
	// unlock thread
	sem_post(&threadLock);
}

void POP3Handler::__filterAck(int)
{
	// empty the filter sync pipe
	static char buf;
	::read(filterSync[0], &buf, 1);
	
	// send the filter data and get a decision
//	filterResponse=SysManager::ref()->getServerFilterChain().action(filterData);
	
	// unlock thread
	sem_post(&threadLock);
}

void POP3Handler::__uidAck(int)
{
	// empty the uid sync pipe
	static char buf;
	::read(uidSync[0], &buf, 1);
	
	// debug
	printf("pop3handler: asking uidjar to save the lists...\n");
	
	// save lists
	POP3_UIDJar::ref()->saveLists();
	
	// unlock thread
	sem_post(&threadLock);
}

