/***************************************************************************
          imap4handler.h  -  the handler for the IMAPv4 protocol
                             -------------------
    begin                : Wed Feb  8 14:02:00 EET 2001
    copyright            : (C) 2001 by theKompany (www.thekompany.com>
    author               : Eugen Constantinescu
    email                : eug@thekompany.com
 ***************************************************************************/

#ifndef IMAP4HANDLER_H
#define IMAP4HANDLER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <semaphore.h>
#include <string>
#include <qobject.h>
#include <qtimer.h>
#include <incomingauthstructure.h>
#include <remotemailfolder.h>
#include <netimap4.h>
#include <indexclass.h>
#include <accounts.h>
#include <commandsprocessor.h>

class IMAP4Handler : public QObject
{
	Q_OBJECT
	
	friend void imap4_signal_progress(string);
	friend void *imap4_thread(void *args);
	friend int processIMAPCommand();
  friend bool whatIsNew(NetIMAP4 *client);
  friend void showCommand(NetIMAP4 *client);

	public:
	  /** Enums.*/
		enum { Download=0, Delete, Ignore, AlwaysIgnore, Ask };
    enum IMAP_THREAD_COMMANDS { NO_COMMANDS=0, NOOP, SYNC,
                                ADD_MESSAGE, DELETE_MESSAGE, UPDATE_MESSAGE,
                                EXPUNGE, CLOSE, LOGOUT, LOGIN, SELECT,
                                CREATE_FOLDER, DELETE_FOLDER, SUBFOLDERS,
                                MOUNT, UNMOUNT};

    // static methods
		/** Return a reference to the IMAP handler class.*/
		static IMAP4Handler *ref();
    /** The remote folder reference.*/
    static RemoteMailFolder* imapFolder();
    /** Start the thread.*/
		static void startThread();
    /** Clean the command data buffer.*/
    static void cleanCommandData();
		
    // public methods
		IMAP4Handler();
	  /** Read the config file for setting the noop() timer.*/
		void readConfiguration();
		/** Stop the messages download.*/
		bool stopDelivery();
    /**
      * Mount the IMAP folder.
      * IMAP folders needs a mount action for starting the connection
      * and an unmount action for closing the connection.
      * In this way the connection will be open only when you are seeing
      * the IMAP folder.
      * A mount acction will read the folder content too.
      */
    bool mount(QString url, bool bSync=true);
    /** Unmount an IMAP folder, see mount.*/
    bool unmount(QString url);
    /** It searches for the subfolders.*/
    bool subfolders(QString account, QString mailbox, QString other);
    /** Sync.*/
    bool sync(QString account);
    /** Login. It doesn't perform select.*/
    bool login(QString account);
    /** Logout.*/
    bool logout(QString account);
    /** Update the message contents.*/
    bool updateMessage(IndexClass &index);
    /** Add a message to the mailbox.*/
    bool addMessage(QString url, IndexClass *index);
    /** Check for the new mails.*/
    bool getMail(Account*);
    /** Delete a message from the mailbox.*/
    bool deleteMessage(IndexClass &index);
    /** Keep the connection alive.*/
    bool noop(QString account);
    /** Create a new mailbox on the IMAP server.*/
    bool createFolder(QString account, QString mailbox, QString url);
    /** Delete a mailbox from the IMAP server.*/
    bool deleteFolder(QString account, QString mailbox, QString url);

    // Public data
		/** The data about the received message.*/
		static IMAP_MESSAGE incomingMessage;
		/** The subfolders list.*/
    static MboxList subfoldersList;
		
		/** The list commands.*/
		CommandProcessor commandsList;
		
	private slots:
		bool __timerAck();
		void __rcvAck(int);
		void __progressAck(int);
		void __commandAck(int);
		void __updateAck(int);
		void __logout(int);
		void __subfolders(int);
		void __authSync(int);
		void __createFolder(int);
		void __deleteFolder(int);
		void __renameFolder(int);
		void __checkList(int);
		
	private:
		// LOCKS
		/** The general imap lock.*/
		static sem_t imap4Lock;
		/** A lock for imap thread.*/
		static sem_t threadLock;
		/** A lock for end of thread sync.*/
		static sem_t stopThread;
		/** A lock for commands sync.*/
		static sem_t imap4Command;
		
		// PIPES
		/** The pipe for receive mail syncronisation.*/
		static int rcvSync[2];
		/** The pipe for progress bar syncronisation.*/
		static int progressSync[2];
		/** The pipe for processing command syncronisation.*/
		static int commandSync[2];
		/** The pipe for updating the messages.*/
		static int updateSync[2];
		/** The pipe for logout signal.*/
		static int logoutSignal[2];
		/** The pipe for subfolders syncronisation.*/
		static int subfoldersSync[2];
		/** The pipe for auth syncronisation.*/
		static int authSync[2];
		/** The pipe for creating the folders.*/
		static int createSync[2];
		/** The pipe for deleting the folders.*/
		static int deleteSync[2];
		/** The pipe for renaming the folders.*/
		static int renameSync[2];
		/** Check if the command list is empty.*/
		static int checkList[2];
		
		// Attributes
		/** IMAP handler is a single ton object.*/
		static IMAP4Handler *thisInstance;
		/** The authenticate data.*/
		static IncomingAuthStructure *incomingConnectionData;
		/** The data used for progress bar syncronisation.*/
		static string progressData;
		/** The data used for command syncronisation.*/
		static COMMAND_STRUCT commandData;
		/** Flag for command list status.*/
		static bool emptyList;
	  /** A timer for checking interval.*/
		QTimer *timer;
};

#endif
















