/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */


/*

Note: performance on a PIII/450Mhz: it parsed a 30Mb mailbox containing 4035 messages in approx. 9 seconds.

*/

#include <headerclass.h>
#include <addressclass.h>
#include <addresslistclass.h>
#include <dateclass.h>
#include <qtextstream.h>
#include <qcstring.h>
#include <qstringlist.h>
#include <qstrlist.h>
#include <qvaluelist.h>
#include <mimecodec.h>

HeaderClass::HeaderClass(const char *c)
{
  if(!c) return;

  QCString msg(c), header, hline="a";

  // isolate and unfold the header
  QTextIStream htxt(msg);

  while(!hline.isEmpty() && !htxt.atEnd())
  {
    hline=(const char *)htxt.readLine();
    if(!hline.isEmpty())
    {
      if(!(QChar(hline[0]).isSpace()))
        header.append("\n");
      else
        header.append(" ");

      stripSpaces(hline);

      header.append(hline);
    }
  }

  // process the header
  QCString field, value, sender, resent, xpriority, inrep="", ref="";
  QTextStream txt(header, IO_ReadOnly);

  while(!txt.atEnd())
  {
    hline=txt.readLine();
    int pos=hline.find(":");
    if(pos!=-1)
    {
      field=hline.mid(0, pos).lower();
      value=hline.mid(pos+2);
      stripSpaces(value); // this shouldn't be necessary, since more than one space is a protocol violation - but who knows..
      stripComments(value);

      // check for known fields
      if(field=="received")
        Received.append(value);
      else if(field=="date")
        Date=DateClass(value);
      else if(field=="from")
        From=AddressClass(value);
      else if(field=="reply-to")
        Reply_To=AddressClass(value);
      else if(field=="to")
        To=AddressListClass(value);
      else if(field=="cc")
        Cc=AddressListClass(value);
      else if(field=="bcc")
        Bcc=AddressListClass(value);
      else if(field=="subject")
        Subject=MIMECodec::translate(value);
      else if(field=="message-id")
        Message_ID=value;
      else if(field=="in-reply-to")
        inrep=value;
      else if(field=="organization")
        Organization=value;
      else if(field=="status")
        Status=value;
      else if(field=="priority")
        Priority=value;
      else if(field=="content-type")
        Content_Type=value;
      else if(field=="content-transfer-encoding")
        Content_Transfer_Encoding=value;
      else if(field=="references")
        ref=value;
      else if(field=="x-priority")
        xpriority=value;
      else if(field=="sender")
        sender=value;
      else if(field=="resent-from")
        resent=value;
    }
  }

  // strip "<>" from Message_ID
  stripParanthesis(Message_ID);

  // process In_Reply_To and References
  QStrList tlist;

  if(!inrep.isEmpty())
  {
    tlist=chunk(inrep);
    if(tlist.count())
      In_Reply_To=tlist.at(0);
  }

  if(!ref.isEmpty())
  {
    References=chunk(ref);
    for(unsigned i=1;i<tlist.count();i++)
      References.append(tlist.at(i));
  }
  if(!In_Reply_To.isEmpty() && References.count())
    while(References.remove((const char *)In_Reply_To));

  // assign sensible values to Status
  if(Status=="O" || Status=="R")
    Status="Read";
  else
    Status="New";

  // parse the "Content-Type" structured field
  QStrList tokens=split(';', Content_Type);

  Content_Type=tokens.at(0);

  // check some redundant values
  if(!(QCString)From && !sender.isEmpty())
    From=AddressClass(sender);
  if(!(QCString)From && !resent.isEmpty())
    From=AddressClass(resent);
  if(!Priority && !xpriority.isEmpty())
    Priority=xpriority;

  // check and beautify the priority value, as some mailers use numbers
  int pr=Priority.toInt();
  if(pr>0 && pr<6)
    switch(pr)
    {
      case 5:
        Priority="Lowest";
        break;
      case 4:
        Priority="Low";
        break;
      case 3:
        Priority="Normal";
        break;
      case 2:
        Priority="High";
        break;
      case 1:
        Priority="Highest";
    }
  else
  {
    Priority=Priority.lower();
    if(Priority!="lowest" && Priority!="low" && Priority!="normal" && Priority!="high" && Priority!="highest")
      Priority="normal";
    QString k=QChar(Priority[0]);
    k=k.upper();
    Priority.replace(0, 1, (const char *)k);
  }

  // give some default values for the very important fields
  QCString def="unknown", defd="Mon, 1 Jan 1999 0:0:0 UTF";
  if(!(QCString)From)
    From=AddressClass(def);
  if(!(QCString)To)
    To=AddressListClass(def);
  if(!Subject)
    Subject="Not specified";
  if(!Date.day_of_week)
    Date=DateClass(defd);
  if(!Content_Type)
    Content_Type="text/plain"; // this is needed, for MIME-1.0 compliance
  if(!Content_Transfer_Encoding)
    Content_Transfer_Encoding="8bit"; // so is this
}

void HeaderClass::stripQuotes(QCString &s)
{
  if(s.length() && s[0]=='"')
    s.remove(0, 1);
  if(s.length() && s[s.length()-1]=='"')
    s.truncate(s.length()-1);
}

void HeaderClass::stripParanthesis(QCString &s)
{
  if(s.length() && s[0]=='<')
    s.remove(0, 1);
  if(s.length() && s[s.length()-1]=='>')
    s.truncate(s.length()-1);
}

void HeaderClass::stripSpaces(QCString &s)
{
  char c;
  while(s.length() && ((c=s[0])==' ' || c=='\t' || c=='\n'))
    s.remove(0, 1);
  while(s.length() && ((c=s[s.length()-1])==' ' || c=='\t' || c=='\n'))
    s.truncate(s.length()-1);
}

void HeaderClass::stripComments(QCString &s)
{
  int nesting=0;
  bool qnesting=false;
  QCString t;
  for(unsigned i=0;i<s.length();i++)
  {
    char c=s[i];
    if(c=='"' && !nesting) qnesting=!qnesting;
    if(c=='(' && !qnesting)
      nesting++;
    if(nesting==0 || qnesting) t+=c;
    if(c==')' && nesting && !qnesting)
      nesting--;
  }
  s=t;
}

QStrList HeaderClass::split(char c, QCString &s)
{
  QStrList t;
  QCString token;
  for(unsigned i=0;i<s.length();i++)
  {
    if(s[i]==c)
    {
      stripSpaces(token);
      t.append(token);
      token.truncate(0);
    }
    else
      token+=s[i];
  }
  if(!token.isEmpty())
  {
    stripSpaces(token);
    t.append(token);
  }
  return t;
}

QStrList HeaderClass::chunk(QCString &c)
{
  QStrList t;
  QCString token;

  for(unsigned i=0;i<c.length();i++)
  {
    if(c[i]=='>' && !token.isEmpty())
    {
      t.append(token);
      token.truncate(0);
    }
    else if(c[i]=='<')
      token.truncate(0);
    else
      token+=c[i];
  }

  return t;
}

void HeaderClass::importMessage(MessageClass &m)
{
}

HeaderClass::HeaderClass(MessageClass &m)
{
	importMessage(m);
}





