/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#ifndef __ClientVFS__
#define __ClientVFS__

#include <vfolder.h>
#include <vobject.h>
#include <vreference.h>
#include <qdict.h>
#include <vobjectfactory.h>
#include <qlist.h>
#include <qstringlist.h>
#include <stdio.h>
#include <clientnotifier.h>

class ClientVFS
{
	friend class ClientNotifier;
	public:
		ClientVFS();
		~ClientVFS();
		static ClientVFS *thisInstance();

		bool initHierarchy();
		bool createVInstances();
		bool rehashHierarchy();
		bool rehashVInstances();

		QDict<VFolder> folderDict();
		QString parentFolder(QString url);
		VFolder *parentFolder(VFolder *);
		QStringList folderList(QString folder); // this implementation guarantees the existence of "/Magellan"
		QStringList entryList(QString folder);
		
		// methods controlling the behaviour of the ClientVFS object
		void autogenTemporary(bool temp);
		void useFactories(bool fact);
		void autogenVInstances(bool vinst, bool nonreg);
		
		// references - all VReferences are always valid if they exist somewhere in Magellan
		// since they register automatically with ClientVFS
		
		// a "temporary://" (always local!) virtual folder will be automatically generated if a reference is requested and the folder doesn't exist, unless disabled by autogenTemporary()
		// this is the only situation where it is allowed to use an url (otherwise ClientVFS woulnd't know a temp folder was requested)
		// if only "temporary" is passed as a parameter a unique name will be automatically generated
		// temporary folders are registered under the root folder (!), but they have no associated binding or server path
		// temporary folders can have only VTemporary type entries, and they are NOT handled by BrokerInterface (VTemporary objects are)
		VFolder *folderRef(QString folder);
		VFolder *folderRef(QString parent, QString name);
		VFolder *folderRef(VFolder *parent, QString name);
		// object references work here exclusively on folder/object relationships
		// don't use logical parents, subobjects, attributes, related lists etc. as the VFS map doesn't include such run-time generated data
		// use the BrokerInteface instead, if needed
		// March 13,2000 - disregard the previous three lines; in order to optimize the query processor
		// I've added object registration for all VObjects
		VObject *objectRef(QString object);
		VObject *objectRef(VFolder *parent, QString id);
		VObject *objectRef(QString folder, QString id);		
		// general reference
		VReference *reference(QString url);

		// init folder cache
		void initFolderCache(const QString &type);
		void discardFolderCache(const QString &type);
				
		// register VObject factories
		void registerFactory(QString type, VObjectFactory *factory);

		// this makes a new folder
		VFolder *_new(QString path, QString type, QString viewType, QString mimeType, QString binding=QString::null, QString spath=QString::null);
		
		VReference *_fact_new(QString path, QString type);
		VReference *_fact_ref(QString path, QString type);
		
		// template VObject factories
		// use _new to create a new VObject; this returns 0 if you pass an undefinable path (i.e. which is
		// not defined on the server and it is not temporary) or if an object with that address already exists
		template <class T>
		T *_new(QString url)
		{
      // debug
      // printf("ClientVFS::_new called, path %s\n", (const char *)url);
      // printf("can define: %s\n", canDefine(url)?"yes":"no");
      // printf("existent reference: %p\n", reference(url));

			if(canDefine(url) && !reference(url))
			  return new T(url);
			return 0;
		}
		
		// template VObject factories
		// use _new to create a new VObject; this returns 0 if you pass an undefinable path (i.e. which is
		// not defined on the server and it is not temporary) or if an object with that address already exists
		template <class T>
		T *_new(QString url, QByteArray &data)
		{
      // debug
      // printf("ClientVFS::_new called, path %s\n", (const char *)url);
      // printf("can define: %s\n", canDefine(url)?"yes":"no");
      // printf("existent reference: %p\n", reference(url));

//			if(canDefine(url) && !reference(url)) return new T(url, data);
      // I disable the checking for speeding up the objects loading - Eugen C.
			return new T(url, data);
			return 0;
		}
		
		// this works like new, but it will attempt to find and return the existing implementation first
		template <class T>
		T *_ref(QString url)
		{
			if(VReference *r=reference(url)) return (T*)r;
			if(canDefine(url)) return new T(url);
			return 0;
		}
		
		bool canDefine(QString url);
						
		// generate temporary folder name
		QString genTempFolderName();
		
		// navigation
		QString pwd();
		QStringList dir(QString params=QString::null);
		bool cd(QString path);
		void cdUp();
		void cdRoot();
		void pushPath();
		void popPath();
		QString root();
				
		bool createVInstances(VFolder *folder); // use this with caution since it can cause major structure disruption; normally, it shouldn't be called manually
		
		void saveStructure();
		void saveFolder(VFolder *folder);
		
		// maps a server address on a client address, or returns QString::null if such a mapping does not exist
		/** sever url -> client url.*/
		QString reverseMap(QString path);
		/** sever urls -> client urls.*/
		QStringList reverseMap(QStringList list);
		
		// maps a client address on a server address
		/** client url -> server url.*/
		QString map(QString url);
		/** client urls -> server urls.*/
		QStringList map(QStringList urlList);
		
		// create and delete folders
		/** Create a folder from the client.*/
		VFolder *createFolder(VFolder *parent, const QString &name, const QString &type, const QString &viewType);
		/** Create a new folder notified by the server.*/
    void createFolder(const QString &parentFolder, const QString &name, const QString &type, const QString &viewType, const QString &serverPath=QString::null);
    /** Delete the folder.*/
		bool deleteFolder(const QString &name);
		/** Delete a folder notified by the server.*/
    void deleteFolder(const QString &parent, const QString &folderName, const QString &type);
		/** Remove the VFS folder.*/
		void deleteVFolder(const QString &name);
		/** Remove the VFS folder.*/
		void deleteVFolder(VFolder *folder);
		/** Move the folder.*/
		bool moveFolder(const QString &path, const QString &newPath);
		
	private:	
		bool makeBranch(QString folderName, QString path);
		void saveFolderBranch(VFolder *folder);
		
		void updateVInstance(QString path);
		void updateVInstances(QStringList &path);
		void deleteVInstance(QString path);
		void createVInstance(QString path);
		bool createVInstances(QStringList &paths);
			
		QDict<VObjectFactory> factoryDict;
		QDict<VFolder> folders;
		QString currentPath;
		QStringList pStack;
		bool autogenTemp, useFact, genVInstances, genNonRegistered;
};

#endif









