#ifndef _3DS_H
#define _3DS_H

#include <iostream>
#include <stdio.h>
#include <stdlib.h>

# include <string>
# include <vector>

#include "render.h"

//>------ Primary Chunk, at the beginning of each file
#define PRIMARY       0x4D4D

//>------ Main Chunks
#define OBJECTINFO    0x3D3D                // This gives the version of the mesh and is found right before the material and object information
#define SCOURGE_3DS_VERSION       0x0002                // This gives the version of the .3ds file
#define EDITKEYFRAME  0xB000                // This is the header for all of the key frame info

//>------ sub defines of OBJECTINFO
#define MATERIAL      0xAFFF                // This stored the texture info
#define OBJECT        0x4000                // This stores the faces, vertices, etc...

//>------ sub defines of MATERIAL
#define MATNAME       0xA000                // This holds the material name
#define MATDIFFUSE    0xA020                // This holds the color of the object/material
#define MATMAP        0xA200                // This is a header for a new material
#define MATMAPFILE    0xA300                // This holds the file name of the texture

#define OBJECT_MESH   0x4100                // This lets us know that we are reading a new object

//>------ sub defines of OBJECT_MESH
#define OBJECT_VERTICES     0x4110          // The objects vertices
#define OBJECT_FACES        0x4120          // The objects faces
#define OBJECT_MATERIAL     0x4130          // This is found if the object has a material, either texture map or color
#define OBJECT_UV           0x4140          // The UV texture coordinates


// Here is our structure for our 3DS indicies (since .3DS stores 4 unsigned shorts)
struct tIndices {                           

    unsigned short a, b, c, bVisible;       // This will hold point1, 2, and 3 index's into the vertex array plus a visible flag
};

// This holds the chunk info
struct tChunk
{
    unsigned short int ID;                  // The chunk's ID       
    unsigned int length;                    // The length of the chunk
    unsigned int bytesRead;                 // The amount of bytes read within that chunk
};

// This class handles all of the loading code
class CLoad3DS
{
public:
    CLoad3DS();                             // This inits the data members

    // This is the function that you call to load the 3DS
    bool Import3DS(t3DModel *pModel, char *strFileName);

private:
    // This reads in a string and saves it in the char array passed in
    int GetString(char *);

    // This reads the next chunk
    void ReadChunk(tChunk *);

    // This reads the next large chunk
    void ProcessNextChunk(t3DModel *pModel, tChunk *);

    // This reads the object chunks
    void ProcessNextObjectChunk(t3DModel *pModel, t3DObject *pObject, tChunk *);

    // This reads the material chunks
    void ProcessNextMaterialChunk(t3DModel *pModel, tChunk *);

    // This reads the RGB value for the object's color
    void ReadColorChunk(tMaterialInfo *pMaterial, tChunk *pChunk);

    // This reads the objects vertices
    void ReadVertices(t3DObject *pObject, tChunk *);

    // This reads the objects face information
    void ReadVertexIndices(t3DObject *pObject, tChunk *);

    // This reads the texture coodinates of the object
    void ReadUVCoordinates(t3DObject *pObject, tChunk *);

    // This reads in the material name assigned to the object and sets the materialID
    void ReadObjectMaterial(t3DModel *pModel, t3DObject *pObject, tChunk *pPreviousChunk);
    
    // This computes the vertex normals for the object (used for lighting)
    //void ComputeNormals(t3DModel *pModel);

    // This frees memory and closes the file
    void CleanUp();
    
    // The file pointer
    FILE *m_FilePointer;
    
    // These are used through the loading process to hold the chunk information
    tChunk *m_CurrentChunk;
    tChunk *m_TempChunk;
};


#endif


/////////////////////////////////////////////////////////////////////////////////
//
// * QUICK NOTES * 
// 
// This file is created in the hopes that you can just plug it into your code
// easily.  You will probably want to query more chunks though for animation, etc..
//
// 
// Ben Humphrey (DigiBen)
// Game Programmer
// DigiBen@GameTutorials.com
// Co-Web Host of www.GameTutorials.com
//
//
