/*
 * Pspell interface header file
 * Copyright 2000 by Kevin Atkinson under the terms of the LGPL
 *
 */

#ifndef PSPELL__MANAGER
#define PSPELL__MANAGER

#include "error.h"
#include "config.h"
#include "wordlist.h"

#ifdef __cplusplus

//supported encodings

// iso8859-1
// ...: other 8 bit encoding specific to the language
// utf8
// machine unsigned 16: Machine specific unsigned 16 bit wide characters
// machine unsigned 32: Machine specific unsigned 32 bit wide characters

class PspellManager : public PspellCanHaveError
{
 public:

  virtual PspellConfig & config() = 0;

  ////////////////////////////////////////////////////////////////
  // 
  // Strings from this point on are expected to be in the 
  // encoding specified by encoding()
  //

  virtual int check(const char *) = 0;
  // returns  0 if it is not in the dictionary;
  //          1 if it is;
  //         -1 on error;
  
  virtual bool add_to_personal(const char *) = 0;
  virtual bool add_to_session(const char *)  = 0;
  // returns false if the operation fails
    
  // because the word lists may potently have to convert from one
  // encoding to another the pointer returned by the emulation is only
  // valid to the next call.

  virtual const PspellWordList * master_word_list()   = 0;
  virtual const PspellWordList * personal_word_list() = 0;
  virtual const PspellWordList * session_word_list()  = 0;
  // returns null on error
  
  virtual bool save_all_word_lists() = 0;
  
  virtual bool clear_session() = 0;

  virtual const PspellWordList * suggest(const char *) = 0;
  // return null on error
  // the word list returned by suggest is only valid until the next
  // call to suggest
  
  virtual bool store_replacement(const char * mis, const char * cor) = 0;

  virtual ~PspellManager() {}
};

#else

typedef struct PspellManager PspellManager;

#endif

#ifdef __cplusplus
extern "C" {
#endif

  PspellCanHaveError * new_pspell_manager(PspellConfig * config);
  /* returns a new manager class based on the settings in config  
   * It is allocated with new however you SHOULD NOT just delete
   * it as that will not probally release the shared library
   * linked in
   */

  PspellManager * to_pspell_manager(PspellCanHaveError *);
  /* casts a PspellCanHaveError to a PspellManager.  Do not cast
   * directly in C as it might not do it correctly.
   */

  void delete_pspell_manager(PspellManager *);
  /* deletes a PspellManager class and releases the shared library */


  int pspell_manager_error_number(const PspellManager *);
  
  const char * pspell_manager_error_message(const PspellManager *);

  const PspellErrorExtraInfo * 
    pspell_manager_error_extra_info(const PspellManager *);


  PspellConfig * pspell_manager_config(PspellManager *);

  int pspell_manager_check(PspellManager *, const char *);
  
  int pspell_manager_add_to_personal(PspellManager *, const char *);
  int pspell_manager_add_to_session(PspellManager *, const char *);

  const PspellWordList * pspell_manager_master_word_list(PspellManager *);   
  const PspellWordList * pspell_manager_personal_word_list(PspellManager *); 
  const PspellWordList * pspell_manager_session_word_list(PspellManager *); 
  
  int pspell_manager_save_all_word_lists(PspellManager *); 
  
  int pspell_manager_clear_session(PspellManager *);

  const PspellWordList * pspell_manager_suggest(PspellManager *, const char *);
  
  int pspell_manager_store_replacement(PspellManager *, 
				       const char * mis, const char * cor);

#ifdef __cplusplus
}
#endif

#endif
