#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use Test::More tests => 11;
use File::Spec::Functions;

# Set up $ENV{XDG_CONFIG_HOME} and $ENV{XDG_DATA_HOME} to the working
# directory (the same directory as the test script) in '.config' and '.data'
$ENV{XDG_CONFIG_HOME} = catdir($ENV{PWD}, '.config');
$ENV{XDG_DATA_HOME} = catdir($ENV{PWD}, '.data');

### use VASM::Resource + ability test (VASM::Resource is not a class, and
### can therefore not use can_ok
BEGIN { 
  use_ok(
    'VASM::Resource', 
    qw/configResourceFind
       dataResourceFind
       configResourcePathCreate
       dataResourcePathCreate
       configResourceCreate
       dataResourceCreate/) 
}

### configResourcePathCreate and dataResourcePathCreate
my $newConfigPath = configResourcePathCreate(qw/some goofy/);
my $newDataPath = dataResourcePathCreate(qw/some other/);

# Check paths...too lazy to write this as a loop, and not really needed
is(
  $newConfigPath, catdir($ENV{XDG_CONFIG_HOME}, qw/vasm some goofy/),
  'Config path valid'
);
is(
  $newDataPath, catdir($ENV{XDG_DATA_HOME}, qw/vasm some other/),
  'Data path valid'
);

# Check permissions
is((stat($newConfigPath))[2] & 0700, 0700, 'Config path permissions correct');
is((stat($newDataPath))[2] & 0700, 0700, 'Data path permissions correct');

### configResourceCreate and dataResourceCreate
my $newConfigHandle = (configResourceCreate(qw/some goofy nonsense.txt/))[0];
my $newDataHandle = (dataResourceCreate(qw/some goofy nonsense.txt/))[0];

for my $fh ($newConfigHandle, $newDataHandle) {
  $fh->print("GRONK GRONK GRONK\n");
  $fh->close;
}

### configResourceFind and dataResourceFind
my @resources = (
  [ 'Config', configResourceFind(qw/some goofy nonsense.txt/) ],
  [ 'Data', dataResourceFind(qw/some goofy nonsense.txt/) ]
);

for my $resource (@resources) {
  # $resource is a IO::File
  isa_ok($resource->[1], 'IO::File', "$resource->[0] handle is an IO::File");
  # Verifying contents of (file represented by) $fh
  is(
    $resource->[1]->getline, "GRONK GRONK GRONK\n",
    "$resource->[0] handle contents valid"
  );
  # Check filenames
  is(
    $resource->[2],
    catfile(
      $ENV{'XDG_' . uc($resource->[0]) . '_HOME'},
      qw/vasm some goofy nonsense.txt/
    ), 
    "$resource->[0] resource filename valid"
  );
  # Close the filehandle
  $resource->[1]->close;
}
