#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { false => 0, true => 1 };
use VASM::Resource::Catalog::Message;
use VASM::Utility::FileSystem::User;
use VASM::Utility::FileSystem::Group;
use VASM::Utility::FileSystem::Which;
use File::Spec::Functions qw/rel2abs curdir splitpath/;
use UI::Dialog::Backend::CDialog;
use POSIX;

# Message files
my $commonMsg = messageCatalogFind(qw/common message/);
my $uiMsg = messageCatalogFind(qw/useradd ui cdialog/);
my $errorMsg = errorCatalogFind(qw/useradd ui error/);

# Dialog instance
my $d = UI::Dialog::Backend::CDialog->new(
  backtitle => $commonMsg->render('vasm backtitle'),
  title => $uiMsg->render('title'),
  'ok-label' => $commonMsg->render('ok'),
  'cancel-label' => $commonMsg->render('exit'),
  'yes-label' => $commonMsg->render('yes'),
  'no-label' => $commonMsg->render('no'),
  width => 75,
  height => 20,
  menuheight => 10
);

sub mainMenu {
  # Check for root privileges
  unless (getpwuid $> eq 'root') {
    $d->msgbox(text => $errorMsg->render('useradd for root only'));
    exit EXIT_FAILURE;
  }

  # Instantiate the group directory and menu choices
  my $groupDirectory = groupDirectory();
  my $groupChoices = [ 
    map {
      $groupDirectory->_recodeString($_), 
      [ $groupDirectory->render($_), false ]
    } sort grep {
      $_ ne 'users'
    } $groupDirectory->list
  ];

  my %userAttrs;
  DIALOG: {
    # Let the user know what's going on
    $d->msgbox(text => $uiMsg->render('useradd introduction'));
    # Ask for a username
    $userAttrs{userName} = userNameAsk();
    # Present groups menu
    $userAttrs{groups} = [ userGroupsAsk($groupChoices) ];
    # Ask comment
    $userAttrs{comment} = userCommentAsk();
    # Ask shell
    $userAttrs{shell} = userShellAsk();

    # Make sure all these figures are acceptable before moving on to the
    # adding the user
    redo DIALOG unless userAttrsOk(%userAttrs);

    # Add the user
    userAdd(%userAttrs);
    # Confirm this to the user
    $d->msgbox(text => $uiMsg->render('user added', $userAttrs{userName}));
    # Ask about setting up a password
    userPasswordConfigure($userAttrs{userName});
  }

  exit EXIT_SUCCESS;
}

sub userNameAsk {
  DIALOG: {
    my $userName = $d->inputbox(text => $uiMsg->render('user name prompt'));

    # Check validity
    $d->msgbox(text => $errorMsg->render('field blank')) and redo DIALOG
      unless length $userName;
    $d->msgbox(text => $errorMsg->render('field invalid')) and redo DIALOG
      if $userName =~ /[^[:alnum:]_-]/;
    $d->msgbox(text => $errorMsg->render('user name taken')) and redo DIALOG
      if userExists($userName);
  
    if ($d->state eq 'OK') {
      # As Easter Egg as an open-source program can be...
      $d->msgbox(text => $uiMsg->render('good choice'))
        if $userName eq 'hanumizzle';
      # Now, back to business:
      return $userName;
    } elsif ($d->state eq 'CANCEL') {
      exit EXIT_SUCCESS;
    }
  }
}

sub userGroupsAsk {
  my ($choices) = @_;

  my @groups = $d->checklist(
    text => $uiMsg->render('groups prompt'), list => $choices
  );
  
  if ($d->state eq 'OK') {
    return @groups;
  } elsif ($d->state eq 'CANCEL') {
    exit EXIT_SUCCESS;
  }
}

sub userCommentAsk {
  DIALOG: {
    my $comment = $d->inputbox(text => $uiMsg->render('comment prompt'));
  
    if ($d->state eq 'OK') {
      $d->msgbox(text => $errorMsg->render('field invalid')) and redo DIALOG
        if $comment =~ /[:']/;
      return $comment;
    } elsif ($d->state eq 'CANCEL') {
      exit EXIT_SUCCESS;
    }
  }
}

sub userShellAsk {
  my $shellEntry = '/bin/bash';
  
  DIALOG: {
    my $shell = $d->inputbox(
      'extra-button' => true, 'extra-label' => $uiMsg->render('find'),
      text => $uiMsg->render('shell prompt'), entry => $shellEntry
    );
  
    if ($d->state eq 'OK') {
      $d->msgbox(text => $errorMsg->render('field invalid')) and redo DIALOG
        if $shell =~ /[:']/;
      if (length $shell) {
        $d->msgbox(text => $errorMsg->render('shell unavailable'))
        and redo DIALOG
          unless defined which($shell);
      }
      return $shell;
    } elsif ($d->state eq 'EXTRA') {
      $shellEntry = userShellFind();
      redo DIALOG;
    } elsif ($d->state eq 'CANCEL') {
      exit EXIT_SUCCESS;
    }
  }
}

BEGIN {
  # userShellFind has its own 'static' variable for the current working
  # directory; see perlfaq7
  my $wd = rel2abs(curdir);

  sub userShellFind {
    # Get the filename of a shell; start at current directory
    my $shell = $d->fselect(path => $wd, height => 10);

    if ($d->state eq 'OK') {
      # Store new current working directory
      $wd = (splitpath($shell))[1];
      # Return shell
      return $shell;
    } elsif ($d->state eq 'CANCEL') {
      return q();
    }
  }
}

sub userAttrsOk {
  my (%userAttrs) = @_;

  return $d->yesno(
    text => $uiMsg->render(
      'user attributes acceptable?',
      map {
        $uiMsg->_recodeString($_)
      } (
        $userAttrs{userName},
        $userAttrs{comment},
        join(', ', 'users', @{ $userAttrs{groups} }),
        $userAttrs{shell}
      )
    )
  );
}

sub userPasswordConfigure {
  my ($userName) = @_;

  if ($d->yesno(text => $uiMsg->render('configure password?', $userName))) {
    my $vpasswd = whichVasm('vpasswd');

    if (defined $vpasswd) {
      exit system "$vpasswd $userName";
    } else {
      $d->msgbox(text => $errorMsg->render('vpasswd unavailable'));
      exit EXIT_FAILURE;
    }
  }
  
  return;
}

mainMenu();
