#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { false => 0, true => 1 };
use VASM::Resource::Catalog::Message;
use VASM::Utility::FileSystem::InitTab;
use VASM::Utility::FileSystem::Service;
use UI::Dialog::Backend::CDialog;
use POSIX;

# Message files
my $commonMsg = messageCatalogFind(qw/common message/);
my $uiMsg = messageCatalogFind(qw/srvset ui cdialog/);
my $errorMsg = errorCatalogFind(qw/srvset ui error/);

# Dialog instance
my $d = UI::Dialog::Backend::CDialog->new(
  backtitle => $commonMsg->render('vasm backtitle'),
  title => $uiMsg->render('title'),
  'ok-label' => $commonMsg->render('ok'),
  'cancel-label' => $commonMsg->render('exit'),
  'yes-label' => $commonMsg->render('yes'),
  'no-label' => $commonMsg->render('no'),
  width => 75,
  height => 20,
  menuheight => 10
);

sub bootLevelMenu {
  unless (getpwuid $> eq 'root') {
    $d->msgbox(text => $errorMsg->render('srvset for root only'));
    exit EXIT_FAILURE;
  }

  my $bootLevelDirectory = bootLevelDirectory();
  my $serviceDirectory = serviceDirectory();
  
  # Weed the single-user message out of the boot level directory, I don't need
  # it for this program
  $bootLevelDirectory->delete('1');

  # Choices for main menu -- constant
  my $choices = [
    map {
      $bootLevelDirectory->_recodeString($_), $bootLevelDirectory->render($_)
    } $bootLevelDirectory->list
  ];
  
  while (true) {
    # Prompt choice
    my $bootLevel = $d->menu(
      text => $uiMsg->render('boot level menu description'),
      list => $choices
    );

    if ($d->state eq 'OK') {
      # Configure services for that level
      serviceMenu($bootLevel, $serviceDirectory);
    } elsif ($d->state eq 'CANCEL') {
      exit EXIT_SUCCESS;
    }
  }

  return; # Won't actually get here
}

sub serviceMenu {
  my ($bootLevel, $serviceDirectory) = @_;

  # Determine which services are enabled/disabled
  my %serviceOnOffHash = serviceOnOffHash($bootLevel);

  # Choices for service menu -- not constant, regenerated every time function
  # is called
  my $choices = [
    map {
      $serviceDirectory->_recodeString($_), 
      [ 
        $serviceDirectory->render($_),
        ($serviceOnOffHash{$_} ? true : false)
      ]
    } grep {
      defined $serviceOnOffHash{$_}
    } $serviceDirectory->list
  ];

  my @servicesEnabled = $d->checklist(
    'extra-button' => true, 'extra-label' => $commonMsg->render('back'),
    text => $uiMsg->render('service menu description'), list => $choices
  );

  if ($d->state eq 'OK') {
    # Wipe out the hash and build it up from @servicesEnabled
    %serviceOnOffHash = map { $_ => false } keys %serviceOnOffHash;
    for my $service (@servicesEnabled) { $serviceOnOffHash{$service} = true }
    # Commit these changes
    serviceSet($bootLevel, %serviceOnOffHash); 
    # Confirm the change in configuration
    $d->msgbox(text => $uiMsg->render('service configuration changed'));
  } elsif ($d->state eq 'EXTRA') {
    return;
  } elsif ($d->state eq 'EXIT') {
    exit EXIT_SUCCESS;
  }
}

bootLevelMenu();
