#!/usr/bin/perl
# -*- project-name: VASM; mode: cperl; mode: outline-minor -*-

### * Preamble
use strict;
use warnings;
use constant +{ 
  false => 0, 
  true => 1,
  labelReal => 0,
  labelShown => 1,
  pad => 2
};
use POSIX;
use VASM::Resource::Navigator;
use VASM::Resource::Catalog::Message;
use VASM::Utility::Gtk2::MessageDialog;
use VASM::Utility::FileSystem::Which;
use Gtk2 '-init';

# Message files
my $uiMsg = messageCatalogFind(qw/navigator ui gtk2/);
my $errorMsg = errorCatalogFind(qw/navigator ui error/);

### * Navigator Tree
### ** navigatorTreeStoreCreate
sub navigatorTreeStoreCreate {
  my ($navigatorTree, $navigatorDirectory) = @_;
  
  # Set up a tree store corresponding to the navigator tree -- one
  # Glib::String for the text as it appears on the screen, the other for the
  # raw text--the 'real name'--usable to the program.
  my $navigatorTreeStore = Gtk2::TreeStore->new(qw/Glib::String/ x 2);

  # _navigatorTreeStoreCreate needs to be factored out, as it is a /recursive/
  # function that will fill in the tree store itself
  _navigatorTreeStoreCreate(
    $navigatorTree, $navigatorDirectory, $navigatorTreeStore
  );
  
  return $navigatorTreeStore;
}

### ** _navigatorTreeStoreCreate
sub _navigatorTreeStoreCreate {
  my ($navigatorTree, $navigatorDirectory, $navigatorTreeStore, @iters) = @_;

  for my $key (sort keys %$navigatorTree) {
    # If a folder
    if (ref $navigatorTree->{$key} eq 'HASH') {
      # Advance the pointer at the current level
      push @iters, $navigatorTreeStore->append($iters[-1] or undef);
      # Set the real name at the current depth
      $navigatorTreeStore->set($iters[-1], labelReal, $key);
      # Set the (translated) description as well
      $navigatorTreeStore->set(
        $iters[-1], labelShown, $navigatorDirectory->render($key)
      );
      # Recurse here
      _navigatorTreeStoreCreate(
        $navigatorTree->{$key}, 
        $navigatorDirectory,
        $navigatorTreeStore, 
        @iters
      );
      pop @iters; # Move back up a level
    }
    # If an item (by exclusion)
    else {
      # Advance the pointer at the current level
      my $iter = $navigatorTreeStore->append($iters[-1] or undef);
      # Set the real name and properties at the current path
      $navigatorTreeStore->set($iter, labelReal, $key);
      # Set the visible label now
      $navigatorTreeStore->set(
        $iter, labelShown, 
        "$key (" . $navigatorDirectory->render($key) . ')'
      );
    }
  }

  return;
}

### ** navigatorTreeViewCreate
sub navigatorTreeViewCreate {
  my $navigatorTree = navigatorTree();
  my $navigatorDirectory = navigatorDirectory();

  # The actual data
  my $navigatorTreeStore = navigatorTreeStoreCreate(
    $navigatorTree, $navigatorDirectory
  );

  # The rest of this is mainly 'cookbook' recipe to make a tree view
  # correspond one-for-one with the data in the tree store
  my $navigatorTreeView = Gtk2::TreeView->new_with_model($navigatorTreeStore);
  $navigatorTreeView->grab_focus; # Using keyboard shortcuts becomes easier
  $navigatorTreeView->set_headers_visible(false);

  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $col->pack_start($renderer, false);
  $col->add_attribute($renderer, text => labelShown);
  $navigatorTreeView->append_column($col);

  return $navigatorTreeView;
}

### ** navigatorTreeViewProgramLaunch
sub navigatorTreeViewProgramLaunch {
  my ($navigatorTreeStore, $iter) = $_[0]->get_selection->get_selected;

  # Does the iter represent a folder? If so, that's bullshit
  if (defined $navigatorTreeStore->iter_children($iter)) {
    errorMessage($errorMsg->render('folder may not be executed'));
    return;
  }
  # Extract the program name next
  my $programName = $navigatorTreeStore->get($iter, labelReal);
  # Now, try to run the selected program
  my $commandString = whichVasm($programName);
  if (defined $commandString) {
    return $commandString;
  } else {
    errorMessage($errorMsg->render('vasm command not found'));
    return;
  }
}

### * Main
### ** navigatorMenu
sub navigatorMenu {
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $uiMsg->render('title'), undef, [],
    'gtk-ok' => 'ok', 'gtk-cancel' => 'cancel'
  );
  $dialog->set_position('center');
  $dialog->set_default_size(400, 300);
  
  ## Navigator area
  # TreeView of navigator
  my $navigatorTreeView = navigatorTreeViewCreate();

  # Pack the tree view into a scrolled window. The addition of scrollbars
  # will be done automatically in either direction
  my $viewScrolled = Gtk2::ScrolledWindow->new;
  $viewScrolled->set_policy(qw/automatic/ x 2);
  $viewScrolled->add($navigatorTreeView);
  $dialog->vbox->pack_start($viewScrolled, true, true, pad);

  # Expander with detailed description
  my $descriptionShort = Gtk2::Expander->new(
    $uiMsg->render('navigator menu short description')
  );
  my $descriptionLong = Gtk2::Label->new(
    $uiMsg->render('navigator menu long description')
  );
  $descriptionLong->set_line_wrap(true);
  $descriptionShort->add($descriptionLong);
  $dialog->vbox->pack_start($descriptionShort, false, false, pad);

  $dialog->show_all;

  DIALOG: {
    my $response = $dialog->run;

    if ($response eq 'ok') {
      # Try to run program
      my $commandString = navigatorTreeViewProgramLaunch($navigatorTreeView);
      redo DIALOG unless defined $commandString;

      $dialog->destroy;
      exit system $commandString;
    } elsif ($response eq 'close') {
      $dialog->destroy;
      exit EXIT_SUCCESS;
    }
  }
}

navigatorMenu();
