#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { false => 0, true => 1 };
use POSIX;
use VASM::Resource::Catalog::Message;
use VASM::Utility::FileSystem::Mouse;
use UI::Dialog::Backend::CDialog;

# Message files
my $commonMsg = messageCatalogFind(qw/common message/);
my $uiMsg = messageCatalogFind(qw/mouseset ui cdialog/);
my $errorMsg = errorCatalogFind(qw/mouseset ui error/);
my $mouseDirectory = mouseDirectory;

# Dialog instance
my $d = UI::Dialog::Backend::CDialog->new(
  backtitle => $commonMsg->render('vasm backtitle'),
  title => $uiMsg->render('title'),
  'ok-label' => $commonMsg->render('ok'),
  'cancel-label' => $commonMsg->render('exit'),
  'yes-label' => $commonMsg->render('yes'),
  'no-label' => $commonMsg->render('no'),
  width => 75,
  height => 20,
  menuheight => 10
);

sub mouseConfigMain {
  my %mouseAttrs;

  unless (getpwuid $> eq 'root') {
    $d->msgbox($errorMsg->render('mouseset for root only'));
    exit EXIT_FAILURE;
  }

  # Let the user know what's going on
  $d->msgbox(text => $uiMsg->render('mouseset intro'));

  # Probe for type and device
  %mouseAttrs = mouseProbe;

  # If autodetection was successful:
  if (defined $mouseAttrs{type}) {
    # Tell the user the good news
    $d->msgbox(
      text => $uiMsg->render(
        'detection successful', 
        $mouseDirectory->render($mouseAttrs{type})
      )
    );
  } else {
    # Ask the user the mouse type explicitly
    mouseTypeAsk(\%mouseAttrs);
  }

  # Ask whether the user wants to emulate three buttons
  mouseEmulateThreeButtonsAsk(\%mouseAttrs);
  # Ask about wheel mouse
  mouseZAxisMappingAsk(\%mouseAttrs);

  # Set mouse
  mouseSet(%mouseAttrs);
  # Confirm this to user
  $d->msgbox(text => $uiMsg->render('configuration complete'));

  exit EXIT_SUCCESS;
}

sub mouseTypeAsk {
  my ($mouseAttrs) = @_;

  my $choices = [
    map {
      $mouseDirectory->_recodeString($_), $mouseDirectory->render($_)
    } $mouseDirectory->list
  ];

  # Prompt for mouse type
  my $mouseType = $d->menu(
    text => $uiMsg->render('mouse type menu description'),
    list => $choices
  );

  if ($d->state eq 'OK') {
    # Do something here...this where I need that extra information on devices
    # in case unusual mice need to be supported
    #
    # If mouse is serial, then serial port needs to be asked at this point
  } elsif ($d->state eq 'CANCEL') {
    exit EXIT_SUCCESS;
  }
  
  return;
}

sub mouseEmulateThreeButtonsAsk {
  my ($mouseAttrs) = @_;

  my $emulateThreeButtons = $d->yesno(
    text => $uiMsg->render('emulate three buttons?'),
  );

  $mouseAttrs->{emulateThreeButtons} = $emulateThreeButtons ? true : false;
  return;
}

sub mouseZAxisMappingAsk {
  my ($mouseAttrs) = @_;
  
  my $choices = [
    map { 
      $uiMsg->render($_), $uiMsg->render("$_ description")
    } ('no wheel', 'wheel', 'wheel and trackball')
  ];

  my $wheelStatus = $d->menu(
    text => $uiMsg->render('wheel mouse?'), 
    list => $choices
  );

  # ASSumption here, are all PS/2 wheel mice IMPS/2?
  if ($d->state eq 'OK') {
    # Do nothing if there was no wheel
    if ($wheelStatus eq $uiMsg->render('wheel')) {
      $mouseAttrs->{zAxisMapping} = '4 5';
      $mouseAttrs->{type} = 'imps2' if $mouseAttrs->{type} eq 'ps2';
    } elsif ($wheelStatus eq $uiMsg->render('wheel and trackball')) {
      $mouseAttrs->{zAxisMapping} = '4 5 6 7';
      $mouseAttrs->{type} = 'imps2' if $mouseAttrs->{type} eq 'ps2';
    }
  } elsif ($d->state eq 'CANCEL') {
    exit EXIT_SUCCESS;
  }

  return;
}

mouseConfigMain();
