#!/usr/bin/perl
# -*- project-name: VASM; mode: cperl; mode: outline-minor -*-

### * Preamble
use strict;
use warnings;
use constant +{
  false => 0, 
  true => 1,
  data => 0,
  label => 1,
  value => 0,
  key => 1,
  keyReal => 2,
  pad => 2
};
use POSIX;
use VASM::Resource::Catalog::Message;
use VASM::Resource::Catalog::Menu;
use VASM::Utility::Gtk2::MessageDialog;
use VASM::Utility::FileSystem::Which;
use File::Spec::Functions qw/curdir rel2abs splitpath/;
use Gtk2 '-init';

# Message files
my $uiMsg = messageCatalogFind(qw/menu ui gtk2/);
my $errorMsg = errorCatalogFind(qw/menu ui error/);

# Error checks for menuTreeStoreElementEdit
my @menuTreeStoreElementEditChecks = (
  sub {
    my ($attrListView) = @_;
    return 'missing fields' unless attrListFull($attrListView->get_model);
    return;
  },
  sub {
    my ($attrListView, $menuTreeStore, $menuTreeStoreElement) = @_;
    return 'element not unique' unless 
      attrListUnique(
        $menuTreeStore, 
        $menuTreeStoreElement, 
        $attrListView->get_model
      );
    return;
  },
  sub {
    my ($attrListView) = @_;
    return 'path invalid' # Must pass -x test
      unless attrListPathValid($attrListView->get_model);
    return;
  },
  sub {
    my ($attrListView) = @_;
    # Must pass -r test
    return 'icon invalid' unless attrListIconValid($attrListView->get_model);
    return;
  }
);

### * Menu Tree
### ** menuTreeStoreCreate
sub menuTreeStoreCreate {
  my ($menuCatalog) = @_;

  # Set up a tree store corresponding to the menu; the single Glib::String
  # column indicates a particular folder or item name in the menu object
  my $menuTreeStore = Gtk2::TreeStore->new('Glib::Scalar', 'Glib::String');

  # _menuTreeStoreCreate needs to be factored out, as it is a /recursive/
  # function that will fill in the tree store itself
  _menuTreeStoreCreate($menuCatalog, $menuTreeStore);

  return $menuTreeStore;
}

### ** _menuTreeStoreCreate
sub _menuTreeStoreCreate {
  my ($menuCatalog, $menuTreeStore, @iters) = @_;

  # Get the current 'stack' of folders
  my @path = map { $menuTreeStore->get($_, label) } @iters;

  # Create more folders and items under it
  for my $child ($menuCatalog->children(@path)) {
    # If an item (only items have the path attribute)
    if ($menuCatalog->retrieve(@path, $child)->{path}) {
      # Advance the pointer at the current level
      my $iter = $menuTreeStore->append($iters[-1] or undef);
      # Set the item name and properties at the current path
      $menuTreeStore->set($iter, data, $menuCatalog->retrieve(@path, $child));
      $menuTreeStore->set($iter, label, $child);
    }
    # If a folder (by exclusion)
    else {
      push @iters, $menuTreeStore->append($iters[-1] or undef);
      # Set the folder name at the current path
      $menuTreeStore->set($iters[-1], label, $child);
      $menuTreeStore->set(
        $iters[-1], data, $menuCatalog->retrieve(@path, $child)
      );
      _menuTreeStoreCreate($menuCatalog, $menuTreeStore, @iters); # Recurse
      pop @iters; # Move back up a level
    }
  }

  return;
}

### ** menuTreeViewCreate
sub menuTreeViewCreate {
  my ($modified) = @_;

  my $menuCatalog = menuFind;
  my $menuTreeStore = menuTreeStoreCreate($menuCatalog); # The actual data

  # If the tree is changed in any way, set the modified flag to true
  for my $signal (qw/row-changed row-inserted row-deleted rows-reordered/) {
    $menuTreeStore->signal_connect($signal => sub { $$modified = true });
  }

  # The rest of this is mainly 'cookbook' recipe to make a tree view
  # correspond one-for-one with the data in the tree store
  my $menuTreeView = Gtk2::TreeView->new_with_model($menuTreeStore);
  $menuTreeView->grab_focus; # Using keyboard shortcuts becomes easier
  $menuTreeView->set_headers_visible(false);

  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $col->pack_start($renderer, false);
  $col->add_attribute($renderer, text => label);
  $menuTreeView->append_column($col);

  # Attach a signal handler for double clicks on row
  $menuTreeView->signal_connect(
    'row-activated' => 
      sub {
        my ($menuTreeStore, $iter) = $_[0]->get_selection->get_selected;
        menuTreeStoreElementEdit($menuTreeStore, $iter);

        return;
      }
  );

  return $menuTreeView;
}

### ** menuTreeStoreElementSet
sub menuTreeStoreElementSet {
  my ($menuTreeStore, $menuTreeStoreElement, $attrList) = @_;
  my %attrList = attrListGetAsHash($attrList);

  # Set the label first
  $menuTreeStore->set($menuTreeStoreElement, label, $attrList{label});
  # Now set all that other stuff
  delete $attrList{label};
  $menuTreeStore->set($menuTreeStoreElement, data, \%attrList);
}

### ** menuTreeStoreElementGetAsHash
sub menuTreeStoreElementGetAsHash {
  my ($menuTreeStore, $menuTreeStoreElement) = @_;
  my %attrList;

  $attrList{label} = $menuTreeStore->get($menuTreeStoreElement, label);
  my $menuAttrList = $menuTreeStore->get($menuTreeStoreElement, data);
  # Add attributes from the data column of the row if defined
  for my $attr (qw/path icon/) {
    $attrList{$attr} = $menuAttrList->{$attr} if defined $menuAttrList->{$attr};
  }

  return %attrList;
}

### ** menuTreeStoreRowIdentify
sub menuTreeStoreRowIdentify {
  my ($menuTreeStore, $iter) = @_; my @path;

  # Walk from branch to trunk, unshifting the 'real' column at each point. The
  # result will define the path in the tree store
  do {
    unshift @path, $menuTreeStore->get($iter, label);
  } while ($iter = $menuTreeStore->iter_parent($iter));

  return @path;
}

### ** menuTreeStoreElementEdit
sub menuTreeStoreElementEdit {
  my ($menuTreeStore, $menuTreeStoreElement) = @_;

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $uiMsg->render('edit properties'), undef, [],
    'gtk-ok' => 'ok', 'gtk-cancel' => 'cancel'
  );
  $dialog->set_resizable(false);
  $dialog->set_position('center');

  # Attributes list
  my $attrListView = attrListViewCreate($menuTreeStore, $menuTreeStoreElement);
  $dialog->vbox->pack_start($attrListView, true, true, pad);

  # Operations (actually, only one)
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = ( [ 'gtk-find', \&attrListFind ] );
  # Add them, well...it...one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $attrListView);
    $operationHButtonBox->pack_start($button, false, false, pad);
  }
  $dialog->vbox->pack_start($operationHButtonBox, false, false, pad);

  # Self-documentation
  my $shortDescription = Gtk2::Expander->new(
    $uiMsg->render('edit properties short description')
  );
  my $longDescription = Gtk2::Label->new(
    $uiMsg->render('edit properties long description')
  );
  $longDescription->set_line_wrap(true);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, false, false, pad);

  $dialog->show_all;

  DIALOG: {
    my $response = $dialog->run; # Do it Jah!
    if ($response eq 'ok') {
      # Run error checks
      for my $check (@menuTreeStoreElementEditChecks) {
        my $result = $check->(
          $attrListView, $menuTreeStore, $menuTreeStoreElement
        );
        # If there was an error
        if (defined $result) {
          errorMessage($errorMsg->render($result)); # Report error
          redo DIALOG; # Do over
        }
      }

      # If we made it here, that means we get to store the attributes loaded
      # in the editing window
      menuTreeStoreElementSet(
        $menuTreeStore, $menuTreeStoreElement, $attrListView->get_model
      );
    }

    $dialog->destroy;
    return ($response eq 'ok') ? 1 : 0;
  }
}

### ** _menuTreeStoreElementAdd
sub _menuTreeStoreElementAdd {
  my ($menuTreeStore, $parent) = @_;

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $uiMsg->render('select type'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_resizable(false);
  $dialog->set_position('center');

  # Radio buttons
  # Speaking of radio: http://wdiy.org -- do give it a look!
  my $selected = 'item';

  # Folder
  my $radio = Gtk2::RadioButton->new(undef, $uiMsg->render('folder'));
  $radio->signal_connect(toggled => sub { $selected = 'folder' });
  $dialog->vbox->pack_start($radio, true, true, pad);
  # Item (default)
  $radio = Gtk2::RadioButton->new($radio, $uiMsg->render('item'));
  $radio->signal_connect(toggled => sub { $selected = 'item' });
  $radio->set_active(true);
  $dialog->vbox->pack_start($radio, true, true, pad);

  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $uiMsg->render('select type short description'));
  my $longDescription = Gtk2::Label->new(
    $uiMsg->render('select type long description'));
  $longDescription->set_line_wrap(true);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, false, false, pad);

  $dialog->show_all;
  my $response = $dialog->run;
  $dialog->destroy;

  return unless $response eq 'ok';

  # Otherwise, call a function according to the selected radio button
  my $menuElementNew = $menuTreeStore->append($parent);
  # Fill in some blank 'default' values
  $menuTreeStore->set($menuElementNew, label, q());
  if ($selected eq 'folder') { 
    $menuTreeStore->set($menuElementNew, data, { icon => q() });
  } elsif ($selected eq 'item') {
    $menuTreeStore->set($menuElementNew, data, { path => q(), icon => q() });
  }

  # Remove the new iter unless the user didn't decide to cancel
  $menuTreeStore->remove($menuElementNew)
    unless menuTreeStoreElementEdit($menuTreeStore, $menuElementNew);

  return;
}

### ** menuTreeStoreElementAdd
sub menuTreeStoreElementAdd {
  my ($menuTreeStore, $parent) = $_[1]->get_selection->get_selected;

  if ($menuTreeStore->get($parent, data)->{path}) { 
    # If it is an item (can't do)
    errorMessage($errorMsg->render('attempted addition to item'));
  } else { 
    # If it is a folder, go ahead
    _menuTreeStoreElementAdd($menuTreeStore, $parent);
  }

  return;
}

### ** menuTreeStoreElementRemove
sub menuTreeStoreElementRemove {
  my $view = $_[1];
  my $menuTreeStore = $view->get_model;
  my $iter = $view->get_selection->get_selected;

  # Confirm the deletion, if the iter has children
  if ($menuTreeStore->iter_children($iter)) {
    my $response = confirmMessage($uiMsg->render('confirm deletion?'));
    return unless $response eq 'yes';
  }

  $menuTreeStore->remove($iter);
  return;
}

### ** menuTreeStoreElementBack
sub menuTreeStoreElementBack {
  my $view = $_[1];
  my $menuTreeStore = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $prev = eval {
    my $path = $menuTreeStore->get_path($iter);
    return $menuTreeStore->get_iter($path) if $path->prev;
  };
  $menuTreeStore->swap($prev, $iter) if $prev;

  return;
}

### ** menuTreeStoreElementForward
sub menuTreeStoreElementForward {
  my $view = $_[1];
  my $menuTreeStore = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $next = eval {
    my $path = $menuTreeStore->get_path($iter); $path->next;
    return $menuTreeStore->get_iter($path);
  };
  $menuTreeStore->swap($iter, $next) if $next;

  return;
}

### ** menuTreeStoreElementUp
sub menuTreeStoreElementUp {
  my $view = $_[1];
  my $menuTreeStore = $view->get_model;
  my $iter = $view->get_selection->get_selected;

  # Get parent and make sure it exists
  my $parent = $menuTreeStore->iter_parent($iter);
  unless (defined $parent) {
    errorMessage(
      $errorMsg->render('attempted movement of top-level upward')
    );
    return;
  }

  # Copy the values to the new iter and delete the existing one
  my $iterNew = $menuTreeStore->append($parent);
  for my $index (label, data) {
    $menuTreeStore->set($iterNew, $index, $menuTreeStore->get($iter, $index));
  }
  $menuTreeStore->remove($iter);

  return;
}

### ** _menuTreeStoreElementDown
sub _menuTreeStoreElementDown {
  my (@folders) = @_; my (@radioButtons, $selected);

  # Create a group of radio buttons with corresponding labels
  for my $folder (@folders) {
    my $radioButton = Gtk2::RadioButton->new_with_label(
      @radioButtons ? $radioButtons[0]->get_group : undef,
      $folder->[0]);
    unless (@radioButtons) {
      $selected = $folder->[1];
      $radioButton->set_active(true);
    }
    $radioButton->signal_connect(toggled => sub { $selected = $folder->[1] });
    push @radioButtons, $radioButton;
  }

  # Run a dialog with these radio buttons
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $uiMsg->render('choose folder'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel'
  );
  $dialog->set_resizable(false);
  $dialog->set_position('center');

  # Add the radio buttons
  for my $radioButton (@radioButtons) {
    $dialog->vbox->pack_start($radioButton, true, true, pad);
  }

  # Self-documentation
  $dialog->vbox->pack_start(
    Gtk2::Label->new($uiMsg->render('choose folder description')),
    false, false, pad
  );

  $dialog->show_all;
  my $response = $dialog->run;
  $dialog->destroy;

  return $selected if $response eq 'ok';
}

### ** menuTreeStoreElementDown
sub menuTreeStoreElementDown {
  my $view = $_[1];
  my $menuTreeStore = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $parent = $menuTreeStore->iter_parent($iter);

  my @folders; # alist of labels and iters for folders at the current level
  for my $index (0..$menuTreeStore->iter_n_children($parent) - 1) {
    my $child = $menuTreeStore->iter_nth_child($parent, $index);
    my $attrList = $menuTreeStore->get($child, data);

    # If there is no path attribute, then it must be a folder. Skip the folder
    # item being moved down
    unless (defined $attrList->{path} or 
            $menuTreeStore->get_string_from_iter($child) eq
            $menuTreeStore->get_string_from_iter($iter)) {
      push @folders, [ $menuTreeStore->get($child, label), $child ];
    }
  }

  return unless @folders; # Can't work if there be no candiate folders!

  # Many are called, few are chosen -- select a new parent for the item
  my $parentNew = _menuTreeStoreElementDown(@folders);
  return unless $parentNew;

  # If needed, copy the values to the new iter and delete the existing one
  my $iterNew = $menuTreeStore->append($parentNew);
  for my $index (label, data) {
    $menuTreeStore->set($iterNew, $index, $menuTreeStore->get($iter, $index));
  }
  $menuTreeStore->remove($iter);

  return;
}

### * Attribute List
### ** attrListCreate
sub attrListCreate {
  my %attrList = @_; my $iter;
  my $attrList = Gtk2::ListStore->new(qw/Glib::String/ x 3);

  for my $attr (qw/label path icon/) {
    # If the folder or item has this attr...
    if (defined $attrList{$attr}) {
      $iter = $attrList->append;
      # Name of attribute, translated
      $attrList->set($iter, key, $uiMsg->render($attr));
      # Used for ID
      $attrList->set($iter, keyReal, $attr);
      # Value
      $attrList->set($iter, value, $attrList{$attr});
    }
  }

  return $attrList;
}

### ** attrListViewCreate
sub attrListViewCreate {
  my ($menuTreeStore, $menuTreeStoreElement) = @_;

  my $attrList = attrListCreate(
    menuTreeStoreElementGetAsHash($menuTreeStore, $menuTreeStoreElement)
  );
  my $attrListView = Gtk2::TreeView->new_with_model($attrList);
  $attrListView->grab_focus; # Using keyboard shortcuts becomes easier

  my @fields = qw/attribute value/;
  for my $field (key, value) {
    my $col = Gtk2::TreeViewColumn->new;
    # Set 'Attribute' and 'Value' headers
    $col->set_title($uiMsg->render($fields[$field]));
    my $renderer = Gtk2::CellRendererText->new;
    # The 'value' column is editable
    if ($field eq value) {
      $renderer->set(editable => true);
      $renderer->signal_connect(edited => \&attrListCellEdited, $attrList);
    }
    $col->pack_start($renderer, false);
    $col->add_attribute($renderer, text => $field);
    $attrListView->append_column($col);
  }

  return $attrListView;
}

### ** attrListGetAsHash
sub attrListGetAsHash {
  my ($attrList) = @_; my %attrList;
  my $iter = $attrList->get_iter_first;

  # Iterate over the list store and grab pairs of keyReal => value, the
  # non-translated field name and its value
  do {
    $attrList{$attrList->get($iter, keyReal)} = $attrList->get($iter, value);
  } while ($iter = $attrList->iter_next($iter));

  return %attrList;
}

### ** attrListFull
sub attrListFull {
  my ($attrList) = @_;
  my %attrList = attrListGetAsHash($attrList);

  # Return true if the number of elements in $attrList with a length of 1 or
  # more is equal to the number of keys
  return true if (grep { length } values %attrList) == keys %attrList;
}

### ** attrListUnique
sub attrListUnique {
  my ($menuTreeStore, $menuTreeStoreElement, $attrList) = @_;
  my %attrList = attrListGetAsHash($attrList);
  # The first row at the same level as $menuTreeStoreElement
  my $iter = $menuTreeStore->iter_children(
    $menuTreeStore->iter_parent($menuTreeStoreElement)
  );

  do {{
    # Ignore comparisons between the element being edited and itself
    next if $menuTreeStore->get_string_from_iter($menuTreeStoreElement) eq
            $menuTreeStore->get_string_from_iter($iter);
    my %attrListOther = menuTreeStoreElementGetAsHash($menuTreeStore, $iter);
    # Return false if the label for the edited menu element is not unique
    return false if $attrList{label} eq $attrListOther{label};
  }
    } while ($iter = $menuTreeStore->iter_next($iter));

  # Otherwise, it falls through and returns true
  return true;
}

### ** attrListPathValid
sub attrListPathValid {
  my ($attrList) = @_;
  my %attrList = attrListGetAsHash($attrList);

  # Return true if the path attribute of %attrList passes -x...if even
  # necessary
  if (defined $attrList{path}) {
    return false unless which($attrList{path});
  }

  return true;
}

### ** attrListIconValid
sub attrListIconValid {
  my ($attrList) = @_;
  my %attrList = attrListGetAsHash($attrList);

  # Return true if the icon attribute of %attrList passes -r...no further
  # checks (valid format, whatever)
  return true if -r $attrList{icon};
  return false;
}

### ** attrListCellEdited
sub attrListCellEdited {
  my ($path, $text, $attrList) = @_[1..3];

  # Just pass the text through
  $attrList->set($attrList->get_iter_from_string($path), value, $text);

  return;
}

### ** attrListFind
BEGIN {
  my $wd = rel2abs(curdir);

  sub attrListFind {
    my $view = $_[1];
    my ($attrList, $iter) = $view->get_selection->get_selected;

    # The label attribute has no 'Find' option!!
    return if $attrList->get($iter, keyReal) eq 'label';

    my $fileChooser = Gtk2::FileChooserDialog->new(
      $uiMsg->render('select file'), undef, 'open',
      'gtk-ok' => 'ok',
      'gtk-cancel' => 'cancel'
    );
    $fileChooser->set_current_folder($wd);
    $fileChooser->set_resizable(false);
    $fileChooser->set_position('center');

    if ($attrList->get($iter, keyReal) eq 'icon') {
      # Set filter on file chooser as well -- only image formats
      my $filter = Gtk2::FileFilter->new; $filter->add_pixbuf_formats;
      $fileChooser->set_filter($filter);
      # Icons will have preview
      my $preview = Gtk2::Image->new;
      $fileChooser->set_preview_widget($preview);
      $fileChooser->signal_connect(
        'update-preview' => \&iconPreviewUpdate, $preview
      );
    }

    $fileChooser->show_all;
    my $response = $fileChooser->run;

    if ($response eq 'ok') {
      my $file = $fileChooser->get_filename;
      $wd = (splitpath($file))[1]; # Store new current directory
      $attrList->set($iter, value, $file); # Set element in store
    }

    $fileChooser->destroy;

    return;
  }
}

### ** iconPreviewUpdate
sub iconPreviewUpdate {
  my ($fileChooser, $preview) = @_;

  my $file = $fileChooser->get_preview_filename;
  # For some reason, this callback gets spurious bullshit from time to time
  if (defined $file and -f $file) {
    my $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file_at_size($file, 64, 64);
    $preview->set_from_pixbuf($pixbuf);
    $fileChooser->set_preview_widget_active(defined $pixbuf ? true : false);
  } else {
    $fileChooser->set_preview_widget_active(false);
  }

  return;
}

### * Main
### ** mainMenu
sub mainMenu {
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $uiMsg->render('title'), undef, [],
    'gtk-apply' => 'apply',
    'gtk-ok' => 'ok',
    'gtk-close' => 'close'
  );
  $dialog->set_position('center');
  $dialog->set_default_size(400, 300);

  ## Action area
  # TreeView of menu
  my $modified = false; my $menuTreeView = menuTreeViewCreate(\$modified);

  # Pack the tree view into a scrolled window. The addition of scrollbars
  # will be done automatically in either direction
  my $viewScrolled = Gtk2::ScrolledWindow->new;
  $viewScrolled->set_policy(qw/automatic/ x 2);
  $viewScrolled->add($menuTreeView);
  $dialog->vbox->pack_start($viewScrolled, true, true, pad);

  # Operations
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = ( 
    [ 'gtk-add' => \&menuTreeStoreElementAdd ],
    [ 'gtk-remove' => \&menuTreeStoreElementRemove ],
    [ 'gtk-go-back' => \&menuTreeStoreElementBack ],
    [ 'gtk-go-forward' => \&menuTreeStoreElementForward ],
    [ 'gtk-go-up' => \&menuTreeStoreElementUp ],
    [ 'gtk-go-down' => \&menuTreeStoreElementDown ] 
  );

  # Add them, one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $menuTreeView);
    $operationHButtonBox->pack_start($button, true, true, pad);
  }
  $dialog->vbox->pack_start($operationHButtonBox, false, false, pad);

  # Expander with detailed description
  my $descriptionShort = Gtk2::Expander->new(
    $uiMsg->render('main menu short description')
  );
  my $descriptionLong = Gtk2::Label->new(
    $uiMsg->render('main menu long description')
  );
  $descriptionLong->set_line_wrap(true);
  $descriptionShort->add($descriptionLong);
  $dialog->vbox->pack_start($descriptionShort, false, false, pad);

  $dialog->show_all;

  DIALOG: {
    my $response = $dialog->run; my $menuCatalog;

    if ($response eq 'apply') {
      $menuCatalog = menuGenerate($menuTreeView->get_model);
      menuWrite($menuCatalog);
      $modified = false; # Having saved the menu, $modified is now false
      redo DIALOG;
    } elsif ($response eq 'ok') {
      $menuCatalog = menuGenerate($menuTreeView->get_model);
      menuWrite($menuCatalog);
    } elsif ($response eq 'close' and $modified) {
      # Confirm this
      my $response = confirmMessage($uiMsg->render('confirm cancellation?'));
      # If answered no, stay here
      $response eq 'no' and redo DIALOG;
    }
  }

  $dialog->destroy;
  exit EXIT_SUCCESS;
}

### ** menuGenerate
sub menuGenerate {
  my ($menuTreeStore) = @_;
  my $menu = VASM::Catalog::Menu->new;

  # Intern each element of the menu description from $menuTreeStore
  $menuTreeStore->foreach(
    sub {
      my ($menuTreeStore, $iter) = @_[0,2];

      # Textual description of the path
      my @path = menuTreeStoreRowIdentify($menuTreeStore, $iter);
      # Handle top-level special case
      return false unless @path;
      # Element attributes
      my %attrList = menuTreeStoreElementGetAsHash($menuTreeStore, $iter); 
      delete $attrList{label};
      # Store the resulting data
      $menu->store(@path, \%attrList);

      return false;
    }
  );

  return $menu;
}

### * Invocation
mainMenu();
