#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use POSIX;
use constant { false => 0, true => 1 };
use File::Spec::Functions qw/curdir rel2abs splitpath/;
use VASM::Resource::Catalog::XorgConfig;
use VASM::Resource::Catalog::Message;
use UI::Dialog::Backend::CDialog;

# Message files
my $commonMsg = messageCatalogFind(qw/common message/);
my $uiMsg = messageCatalogFind(qw/fontset ui cdialog/);
my $errorMsg = errorCatalogFind(qw/fontset ui error/);

# Dialog instance
my $d = UI::Dialog::Backend::CDialog->new(
  backtitle => $commonMsg->render('vasm backtitle'),
  title => $uiMsg->render('title'),
  'ok-label' => $commonMsg->render('ok'),
  'cancel-label' => $commonMsg->render('exit'),
  'yes-label' => $commonMsg->render('yes'),
  'no-label' => $commonMsg->render('no'),
  width => 75,
  height => 20,
  menuheight => 10
);

sub mainMenu {
  unless (getpwuid $> eq 'root') {
    $d->msgbox(text => $errorMsg->render('fontset for root only'));
    exit EXIT_FAILURE;
  }

  my $catalogChanged = false;
  my @operations = map {
    $uiMsg->render($_), $uiMsg->render("$_ description")
  } qw/list add remove clear save/;

  # First, load the config catalog; generate a new one if necessary
  my $catalog = xorgConfigCatalogFind();
    
  while (true) {
    # Prompt choice
    my $operation = $d->menu(
      text => $uiMsg->render('main menu description'),
      list => [ @operations ]
    );
    
    # Act on input -- ugly, ugly, ugly
    if ($d->state eq 'OK') {
      if ($operation eq $uiMsg->render('list')) {
        fontPathList($catalog);
      } elsif ($operation eq $uiMsg->render('add')) {
        fontPathAdd($catalog, \$catalogChanged);
      } elsif ($operation eq $uiMsg->render('remove')) {
        fontPathRemove($catalog, \$catalogChanged);
      } elsif ($operation eq $uiMsg->render('clear')) {
        fontPathClear($catalog, \$catalogChanged);
      } elsif ($operation eq $uiMsg->render('save')) {
        fontPathSave($catalog, \$catalogChanged);
      }
    } elsif ($d->state eq 'CANCEL' and $catalogChanged) {
      my $lastState = $d->state; # Bottle up the last state

      # Ask whether user wants to save or continue first
      my $confirm = $d->yesno(text => $uiMsg->render('confirm cancellation?'));
      redo unless $confirm;

      # Otherwise, it falls through to here
      exit EXIT_SUCCESS;
    }
  }
}

sub fontPathList {
  my ($catalog) = @_;
  my @fontpaths = map {
    $commonMsg->recodeString($_)
  } $catalog->retrieve('fontpath');
    
  my $uglyListText = $uiMsg->render('font paths listed');
  $uglyListText .= ":\\n\\n\n";
  $uglyListText .= join("\\n\n", @fontpaths);

  $d->msgbox(text => $uglyListText);

  return;
}

BEGIN {
  # fontPathAdd has its own 'static' variable for the current working
  # directory; see perlfaq7
  my $wd = rel2abs(curdir);

  sub fontPathAdd {
    my ($catalog, $catalogChanged) = @_;
    
    DIALOG: {
      my $fontpath = $d->dselect(
        path => $wd, height => 10,
        'cancel-label' => $commonMsg->render('back')
      );

      if ($d->state eq 'OK') {
        # Emit warning message and return unless the path is a valid value
        $d->msgbox(text => $errorMsg->render('null font path')) and return
          unless length $fontpath;

        # Store new current working directory
        $wd = (splitpath($fontpath))[1];
    
        # Ask for stack or queue behavior (Unshift or Push)
        my $addMethod = $d->yesno(
          text => $uiMsg->render('beginning or end?'),
          'yes-label' => $uiMsg->render('beginning'),
          'no-label' => $uiMsg->render('end')
        ) ? 'unshift' : 'push';

        # Ask whether these are scaled or unscaled fonts 
        my $scalable = $d->yesno(
          text => $uiMsg->render('unscalable fonts?'),
          'yes-label' => $uiMsg->render('scalable'),
          'no-label' => $uiMsg->render('unscalable')
        );

        $fontpath .= ':unscaled' unless $scalable;
      
        # Add it to the catalog instance
        $catalog->$addMethod('fontpath', $fontpath);
        # Set the catalog changed flag to true
        $$catalogChanged = true;
      }
    }
  
    return;
  }
}

sub fontPathRemove {
  my ($catalog, $catalogChanged) = @_;

  # Retrieve
  my @fontpaths = $catalog->retrieve('fontpath');
  # Check empty
  $d->msgbox(text => $errorMsg->render('no font paths')) and return
    unless @fontpaths;

  # Generate menu  
  my @choices = map {
    $_, [ $commonMsg->recodeString($fontpaths[$_ - 1]), false ]
  } (1..@fontpaths);
    
  # Get an array of paths nominated for deletion
  my @deleted = $d->checklist(
    text => $uiMsg->render('remove font paths'),
    list => [ @choices ]
  );
  
  if ($d->state eq 'OK') {
    # And an entrance from everyone's least favorite Perl function
    for my $index (@deleted) { splice @fontpaths, $index - 1 }
    # Store the newly reduced list
    $catalog->store(fontpath => @fontpaths);
    # Set the 'catalog changed' flag to true
    $$catalogChanged = true;
  }

  return;
}

sub fontPathClear {
  my ($catalog, $catalogChanged) = @_;
  
  # Confirm this destructive operation
  my $confirm = $d->yesno(text => $uiMsg->render('confirm clear?'));

  $catalog->clear('fontpath') and $$catalogChanged = true
    if $confirm; # "Do it!" -- Yellowman

  return;
}

sub fontPathSave {
  my ($catalog, $catalogChanged) = @_;
  
  xorgConfigCatalogWrite($catalog);
  $d->msgbox(text => $uiMsg->render('configuration saved'));

  # Set the 'catalog changed' flag to false
  $$catalogChanged = false;

  return;
}

mainMenu();
