#!/usr/bin/perl
# -*- project-name: VASM: mode: cperl; mode: outline-minor -*-

### * Preamble
use strict;
use warnings;
use constant +{
  false => 0, 
  true => 1,
  datum => 0, 
  labelReal => 0, 
  labelShown => 1,
  pad => 2
};
use VASM::Tree;
use VASM::Resource;
use VASM::Resource::Catalog::Message;
use VASM::Resource::Catalog::Action;
use VASM::Utility::Gtk2::MessageDialog;
use VASM::Utility::FileSystem::Which;
use Gtk2 '-init';
use File::Spec::Functions qw/curdir rel2abs splitpath/;

# Message files
my $uiMsg = messageCatalogFind(qw/action ui gtk2/);
my $errorMsg = errorCatalogFind(qw/action ui error/);

### * Action Tree
### ** actionTreeStoreCreate
sub actionTreeStoreCreate {
  # Get directories for file and URI schemata
  my $fileDirectory = mediaDirectory('file');
  my $uriDirectory = mediaDirectory('uri');

  # Convert the file directory to data more readily usable by the function
  my $fileTypeTree = VASM::Tree->new;
  for my $mime ($fileDirectory->list) {
    $fileTypeTree->store((split m!/!, $mime), true);
  }

  # Now, set up a tree store corresponding to them and add the 'Template'
  # top-level choice as well. The first Glib::String is the item as it appears
  # on the screen, according to translation and additional notes. The second
  # is the 'real name', usable to the program.
  my $actionTreeStore = Gtk2::TreeStore->new('Glib::String', 'Glib::String');
  my @iters;

  # File
  push @iters, $actionTreeStore->append(undef);
  $actionTreeStore->set(
    $iters[-1],
    labelReal, 'file',
    labelShown, $uiMsg->render('file')
  );

  for my $class (sort $fileTypeTree->children) {
    # Add the class under 'File'
    push @iters, $actionTreeStore->append($iters[-1]);
    $actionTreeStore->set(
      $iters[-1],
      labelReal, $class,
      labelShown, $class
    );
    # Add each member of the class (e.g., File -> image -> jpeg)
    for my $member (sort $fileTypeTree->children($class)) {
      push @iters, $actionTreeStore->append($iters[-1]);
      my $mime = join('/', $class, $member);
      # The type will appear with a parenthetical note describing it in
      # further detail
      $actionTreeStore->set(
        $iters[-1],
        labelReal, $member,
        labelShown, $member . ' (' . $fileDirectory->render($mime) . ')'
      );
    } continue { pop @iters }
  } continue { pop @iters }
  pop @iters;

  # URI
  push @iters, $actionTreeStore->append(undef);
  $actionTreeStore->set(
    $iters[-1],
    labelReal, 'uri',
    labelShown, $uiMsg->render('URI')
  );

  for my $scheme (sort $uriDirectory->list) {
    push @iters, $actionTreeStore->append($iters[-1]);
    $actionTreeStore->set(
      $iters[-1],
      labelReal, $scheme,
      labelShown, $scheme . ' (' . $uriDirectory->render($scheme) . ')'
    );
  } continue { pop @iters }
  pop @iters;

  # Template
  push @iters, $actionTreeStore->append(undef);
  $actionTreeStore->set(
    $iters[-1],
    labelReal, 'template',
    labelShown, $uiMsg->render('template')
  );
  pop @iters; # Neat freak

  return $actionTreeStore;
}

### ** actionTreeViewCreate
sub actionTreeViewCreate {
  my $actionTreeStore = actionTreeStoreCreate(); # The actual data
  # The rest of this is a 'cookbook' recipe to make a tree view correspond
  # one-for-one with the data in the tree store.
  my $actionTreeView = Gtk2::TreeView->new_with_model($actionTreeStore);
  $actionTreeView->set_headers_visible(false);
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $col->pack_start($renderer, false);
  $col->add_attribute($renderer, text => labelShown);
  $actionTreeView->append_column($col);
  
  # Attach a signal handler for double clicks on rows
  $actionTreeView->signal_connect(
    'row-activated' => sub { 
      mediaTypeDispatch(actionTreeStoreRowIdentify($_[0]))
    }
  );

  return $actionTreeView;
}

### ** actionTreeStoreRowIdentify
sub actionTreeStoreRowIdentify {
  my ($actionTreeStore, $iter) = $_[0]->get_selection->get_selected; my @path;

  # Walk from branch to trunk, unshifting the 'real' column at each point. The
  # result will define the path in the tree store.
  do {
    unshift @path, $actionTreeStore->get($iter, labelReal);
  } while ($iter = $actionTreeStore->iter_parent($iter));

  return @path;
}

### ** mediaTypeDispatch
sub mediaTypeDispatch {
  my @path = @_;

  # Fully qualified path was given
  if (($path[0] eq 'file' and @path == 3) or
      ($path[0] eq 'uri' and @path == 2)) {
    mediumConfigure(@path);
  } elsif ($path[0] eq 'template') { # Select a template
    templateSelect();
  }
  
  return;
}

### * Action List
### ** actionListCreate
sub actionListCreate {
  my ($catalog) = @_; my $iter;
  my $actionList = Gtk2::ListStore->new('Glib::String');
  my @programs = $catalog->dump;

  if (@programs) { # Add each program in the catalog to the list store
    for my $program (@programs) {
      $iter = $actionList->append;
      $actionList->set($iter, datum, $program);
    }
  } else { # Empty catalog
    $iter = $actionList->append;
    $actionList->set($iter, datum, q()); # Add a new element
  }
  
  return $actionList;
}

### ** actionListViewCreate
sub actionListViewCreate {
  my ($catalog, $modified) = @_;

  # Cookbook recipe that instantiates a list view store from an actions
  # catalog and imposes an editable view component on it
  my $actionList = actionListCreate($catalog);
  # If the tree is changed in any way, set the modified flag to true
  for my $signal (qw/row-changed row-inserted row-deleted rows-reordered/) {
    $actionList->signal_connect($signal => sub { $$modified = true });
  }
  my $actionListView = Gtk2::TreeView->new_with_model($actionList);
  $actionListView->set_headers_visible(false);
  # $view->set_reorderable(true); # Allow drag and drop :D
  $actionListView->grab_focus; # Using keyboard shortcuts becomes easier
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $renderer->set(editable => true);
  $renderer->signal_connect(edited => \&actionListCellEdited, $actionList);
  $col->pack_start($renderer, false);
  $col->add_attribute($renderer, text => datum);
  $actionListView->append_column($col);

  return $actionListView;
}

### ** actionListFull
sub actionListFull {
  my ($actionList) = @_; my $iter = $actionList->get_iter_first;
  
  # Return false unless all elements of the list store are not blank
  while (defined $iter) {
    return false unless length $actionList->get($iter, datum);
    $iter = $actionList->iter_next($iter);
  }

  return true; # Fall through
}

### ** actionListCommandsValid
sub actionListCommandsValid {
  my ($actionList) = @_; my $iter = $actionList->get_iter_first;
  
  # Return false unless all elements of the list store represent valid
  # commands
  while (defined $iter) {
    return false unless defined which($actionList->get($iter, datum));
    $iter = $actionList->iter_next($iter);
  }

  return true; # Fall through
}

### ** mediumConfigure
sub mediumConfigure {
  my @path = @_;
  my $mediaType = (@path == 3) ? join '/', @path[1..2] : $path[1];

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $mediaType, undef, [],
    'gtk-apply' => 'apply',
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel'
  );
  $dialog->set_position('center');

  # Media list
  # First, load the action catalog; generate a new one if necessary
  my $catalog = actionCatalogFind(@path);
  my $modified = false;
  my $actionListView = actionListViewCreate($catalog, \$modified);

  $dialog->vbox->pack_start($actionListView, true, true, pad);

  # Operations
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = (
    [ 'gtk-find' => \&actionListFind ],
    [ 'gtk-add' => \&actionListAdd ],
    [ 'gtk-remove' => \&actionListRemove ],
    [ 'gtk-go-up' => \&actionListUp ],
    [ 'gtk-go-down' => \&actionListDown ]
  );

  # Add them, one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $actionListView);
    $operationHButtonBox->pack_start($button, false, false, pad);
  }
  $dialog->vbox->pack_start($operationHButtonBox, false, false, pad);

  # Description
  my $shortDescription = Gtk2::Expander->new(
    $uiMsg->render('medium configurator short description', $mediaType)
  );
  my $longDescription = Gtk2::Label->new(
    $uiMsg->render('medium configurator long description')
  );

  $longDescription->set_line_wrap(true);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, false, false, pad);

  $dialog->show_all;

  DIALOG: {
    my $response = $dialog->run;
    # Write out the catalog if so desired
    if ($response eq 'ok' or $response eq 'apply') {
      # Are all fields full?
      unless (actionListFull($actionListView->get_model)) {
        # Report error
        errorMessage($errorMsg->render('missing fields'));
        redo DIALOG; # Do over
      }
      # Are all commands valid?
      unless (actionListCommandsValid($actionListView->get_model)) {
        # Report error
        errorMessage($errorMsg->render('invalid commands'));
        redo DIALOG;
      }

      $catalog = actionCatalogGenerate($actionListView->get_model);
      actionCatalogWrite($catalog, @path);
      
      # Quit or redo dialog depending on whether this was an 'ok' or 'apply',
      # respectively
      if ($response eq 'ok') { last DIALOG }
      if ($response eq 'apply') { $modified = false; redo DIALOG }
    } elsif ($modified) {
      # Ask whether user wants to save or continue first
      $response = confirmMessage($uiMsg->render('confirm cancellation?'));
      redo DIALOG if $response eq 'no';
    }
  }

  $dialog->destroy;

  return;
}

### ** actionListCellEdited
sub actionListCellEdited {
  my ($pathString, $text, $actionList) = @_[1..3];

  # Just pass the text through
  $actionList->set(
    $actionList->get_iter_from_string($pathString), 
    datum, $text
  );
  
  return;
}

### ** actionListAdd
sub actionListAdd {
  my $actionListView = $_[1];
  my $actionList = $actionListView->get_model;
  my $iter = $actionList->insert_after(
    scalar $actionListView->get_selection->get_selected
  );
  $actionList->set($iter, datum, q()); # Add a new element

  return;
}

### ** actionListRemove
sub actionListRemove {
  my $actionListView = $_[1];
  my ($actionList, $iter) = $actionListView->get_selection->get_selected;
  $actionList->remove($iter) if $iter;

  return;
}

### ** actionListUp
sub actionListUp {
  my $actionListView = $_[1];
  my ($actionList, $iter) = $actionListView->get_selection->get_selected;
  my $prev = eval {
    my $path = $actionList->get_path($iter);
    return $actionList->get_iter($path) if $path->prev;
  };
  $actionList->swap($prev, $iter) if $prev;
  
  return;
}

### ** actionListDown
sub actionListDown {
  my $actionListView = $_[1];
  my ($actionList, $iter) = $actionListView->get_selection->get_selected;
  my $next = eval {
    my $path = $actionList->get_path($iter); $path->next;
    return $actionList->get_iter($path);
  };
  $actionList->swap($iter, $next) if $next;
  
  return;
}

### ** actionListFind
BEGIN {
  my $wd = rel2abs(curdir);
  
  sub actionListFind {
    my ($actionListView) = $_[1];
    my ($actionList, $iter) = $actionListView->get_selection->get_selected;

    my $fileChooser = Gtk2::FileChooserDialog->new(
      $uiMsg->render('select file'), undef, 'open',
      'gtk-ok' => 'ok',
      'gtk-cancel' => 'cancel'
    );
    $fileChooser->set_current_folder($wd);
    $fileChooser->set_position('center');

    $fileChooser->show_all;
    my $response = $fileChooser->run;

    if ($response eq 'ok') {
      my $file = $fileChooser->get_filename;
      $wd = (splitpath($file))[1]; # Store new current directory
      $actionList->set($iter, datum, $file); # Set element in store
    }

    $fileChooser->destroy;
    return;
  }
}

### * Template
### ** templateSelect
BEGIN {
  my $wd = rel2abs(curdir);
  
  sub templateSelect {
    my $fileChooser = Gtk2::FileChooserDialog->new(
      $uiMsg->render('select template'), undef, 'open',
      'gtk-ok' => 'ok', 'gtk-cancel' => 'cancel'
    );
    $fileChooser->set_position('center');
    my $filter = Gtk2::FileFilter->new;
    # MIME-type selection seems unfortunately unreliable :?
    for my $ext (qw/tgz tar tar.gz/) { $filter->add_pattern("*.$ext") }
    $fileChooser->set_filter($filter);

    DIALOG: {
      $fileChooser->set_current_folder($wd);
      $fileChooser->show_all;

      my $response = $fileChooser->run;
      # If a given file is accepted...
      if ($response eq 'ok') {
        my $templateFile = $fileChooser->get_filename;
        # Confirm installation of the template and install if confirmed
        unless (templateInstall($templateFile)) {
          errorMessage($errorMsg->carp('Invalid Archive'));
          redo DIALOG;
        }
        # Store new current directory
        $wd = (splitpath($templateFile))[1]; $wd =~ s!/+$!!;
        # Confirm the installation (to the user)
        infoMessage($uiMsg->render('installation complete'));
      }
    }

    $fileChooser->destroy;
    return;
  }
}

### * Main
### ** mainMenu
sub mainMenu {
  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $uiMsg->render('title'), undef, [],
    'gtk-close' => 'close'
  );
  $dialog->set_position('center');
  $dialog->set_default_size(400, 300);
    
  ## Action area
  # Tree with media
  my $actionTreeView = actionTreeViewCreate();
  # Pack the tree view into a scrolled window. The addition of scrollbars
  # will be done automatically in either direction
  my $viewScrolled = Gtk2::ScrolledWindow->new;
  $viewScrolled->set_policy(qw/automatic/ x 2);
  $viewScrolled->add($actionTreeView);
  $dialog->vbox->pack_start($viewScrolled, true, true, pad);
  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $uiMsg->render('main menu short description')
  );
  my $longDescription = Gtk2::Label->new(
    $uiMsg->render('main menu long description')
  );
  $longDescription->set_line_wrap(true);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, false, false, pad);

  # Do it Jah!
  $dialog->show_all;
  $dialog->run;
  $dialog->destroy;
  
  exit 0;
}

### ** actionCatalogGenerate
sub actionCatalogGenerate {
  my ($actionList) = @_; my $iter;
  my $catalog = VASM::Catalog::Action->new;

  # Add an element in the catalog for each member in the list store
  $actionList->foreach(
    sub { $catalog->push($_[0]->get($_[2], datum)); return false }
  );
  
  return $catalog;
}

### * Invocation
mainMenu();
