#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { false => 0, true => 1 };
use POSIX;
use VASM::Resource;
use VASM::Resource::Catalog::Message;
use VASM::Resource::Catalog::Action;
use VASM::Utility::FileSystem::Which;
use File::Spec::Functions qw/curdir rel2abs splitpath/;
use UI::Dialog::Backend::CDialog;

# Message files
my $commonMsg = messageCatalogFind(qw/common message/);
my $uiMsg = messageCatalogFind(qw/action ui cdialog/);
my $errorMsg = errorCatalogFind(qw/action ui error/);

# Dialog instance
my $d = UI::Dialog::Backend::CDialog->new(
  backtitle => $commonMsg->render('vasm backtitle'),
  title => $uiMsg->render('title'),
  'ok-label' => $commonMsg->render('ok'),
  'cancel-label' => $commonMsg->render('exit'),
  'yes-label' => $commonMsg->render('yes'),
  'no-label' => $commonMsg->render('no'),
  width => 75,
  height => 20,
  menuheight => 10
);

sub mainMenu {
  # Choices for main menu -- constant
  my $choices = [
    map {
      $uiMsg->render($_), $uiMsg->render("$_ description")
    } qw/file URI template/
  ];
  
  while (true) {
    # Prompt choice
    my $mediaCategory = $d->menu(
      text => $uiMsg->render('main menu description'),
      list => $choices
    );

    # Act on input -- ugly, but it looks like this is the way it has to be
    if ($d->state eq 'OK') {
      if ($mediaCategory eq $uiMsg->render('file')) {
        mediaMenu('file');
      } elsif ($mediaCategory eq $uiMsg->render('URI')) {
        mediaMenu('uri');
      } elsif ($mediaCategory eq $uiMsg->render('template')) {
        templateSelect();
      }
    } elsif ($d->state eq 'CANCEL') {
      exit EXIT_SUCCESS;
    }
  }
}

sub mediaMenu {
  my ($mediaCategory) = @_;
  my $directory = mediaDirectory($mediaCategory);

  my @choices = map {
    $directory->_recodeString($_), $directory->render($_)
  } sort $directory->list;

  while (true) {
    # Prompt choice
    my $member = $d->menu(
      text => $uiMsg->render('media menu description'),
      'extra-button' => true,
      'extra-label' => $commonMsg->render('back'),
      list => [ @choices ]
    );
    my @path = ($mediaCategory, split(m!/!, $member));

    # Act on input
    if ($d->state eq 'OK') {
      operationMenu(@path);
    } elsif ($d->state eq 'EXTRA') {
      return;
    } elsif ($d->state eq 'CANCEL') {
      exit EXIT_SUCCESS;
    }
  }
}

sub operationMenu {
  my @path = @_;

  # First, load the action catalog
  my $catalog = actionCatalogFind(@path);
  # Flag for tracking changed status of catalog 
  my $catalogChanged = false;
  # Operation choices for menu -- constant
  my @operations = map {
    $uiMsg->render("$_"), $uiMsg->render("$_ description")
  } qw/list add remove clear save/;

  while (true) {
    # Prompt choice
    my $operation = $d->menu(
      text => $uiMsg->render('operation menu description'),
      'extra-button' => true,
      'extra-label' => $commonMsg->render('back'),
      list => [ @operations ]
    );

    # Act on input -- ugly, ugly, ugly
    if ($d->state eq 'OK') {
      if ($operation eq $uiMsg->render('list')) {
        mediumList($catalog);
      } elsif ($operation eq $uiMsg->render('add')) {
        mediumAdd($catalog, \$catalogChanged);
      } elsif ($operation eq $uiMsg->render('remove')) {
        mediumRemove($catalog, \$catalogChanged);
      } elsif ($operation eq $uiMsg->render('clear')) {
        mediumClear($catalog, \$catalogChanged);
      } elsif ($operation eq $uiMsg->render('save')) {
        mediumSave($catalog, @path, \$catalogChanged);
      }
    } elsif ($d->state eq 'EXTRA' or $d->state eq 'CANCEL') {
      my $lastState = $d->state; # Bottle up the last state
      
      # Ask whether user wants to save or continue first
      if ($catalogChanged) {
        my $confirm = $d->yesno(
          text => $uiMsg->render('confirm cancellation?')
        );
        redo unless $confirm;
      }

      # Otherwise, it falls through to here
      $lastState eq 'EXTRA' and return;
      $lastState eq 'CANCEL' and exit EXIT_SUCCESS;
    }
  }
}

sub mediumList {
  my ($catalog) = @_;

  my $uglyListText = $uiMsg->render('commands listed');
  $uglyListText .= ":\\n\n\\n\n";
  $uglyListText .= join(
    "\\n\n", map { $commonMsg->_recodeString($_) } $catalog->dump
  );

  $d->msgbox(text => $uglyListText);

  return;
}

sub mediumAdd {
  my ($catalog, $catalogChanged) = @_; my $commandEntry = q();

  DIALOG: {
    my $command = $d->inputbox(
      text => $uiMsg->render('add command'), entry => $commandEntry,
      'extra-button' => true, 'extra-label' => $uiMsg->render('find'),
      'cancel-label' => $commonMsg->render('back')
    );

    if ($d->state eq 'OK') {
      # Emit error message and redo unless the command is non-blank
      unless (length $command) {
        $d->msgbox(text => $errorMsg->render('null command string'));
        redo DIALOG;
      }
      # Emit error message and return unless the command is valid
      unless (defined which($command)) {
        $d->msgbox(text => $errorMsg->render('invalid command'));
        redo DIALOG;
      }
 
      # Ask for stack or queue behavior (Unshift or Push)
      my $addMethod = $d->yesno(
        text => $uiMsg->render('beginning or end?'),
        'yes-label' => $uiMsg->render('beginning'), 
        'no-label' => $uiMsg->render('end')
      ) ? 'unshift' : 'push';
      # Add it to the catalog instance
      $catalog->$addMethod($command);

      # Set the 'catalog changed' flag to true
      $$catalogChanged = true;
    } elsif ($d->state eq 'EXTRA') {
      $commandEntry = mediumCommandFind();
      redo DIALOG;
    }
  }

  return;
}

BEGIN {
  # mediumCommandFind has its own 'static' variable for the current working
  # directory; see perlfaq7
  my $wd = rel2abs(curdir);

  sub mediumCommandFind {
    # Get the filename of a command; start at current directory
    my $commandEntry = $d->fselect(path => $wd, height => 10);

    if ($d->state eq 'OK') {
      # Store new current working directory
      $wd = (splitpath($commandEntry))[1];
      # Return the command
      return $commandEntry;
    } elsif ($d->state eq 'CANCEL') {
      return q();
    }
  }
}

sub mediumRemove {
  my ($catalog, $catalogChanged) = @_;

  my @commands = $catalog->dump; # First, get all the commands out
  # Only in maths can you take something from nothing
  $d->msgbox(text => $errorMsg->render('empty catalog')) and return
    unless @commands;

  my @choices = map {
    # The necessity for this _recodeString kludge really pisses me off...
    $commonMsg->_recodeString($_), 
    [ $commonMsg->_recodeString($commands[$_ - 1]), false ]
  } (1..@commands);
    
  # Get an array of commands to be deleted
  my @deleted = $d->checklist(
    text => $uiMsg->render('remove commands'),
    list => [ @choices ]
  );

  if ($d->state eq 'OK') {
    # And an entrance from everyone's least favorite Perl function
    for my $index (@deleted) { splice @commands, $index - 1 }
    # Generate the new catalog
    $catalog->clear; for my $command (@commands) { $catalog->push($command) }
    # Set the 'catalog changed' flag to true
    $$catalogChanged = true;
  }

  return;
}

sub mediumClear {
  my ($catalog, $catalogChanged) = @_;
  
  # Confirm this destructive operation
  my $confirm = $d->yesno(text => $uiMsg->render('confirm clear?'));

  $catalog->clear and $$catalogChanged = true 
    if $confirm; # "Do it!" -- Yellowman
  
  return;
}

sub mediumSave {
  my $catalogChanged = pop;
  my ($catalog, @path) = @_;
  
  actionCatalogWrite($catalog, @path);
  $d->msgbox(text => $uiMsg->render('configuration saved'));

  # This operation should always be successful...one hopes :)
  $$catalogChanged = false;

  return;
}

BEGIN {
  # templateSelect has its own 'static' variable for the current working
  # directory; see perlfaq7
  my $wd = rel2abs(curdir);
  
  sub templateSelect {
    DIALOG: {
      # Get the filename of a template archive; start at current directory
      my $templateFile = $d->fselect(
        path => "$wd/", height => 10, 'extra-button' => true, 
        'extra-label' => $commonMsg->render('back')
      );

      if ($d->state eq 'OK') {
        # Confirm this destructive operation
        my $confirm = $d->yesno(
          text => $uiMsg->render('confirm installation?')
        );
      
        if ($confirm) { # Do it...
          # Store new current working directory
          $wd = (splitpath($templateFile))[1]; $wd =~ s!/+$!!;
          # Attempt to install the archive and report error if necessary
          unless (templateInstall($templateFile)) {
            $d->msgbox(text => $errorMsg->carp('invalid archive'));
            redo DIALOG;
          }
        }
      } elsif ($d->state eq 'EXTRA') {
        return;
      } elsif ($d->state eq 'CANCEL') {
        exit EXIT_SUCCESS;
      }
    }

    return;
  }
}

# Start the application
mainMenu();
