# -*- project-name: VASM -*-
package VASM::Utility::FileSystem::WindowManager;

use strict;
use warnings;
use base qw/Exporter/;
use constant { false => 0, true => 1 };
use VASM::Resource::Catalog::Message;
use VASM::Utility::FileSystem::Which;
use File::Spec::Functions;
use Carp;
use Cwd;

our $VERSION = '1.02';
our @EXPORT = qw/windowManagerDirectory windowManagerSet/;

# Silence 'use of undefined value' messages
$ENV{T_PX} = q() unless defined $ENV{T_PX};
$ENV{HOME} = q() unless defined $ENV{HOME};

my $xinitrcSystemDir = catdir($ENV{T_PX}, '/etc/X11/xinit');
my $homeDir = catdir($ENV{T_PX}, $ENV{HOME});
my $kdmSessionsSystemDir = catdir(
  $ENV{T_PX}, '/opt/kde/share/apps/kdm/sessions'
);

my %windowManagerPaths = (
  fluxbox => 'startfluxbox',
  'fluxbox-desk' => 'startfluxbox-desk',
  icewm => 'starticewm',
  'icewm-desk', 'starticewm-desk',
  kde => 'startkde',
  windowmaker => 'wmaker',
  xfce4 => 'startxfce4',
  'xfce4-session' => 'xfce4-session'
);

sub windowManagerDirectory {
  # Get the message catalog
  my $directory = messageCatalogFind(qw/xwmset directory/);
  
  for my $wm (keys %windowManagerPaths) {
    $directory->delete($wm) unless which $windowManagerPaths{$wm};
  }

  # What if empty?
  return $directory;
}

my @windowManagerSetters = (
  # xinitrc
  sub {
    my ($wm, $isGlobal) = @_;

    # Check for presence of xinitrc.$wm -- bail out otherwise
    croak "xinitrc for $wm not available" 
      unless -r catfile($xinitrcSystemDir, "xinitrc.$wm");

    my $oldCwd = getcwd; # Save the current wd

    # Establish link to xinitrc
    if ($isGlobal) {
      # Check for writability of the system xinitrc (or containing dir)
      croak 'xinitrc system dir not writable' unless -w $xinitrcSystemDir;
      croak 'system xinitrc not writable'
        unless -w catfile($xinitrcSystemDir, 'xinitrc');
      chdir $xinitrcSystemDir; # cd to the xinitrc system directory
      unlink 'xinitrc'; # remove the existing xinitrc, if any
      # Put up a new symlink to the chosen xinitrc
      symlink "xinitrc.$wm", 'xinitrc';
    } else {
      # Check for writability of home .xinitrc (or containing dir)
      croak 'Home directory not writable' unless -w $homeDir;
      croak 'Home .xinitrc not writable' 
        unless -w catfile($homeDir, '.xinitrc');
      chdir $homeDir; # cd to home directory
      unlink '.xinitrc'; # remove the existing xinitrc, if any
      # Put up a new symlink to the chosen xinitrc
      symlink catfile($xinitrcSystemDir, "xinitrc.$wm"), '.xinitrc';
      # .xsession will alias .xinitrc
      unlink '.xsession'; symlink '.xinitrc', '.xsession';
    }

    chdir $oldCwd; # Reinstate the now old wd

    return;
  },

  # KDM
  sub {
    my ($wm, $isGlobal) = @_;

    # Check for presence of $wm.desktop
    croak "KDM session file for $wm not readable" 
      unless -r catfile($kdmSessionsSystemDir, "$wm.desktop");
    croak 'Home .dmrc unwritable' unless -w catfile($homeDir, '.dmrc');

    # *AFAIK*, it's not possible to set a global dmrc, although it is possible
    # to set a global xinitrc, of course
    unless ($isGlobal) {
      open my $dmrcFH, '>', catfile($homeDir, '.dmrc');
      # [Desktop]
      # Session=foowm
      print $dmrcFH "[Desktop]\nSession=$wm\n"; # Ensure final newline
      close $dmrcFH;
    }

    return;
  }
);

sub windowManagerSet {
  # I was gonna say '$globalp', but this would be lost on people without Lisp
  # experience
  my ($wm, $isGlobal) = @_;

  croak 'Window manager not given' unless defined $wm;
  croak 'Global Boolean argument not given' unless defined $isGlobal;

  for my $setter (@windowManagerSetters) { $setter->($wm, $isGlobal) }
  return;
}

1;
