# -*- project-name: VASM -*-
package VASM::Utility::FileSystem::Which;

use strict;
use warnings;
use constant { false => 0, true => 1 };
use base qw/Exporter/;
use VASM::Utility::Locale;
use File::Which qw//;
use Carp;

our @EXPORT = qw/which whichVasm/;
our $VERSION = '1.04';

sub which {
  my ($commandString) = @_;

  croak 'Command string must be given' unless defined $commandString;
  my @commandComponents = split /\s+/, $commandString; 
  my $commandFound = false;
  
  for my $index (0..$#commandComponents) {
    # Decide whether this is the name of the program
    unless ($commandComponents[$index] =~ /^\w+=/) {
      $commandComponents[$index] = _which($commandComponents[$index]);
      if (defined $commandComponents[$index]) {
        $commandFound = true; last;
      }
    }
  }

  return unless $commandFound;
  return join(' ', @commandComponents);
}

sub _which {
  my ($commandString) = @_;

  croak 'Command string must be given' unless defined $commandString;
  # Find the absolute path; leave as is if it begins with a '/'
  $commandString = File::Which::which($commandString)
    unless $commandString =~ m#^/#;
  return $commandString if 
    defined $commandString and 
    -x $commandString and
    -f $commandString;
  return;
}

sub whichVasm {
  my ($programName) = @_;

  if (length $ENV{DISPLAY}) {
    # See if we can find a matching gtk2 program
    my $programMatching = which("${programName}-gtk2.pl");
    
    if (defined $programMatching) { # If such a program was found
      my @locale = localeParse; my $locale;
      # If locale was something like ru_RU, it can be suffixed with '.UTF-8'
      $locale = join('_', @locale) . '.UTF-8' if @locale == 2;
      # Construct the command line from this, if possible
      if (defined $locale) {
        return "LC_ALL=$locale $programMatching";
      } else {
        return $programMatching;
      }
    }
  } else { # DISPLAY was not set
    # See if we can find a matching cdialog program
    my $programMatching = which("${programName}-cdialog.pl");
    # Don't fuss with the locale
    return $programMatching if $programMatching;
  }

  # Just try it unadorned with -type here
  my $programMatching = which("${programName}.pl");
  return $programMatching; # undefined if unsuccessful
}

1;

__END__
