# -*- project-name: VASM -*-
package VASM::Utility::FileSystem::User;

use strict;
use warnings;
use base qw/Exporter/;
use constant { false => 0, true => 1 };
use VASM::Utility::FileSystem::Which;
use VASM::Utility::FileSystem::Group;
use Carp;

our $VERSION = '1.03';
our @EXPORT = qw/userAdd userDel userExists/;

sub userAdd {
  my (%userAttrs) = @_;

  # Check completeness: name, groups (listref may be empty), shell
  croak 'User attributes incomplete'
    unless (grep { defined $userAttrs{$_} } qw/userName shell groups/) == 3;
  # Check for existence of 'useradd' utility
  croak 'useradd not available' unless defined which('useradd');

  ## Other checks
  # Username
  croak 'Blank user name' unless length $userAttrs{userName};
  croak 'Invalid user name' if $userAttrs{userName} =~ /[^[:alnum:]_-]/;
  croak 'Username taken' if userExists($userAttrs{userName});
  # Group
  my %groupHash = groupHash;
  for my $group (@{ $userAttrs{groups} }) {
    croak "Invalid group name $group" 
      if $group =~ /[^[:alnum:]_-]/ or not defined $groupHash{$group}
  }
  # Comment
  croak 'Illegal characters in comment field' if $userAttrs{comment} =~ /[:']/;
  # Shell
  croak 'Blank shell entry' unless length $userAttrs{shell};
  croak 'Illegal characters in shell field' if $userAttrs{shell} =~ /[:']/;
  croak 'Shell unavailable' unless defined which($userAttrs{shell});

  # Now run useradd with the specified options; the user will be given a blank
  # password by default; this is obviously insecure, so suggest change
  # immediately!
  my $passwordDefault = crypt(
    q(), join(q(), ('A'..'Z', 'a'..'z', '0'..'9', '.', '/')[rand 64, rand 64])
  );

  my $commandString = 'useradd -m';
  $commandString .= " -c '$userAttrs{comment}'";
  $commandString .= ' -G ' . join(',', @{ $userAttrs{groups} })
    if @{ $userAttrs{groups} };
  $commandString .= " -s $userAttrs{shell}";
  $commandString .= " -p $passwordDefault";
  $commandString .= " $userAttrs{userName}";

  # There should be no reason to worry about this failing by now
  system "$commandString &> /dev/null";

  return;
}

# Be careful!
sub userDel {
  my (%userAttrs) = @_;

  # Check completeness of attributes hash
  croak 'User attributes incomplete'
    unless (grep { defined $userAttrs{$_} } qw/userName removeFiles/) == 2;
  # Check for existence of 'userdel' utility
  croak 'useradd not available' unless defined which('userdel');

  ## Other checks
  # User name
  croak 'Blank user name' unless length $userAttrs{userName};
  croak 'Invalid user name' if $userAttrs{userName} =~ /[^[:alnum:]_-]/;
  croak 'User does not exist' unless userExists($userAttrs{userName});

  # Run userdel
  my $commandString = 'userdel';
  $commandString .= ' -r' if $userAttrs{removeFiles};
  $commandString .= " $userAttrs{userName}";

  # There should be no reason to worry about this failing by now
  system "$commandString &> /dev/null";

  return;
}

sub userExists {
  my ($userName) = @_;
  
  while (my $userNameExisting = (getpwent)[0]) {
    return true if $userName eq $userNameExisting;
  }
  
  return false;
}

# userMod(%)

1;
