# -*- project-name: VASM -*-
package VASM::Utility::FileSystem::Service;

use strict;
use warnings;
use base qw/Exporter/;
use constant { false => 0, true => 1 };
use VASM::Resource::Catalog::Message;
use VASM::Utility::FileSystem::Which;
use File::Spec::Functions qw/catdir catfile splitpath/;
use Carp;

our $VERSION = '1.0';
our @EXPORT = qw/serviceDirectory serviceOnOffHash serviceSet/;

# Silence 'use of undefined value' messages
$ENV{T_PX} = q() unless defined $ENV{T_PX};

my $serviceDir = catdir($ENV{T_PX}, '/etc/rc.d');
my $serviceDirTemplate = catdir($serviceDir, 'init.d');

sub serviceDirectory {
  return messageCatalogFind(qw/srvset directory/);
}

sub serviceOnOffHash {
  my ($bootLevel) = @_;

  croak 'Boot level invalid' unless 
    defined $bootLevel and 
    $bootLevel =~ /^\d$/ and
    not $bootLevel =~ /^[016]$/;

  # The service directory particular to the boot level in question
  my $serviceDirBootLevel = catdir($serviceDir, "rc${bootLevel}.d");

  croak 'Template service directory unavailable' unless -x $serviceDirTemplate;
  croak 'Boot level service directory unavailable'
    unless -x $serviceDirBootLevel;
 
  # Generate hash with default value of off for all services
  my %serviceOnOffHash = map {
    (splitpath($_))[2] => false
  } grep {
    -r $_ and not -d $_
  } glob "$serviceDirTemplate/*";

  # Turn them on according to presence in the boot level directory...if the
  # link is broken, elide it from @bootLevelServices (ran into this problem
  # earlier)
  my @bootLevelServices = map {
    (splitpath($_))[2] =~ /([[:alpha:]]+)$/;
  } grep {
    -r $_
  } glob "$serviceDirBootLevel/S*";
  for my $service (@bootLevelServices) { $serviceOnOffHash{$service} = true }

  return %serviceOnOffHash; # Return hash of services
}

sub serviceSet {
  my ($bootLevel, %services) = @_;

  croak 'Boot level invalid'
    unless defined $bootLevel and
    $bootLevel =~ /^\d$/ and
    not $bootLevel =~ /^[016]$/;
  croak 'No services given to enable or disable'
    unless keys %services;

  # The service directory particular to the boot level in question
  my $serviceDirBootLevel = catdir($serviceDir, "rc${bootLevel}.d");

  croak 'Template service directory unavailable' unless -x $serviceDirTemplate;
  croak 'Boot level service directory unavailable'
    unless -x $serviceDirBootLevel;

  # Croak unless 'service' is available
  my $servicePath = which('service');
  croak 'service utility unavailable'
    unless defined $servicePath and -x $servicePath;

  # Iterate over services and enable/disable them if possible
  for my $service (keys %services) {
    # Check for the existence of the service first...croak if it isn't there
    croak "Requested service $service unavailable"
      unless -r catfile($serviceDirTemplate, $service);
    # If it fell through to here, enable/disable service; with all the
    # subjunctive croak statements up to this point, there is no reason to
    # worry about failure :)
    system(
      'service ' .
      ($services{$service} ? '-s' : '-r') .
      " $service $bootLevel &> /dev/null"
    );
  }

  return;
}

1;

__END__
