# -*- project-name: VASM -*-
package VASM::Utility::FileSystem::Mouse;

use strict;
use warnings;
use base qw/Exporter/;
use constant { false => 0, true => 1 };
use VASM::Resource::Catalog::Message;
use VASM::Resource::Catalog::XorgConfig;
use File::Spec::Functions qw/catdir catfile abs2rel/;
use Carp;
use Cwd;

our $VERSION = '1.01';
our @EXPORT = qw/mouseProbe mouseSet mouseDirectory/;

# ASSumptions:
#   - Primary input device is mouse0 or input0
#   - 'usb' can always be found in output of udevinfo for such a primary input
#   device, if it is USB
#   - 'psmouse' can always be found in output of udevinfo for PS/2 mice
#   - Wheel mice are always imps2

# Be careful with $mouseProperties{type}...it may have to be switched around
# for some settings

# Silence 'use of undefined value' messages
$ENV{T_PX} = q() unless defined $ENV{T_PX};

my $gpmConfigFile = catfile($ENV{T_PX}, '/etc/rc.d/init.d/gpm');
my $devDir = catdir($ENV{T_PX}, '/dev');
my $usbDevicesFile = catfile($ENV{T_PX}, '/proc/bus/usb/devices');
# my $sysDir = catdir($ENV{T_PX}, '/sys');

# A test battery that will attempt to determine the mouse type
my @mouseProbeTests = (
  # USB mouse
  sub {
    my ($udevinfoProcess) = @_;
    
    my $matchingLines = grep { $_ =~ /usb/ } <$udevinfoProcess>;
    return 'usb' if $matchingLines > 0;
    return; # No USB mouse
  },
  
  # PS/2 mice
  sub {
    my ($udevinfoProcess) = @_;
    
    my $matchingLines = grep { $_ =~ /psmouse/ } <$udevinfoProcess>;
    return 'ps2' if $matchingLines > 0;
    return; # No PS/2 mouse
  }
);

# Look under '/class/input/mouse0' and '/class/input/input0', which /appear/
# to be paths for primary input devices
my @mousePrimaryDevices = qw/mouse0 input0/;

sub mouseProbe {
  # Run through mouse test battery
  for my $mouseTest (@mouseProbeTests) {
    # For each primary input device, run the test
    for my $mouseDevice (@mousePrimaryDevices) {
      # Construct a path for udevinfo -- necessary to prefix /sys?
      my $udevDevicePath = catdir('/class/input', $mouseDevice);
      # Try to open process udevinfo -a -p $dev; return if unsuccessful;
      # udevinfo is not available!
      open my $udevinfoProcess, "udevinfo -a -p $udevDevicePath |" or return;
      my $mouseType = $mouseTest->($udevinfoProcess); # Run mouse test here
      close $udevinfoProcess; # Close filehandle
      # If test was successful
      if (defined $mouseType) {
        return (
          device => catfile($devDir, 'input', $mouseDevice),
          type => $mouseType
        );
      }
    }
  }

  return; # Unsuccessful
}

my @mouseSetters = (
  # /dev/mouse link
  sub {
    my (%mouseProperties) = @_;
    
    # check for writability of device directory
    croak 'Device directory unwritable' unless -w $devDir;
    my $oldCwd = getcwd; # save the current wd
    chdir $devDir; # cd to device directory
    unlink 'mouse'; # remove the existing mouse link, if any
    symlink abs2rel($mouseProperties{device}), 'mouse'; # put up new link
    chdir $oldCwd; # reinstate the old wd
    
    return;
  },
  
  # GPM
  sub {
    my (%mouseProperties) = @_; my $gpmConfigFH;

    # check for writability of GPM config file
    croak 'GPM config file not writable' unless -w $gpmConfigFile;
    
    # Open the GPM config file
    open $gpmConfigFH, '<', $gpmConfigFile;
    
    # Frob together the new GPM file contents
    my @gpmConfigNew = map {
      s/-t \w+/-t $mouseProperties{type}/ if /^OPT=/; $_;
    } <$gpmConfigFH>;
    close $gpmConfigFH; # Clean up $gpmConfigFH

    # Print them out
    open $gpmConfigFH, '>', $gpmConfigFile; # Should still be there
    for my $line (@gpmConfigNew) { print $gpmConfigFH $line }
    close $gpmConfigFH;

    return;
  },

  # XorgConfig
  sub {
    my (%mouseProperties) = @_;
    
    # Load the Xorg config catalog
    my $catalog = xorgConfigCatalogFind;
    # Iterate over the keys
    for my $property (keys %mouseProperties) {
      my $value = $mouseProperties{$property};
      # If $value is a listref, expand it; otherwise let it remain as is
      $catalog->store($property, ref $value ? @{ $value } : $value);
    }
    # Now write the catalog back out
    xorgConfigCatalogWrite($catalog);
    
    return;
  }
);

sub mouseSet {
  my (%mouseProperties) = @_;

  my $propertyNumber = grep {
    defined $mouseProperties{$_}
  } qw/device type emulateThreeButtons/;

  # %mouseProperties must contain the full complement of attributes, except
  # zAxisMapping
  croak 'Mouse properties hash incomplete' unless $propertyNumber == 3;

  # Run through setters, passing %mouseProperties each time
  for my $setter (@mouseSetters) { $setter->(%mouseProperties) }
  return;
}

sub mouseDirectory {
  return messageCatalogFind(qw/mouseset directory/);
}

1;

__END__
