# -*- project-name: VASM -*-
package VASM::Utility::Expect::Password;

use strict;
use warnings;
use constant +{
  true => 1,
  passwordOld => 0,
  passwordNew => 1,
  passwordConfirm => 2,
  success => 3
};
use base qw/Exporter/;
use Carp;
use Expect;

our $VERSION = '1.04';
our @EXPORT = qw/passwordOld passwordNew passwordConfirm success/;

# Don't show the stdout of programs; it looks sloppy.
# $Expect::Log_Stdout = 0;

sub new {
  my ($self, $username) = @_;

  # Assume $> for username -- if username is given, validate it; if that
  # fails, return
  if (defined $username) {
    return unless getpwnam $username;
  } else {
    $username = getpwuid $>;
  }

  # The key 'status' is the current stage in the transaction (e.g., waiting
  # for a new password), and 'username' is of course the user for whom we are
  # configuring
  my $instance = { status => passwordOld, username => $username };
  bless $instance, $self;
  $instance->instantiateExpect;
  
  return $instance;
}

my @submitMethods = (
  # From passwordOld to passwordNew
  sub {
    my ($self, $password) = @_;

    # Five seconds seems to be a reasonable timeout to receive an answer
    defined $self->{expect}->expect(5, [ qr/^Old password: / ])
      or croak 'Expect pattern match failed';
    $self->{expect}->send("$password\n");

    # If not running as root, check for the 'New password' prompt immediately;
    # undef is there to tell the method explicity that the user is not root.
    $self->checkNewPasswordPrompt(undef) unless getpwuid $> eq 'root';
    
    return;
  },
  # From passwordNew to passwordConfirm
  sub {
    my ($self, $password) = @_;

    # If running as root, check for the 'New password' prompt; the true flag
    # indicates that the user *is* root. See above.
    $self->checkNewPasswordPrompt(true) if getpwuid $> eq 'root';

    # Send the new password
    $self->{expect}->send("$password\n");

    # Expect either 'Bad password' or 'Re-enter new password'
    my @patterns = ([ qr/^Re-enter new password: / ],
                    [ qr/^The password for $self->{username} is unchanged/ ],
                    [ qr/^Bad password: / ]);
    my $patidx = $self->{expect}->expect(5, @patterns);

    if ($patidx == 1) {
      # Successful -- advance to the next stage
      $self->{status}++;
    } elsif ($patidx == 2) {
      # 'The password for foo is unchanged' -- unsuccessful
      $self->instantiateExpect; # Generate a new Expect instance
    } elsif (not defined $patidx) {
      croak 'Expect pattern match failed';
    }

    return;
  },
  # From passwordConfirm to success
  sub {
    my ($self, $password) = @_;

    # Send the confirmation of the new password
    $self->{expect}->send("$password\n");

    # Expect either:
    # 'Password changed' (success)
    # 'The password for $self->{username} unchanged' (too many tries)
    # 'q{They don't match}' (an unsuccessful try)
    my @patterns = ([ qr/^Password changed/ ],
                    [ qr/^The password for $self->{username} unchanged/ ],
                    [ qr/^They don't match; try again/ ]);
    my $patidx = $self->{expect}->expect(5, @patterns);
    
    if ($patidx == 1) {
      # 'Password changed' -- successful
      $self->{status}++;
      # Wait for EOF and close the Expect instance
      $self->{expect}->expect;
      $self->{expect}->soft_close;
    } elsif ($patidx == 2) {
      # 'The password for foo unchanged' -- unsuccessful
      # Generate a new Expect instance
      $self->instantiateExpect;
    } elsif ($patidx == 3) {
      # q{They don't match; try again}
      # The process will now be awaiting the new password again
      $self->{status} = passwordNew;
    } elsif (not defined $patidx) {
      croak 'Expect pattern match failed';
    }

    return;
  }
);

sub checkNewPasswordPrompt {
  my ($self, $isRoot) = @_;
  
  # Pick up where last method left off, if necessary: old password matched?
  my @patterns = ([ qr/^New password: / ], [ qr/^Incorrect password/ ]);
  my $patidx = $self->{expect}->expect(5, @patterns);

  if ($patidx == 1) {
    # Phra r3wt will start at the passwordNew stage, so don't increment him
    $self->{status}++ unless $isRoot;
  } elsif ($patidx == 2) {
    $self->instantiateExpect;
  } else {
    croak 'Expect pattern match failed';
  }

  return;
}

sub submit {
  my ($self, $password) = @_;

  # There can be no advancement after completion
  return success if $self->status == success;
  
  # Call the submit method associated with the current status
  my $method = $submitMethods[$self->status];
  my $success = $self->$method($password);

  # Return the current status
  return $self->status;
}

sub status {
  my ($self) = @_;
  
  return $self->{status};
}

sub username {
  my ($self) = @_;
  
  return $self->{username};
}

sub instantiateExpect {
  my ($self) = @_;
  
  # Create a new Expect instance for verifying/establishing the password
  $self->{expect} = Expect->new; 
  # Disable echoing and translation
  $self->{expect}->raw_pty(true);

  # Initiate prompt for old password
  {
    # Localized strings from passwd will screw up my Expect patterns
    local $ENV{LC_ALL} = 'C';
    $self->{expect}->spawn("passwd $self->{username}")
      or croak 'Egad! Cannot spawn passwd';
  }

  # The instance will be waiting for the old password then
  $self->{status} = getpwuid $> eq 'root' ? passwordNew : passwordOld;

  return;
}

sub _validateUsername {
  my ($username) = @_;

  # One needs to be root to change the password of a user other than oneself
  return 'not-root' unless getpwuid $> eq 'root';
  # The user needs to exist as well...
  return 'non-existent' unless getpwnam $username;

  return;
}

1;
