# -*- project-name: VASM -*-
package VASM::Tree;

use strict;
use warnings;
use Carp;
use Tie::Hash::Indexed;

our $VERSION = '1.05';

sub new {
  my ($self) = @_;

  # Ain't Perl grand?
  return bless {}, $self;
}

sub store {
  croak 'Insufficient number of arguments given!'
    unless @_ >= 3; # There must be a $datum and one index at least

  # Indices should be self-explanatory be now; $datum is the object to be
  # stored.
  my ($self, $datum, @indices) = (shift, pop, @_);

  nodeTraverse($self, @indices)->{datum} = $datum;
  
  return; # This is a true subroutine...
}

sub retrieve {
  my ($self, @indices) = @_;

  croak 'At least one index must be given!' unless @indices;
  return nodeTraverse($self, @indices)->{datum};
}

sub delete {
  my ($self, @indices) = @_;

  undef %{ nodeTraverse($self, @indices) };
  # Wipe it from the parent's children, if necessary
  if (@indices) {
    my $tail = pop @indices;
    delete nodeTraverse($self, @indices)->{children}->{$tail};
  }

  return; # Likewise a subroutine...
}

# "Do I want one? God forbiddee!" -- Ogden Nash
sub children {
  my ($self, @indices) = @_;

  # If @indices is empty, this gets the children at the root 
  return keys %{ nodeTraverse($self, @indices)->{children} };
}

sub nodeTraverse {
  my ($node, @indices) = @_; # The @indices through which we will traverse

  # If there were no arguments, just return the node that was passed in.
  return $node unless @indices;

  # Traverse through the tree of nodes, dynamically creating new ones if
  # necessary.
  for my $index (@indices) {
    # Create a new children node if necessary, but make it an instance of
    # Tie::Hash::Indexed
    tie %{ $node->{children} = {} }, 'Tie::Hash::Indexed'
      unless defined $node->{children};
    # Also create a new container node
    $node->{children}->{$index} = {}
      unless defined $node->{children}->{$index};
    
    $node = $node->{children}->{$index};
  }

  return $node;
}

1;

__END__

=head1 NAME

VASM::Tree - simple abstracted support for trees of hashrefs

=head1 SYNOPSIS

    use VASM::Tree;

    # Not to be confused with my $ek or my $to
    my $tree = VASM::Tree->new;
    
    # Store some things
    $tree->store(qw/foo bar baz/, 'nonsense names');
    $tree->store(qw/Brahma Vishnu Shiva/, 'Trimurti');
    $tree->store(qw/monkey god/ , 'Hanuman');
    $tree->store('another', 'thing');
    $tree->store(qw/foo fighters/, [ qw/UFO music/ ]);
    
    # Retrieve an element
    print foreach @{ $tree->retrieve(qw/foo fighters/) };
    
    # Delete everything under 'foo'
    $tree->delete(qw/foo/);
    
    # Children preserve order in which they were inserted
    print for $tree->children;

=head1 DESCRIPTION

An instance of VASM::Tree defines a tree of hashrefs in which one may
transparently store and retrieve elements with no direct obligation to
allocate new nodes in the tree. Moreover, VASM::Tree uses nodes tied to
Tie::Hash::Indexed, so the trees maintains the order of its elements at any
given level in the tree. (This is occasionally useful!)

=head1 METHODS

=over

=item new

This constructs a new VASM::Tree and returns the blessed reference (or, more
properly, a reference to the blessed referent). It accepts no arguments. In
fact, it's three or four lines long.

=item store(@indices, $datum)

store accepts a list of arguments whose last item is a scalar to store, and
whose preceding items are all indices in the hashref tree. It returns nothing.

=item retrieve(@indices)

retrieve accepts a list of indices in the tree from which to retrieve a
scalar. One could say it is the qualitative inverse of store.

=item delete([ @indices ])

=item children([ @indices ])

children accepts an optional list of indices and returns a list of the
children nodes beneath it, retaining the order of internment. (Children?
Branches? Mixing botanical and familiar metaphors? Yes, but I don't want to go
around changing the interface now.) If no indices are given, children returns
the list of children at the root, whose parent node (root, maybe?) is
nameless.

=back

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote VASM::Tree.

=cut
