# -*- project-name: VASM -*-
package VASM::Resource::Catalog::Message;

use strict;
use warnings;
use base qw/Exporter/;
use VASM::Catalog::Message;
use VASM::Catalog::Error;
use VASM::Resource;
use I18N::Langinfo qw/langinfo CODESET/;
use Carp;

our @EXPORT = qw/messageCatalogFind errorCatalogFind messageCatalogWrite/;
our $VERSION = '1.08';

sub findCatalog {
  my %args = @_;

  my @path = @{ $args{path} }; # Look under the resource path given
  my @locale = localeParse(); # Finagle locale from %ENV
  my $encoding = langinfo(CODESET); # Finagle locale encoding as well
  undef $encoding if $encoding eq 'UTF-8'; # No need to recode

  # Now, try to find the first matching message.xml in the hierarchy; the
  # search can go as far back as the base @path itself:
  my $fh;
  LOOP: {
    do {
      $fh = (dataResourceFind(@path, @locale, 'message.xml'))[0];
      last LOOP if defined $fh;
    } while (pop @locale);
  }

  # Return nothing unless a file was found, otherwise:
  croak 'Could not find a matching message catalog!' unless $fh;

  $fh->binmode(':utf8'); # Set the IO::File to UTF-8 for XML

  my $catalog; # Switch catalog type depending on request
  # Build argument list for constructor
  my @arglist = (handle => $fh);
  push @arglist, encoding => $encoding if defined $encoding;

  if ($args{type} eq 'message') {
    $catalog = VASM::Catalog::Message->new(@arglist);
  } elsif ($args{type} eq 'error') {
    $catalog = VASM::Catalog::Error->new(@arglist);
  }

  # And clean up
  $fh->close;

  return $catalog;
}

sub messageCatalogFind {
  return findCatalog(path => [ @_ ], type => 'message');
}

sub errorCatalogFind {
  return findCatalog(path => [ @_ ], type => 'error');
}

sub messageCatalogWrite {
  my ($instance, @path) = @_;
  my @locale = localeParse(); # Finagle locale from %ENV
  
  # Generate a new IO::File for the catalog to be written. dataResourceCreate
  # (actually, resourceCreate) will croak if unsuccessful, so no error checking
  # is really necessary.
  my $fh = (dataResourceCreate(@path, @locale, 'message.xml'))[0];
  $instance->write($fh); # Now let the instance do its thing
  $fh->close; # ...close the filehandle and...done
  
  return;
}

1;

__END__

=head1 NAME

VASM::Resource::Catalog::Message - load message catalogs with the VASM
resources framework

=head1 SYNOPSIS

    use VASM::Resource::Catalog::Message;
    
    # Say LANG=de, and there is a catalog in
    # /usr/share/vasm/some/project/message/de/message.xml
    my $catalog = messageCatalogFind(qw/some project message/);
    # See VASM::Catalog::Message
    $catalog->render('hello');
    # => Guten Tag!

    # Write out the message catalog back out -- messageCatalogWrite will
    # interpolate 'de' between the path given above and the catalog name
    messageCatalogWrite($catalog, qw/some project message/);

=head1 DESCRIPTION

VASM::Resource::Catalog::Message enables the programmer to use
VASM::Catalog::Message i18n message catalogs with the resource hierarchy
defined by L<VASM::Resource>. It will search for a message catalog under the a
given resource path, in the fashion prescribed by the XDG Base Directory
standard, used by VASM::Resource. More specifically, VASM::Catalog::Message
will attempt to discern the current locale, and consequently, the appropriate
message file from environment variables that signify the locale (see
L<"VARIABLES"> below). If a message catalog for a given language/country pair
cannot be found, the search will default to a catalog for the language
independent of any national dialect, and, failing that, to a default catalog
most likely written in English.

=head1 FUNCTIONS

=over

=item messageCatalogFind(@path)

messageCatalogFind accepts a list describing a resource path and seeks a
message catalog therein, according to the rules described above. If
successful, it will return an instance of VASM::Catalog::Message after using
it to digest the catalog. Otherwise, it will return nothing.

=item errorCatalogFind(@path)

errorCatalogFind generally acts the same as messageCatalogFind, except that it
promotes the message catalog file to a VASM::Catalog::Error object.

=item messageCatalogWrite($catalog, @path)

This function writes out a message catalog instance to a given resource path,
and requires no counterpart for error messages, as the written format of a
message catalog does not distinguish between the two. Just as
messageCatalogFind and errorCatalogFind attempt to interpolate the current
locale name between the resource path and file, so will messageCatalogWrite
interpolate the current locale between the argument resource path and the file
name, message.xml. In this way, reads will always correspond to writes,
assuming a consistent value for the locale.

=back

=head1 VARIABLES

These environment variables are searched in order for locale information, and
their values must conform to the POSIX convention consisting of a lowercase
two-letter language code optionally followed by an underscore and an uppercase
two-letter country code, e.g., hi_IN. Encoding information qualifiers such as
'.utf8' are ignored, because the VASM project strictly adheres to UTF-8:

=over

=item *

LC_ALL

=item *

LC_MESSAGES

=item *

LANG

=back

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote
VASM::Resource::Catalog::Message. Further considerations by cintyram.

=cut
