# -*- project-name: VASM -*-
package VASM::Resource::Catalog::Menu;

use strict;
use warnings;
use base qw/Exporter/;
use VASM::Catalog::Menu;
use VASM::Resource;
use Carp;

our @EXPORT = qw/menuFind menuWrite menuTranslate/;
our $VERSION = '1.08';

sub menuFind {
  my @locale = localeParse(); # Finagle locale from %ENV

  # Now, try to find the first matching menu.xml in the hierarchy; the search
  # can go as far back as the base menu/source directory itself:
  my $handle;
  LOOP: {
      do {
        $handle = (configResourceFind(qw/menu source/, @locale, 'menu.xml'))[0];
        last LOOP if defined $handle;
      } while (pop @locale);
    }

  # Return nothing unless a file was found, otherwise:
  # carp 'Could not find a matching menu file!'
  return VASM::Catalog::Menu->new unless defined $handle;
  # Digest file, if possible, and return the new menu definition
  $handle->binmode(':utf8'); # Set the IO::File to UTF-8 for XML
  my $menu = VASM::Catalog::Menu->new($handle);
  $handle->close;
  
  return $menu;
}

sub menuWrite {
  my ($instance) = @_;
  my @locale = localeParse(); # Finagle locale from %ENV

  # Generate a new IO::File for the catalog to be written. configResourceCreate
  # (actually, resourceCreate) will croak if unsuccessful, so no error checking
  # is really necessary
  my $menuFH = (configResourceCreate(qw/menu source/, @locale, 'menu.xml'))[0];
  $instance->write($menuFH); # Now let the instance do its thing
  $menuFH->close; # ...close the filehandle and...done
  
  return;
}

# If this accepts an instance of the menu class as a first argument, then why
# not make it OO? Because there is nothing fundamentally OO in nature about
# VASM::Resource. Otherwise, I would have
sub menuTranslate {
  my ($instance, $wm) = @_;

  # Check if the appropriate method exists
  my $method = "translate_$wm";
  croak 'No translator exists for this window manager!'
    unless $instance->can($method);
  
  # Write out the menu
  # What about locales?
  my $menuFH = (configResourceCreate(qw/menu translation source/, $wm))[0];
  $instance->$method($menuFH);
  $menuFH->close; # ...and done

  return;
}

1;

=head1 NAME

VASM::Resource::Catalog::Menu

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 FUNCTIONS

=head1 VARIABLES

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote VASM::Resource::Catalog::Menu.

=cut
