# -*- project-name: VASM -*-
package VASM::Resource::Catalog::Action;

use strict;
use warnings;
use Cwd;
use constant { false => 0, true => 1 };
use base qw/Exporter/;
use VASM::Resource;
use VASM::Catalog::Action;
use VASM::Resource::Catalog::Message;
use File::MimeInfo::Magic qw/magic/;
use Archive::Tar;
use Carp;
use URI;

our @EXPORT = qw/actionCatalogFind
                 actionCatalogFindByMedium
                 actionCatalogWrite
                 mediaDirectory
                 templateInstall/;
our $VERSION = '1.12';

sub actionCatalogFind {
  # The resource path in which to find the action catalog
  my @path = @_;

  # Find the matching file with the highest precedence
  my $fh = (configResourceFind('action', @path, 'action.xml'))[0];

  # Not necessarily an error
  # carp 'Could not find a matching action catalog!'
  return VASM::Catalog::Action->new unless $fh;
  # Digest file, if possible, and return the new menu definition
  $fh->binmode(':utf8'); # Set the IO::File to UTF-8 for XML
  my $catalog = VASM::Catalog::Action->new($fh);
  $fh->close;

  return $catalog;
}

sub actionCatalogFindByMedium {
  # A URI or absolute local filename. Note that 'file:///home/hanumizzle' is
  # different from '/home/hanumizzle', because many programs do not accept the
  # URI form (such as qiv, for instance).
  my ($path) = @_; my ($scheme, $mimeType, $catalog);

  # If $path is a valid URI, good, try to find an actions catalog for it:
  if (defined ($scheme = URI->new($path)->scheme())) {
    $catalog = actionCatalogFind('uri', $scheme);
  } else { # By elimination, assume that this is a file
    # The file must exist for magic detection to work, obviously
    croak 'MIME type unavailable or file non-existent!'
      unless defined ($mimeType = magic($path));
    $catalog = actionCatalogFind('file', (split m!/!, $mimeType));

    # If that fails, check each superclass of the mimetype
    unless (defined $catalog) {
      for my $superclass (File::MimeInfo::mimetype_isa($mimeType)) {
        $catalog = actionCatalogFind('file', (split m!/!, $superclass));
        # Exit if satisfied
        last if defined $catalog;
      }
    }
  }

  return $catalog; # Will be undef if nothing was found
}

sub actionCatalogWrite {
  my ($instance, @path) = @_;
  
  # Generate a new IO::File for the catalog to be written.
  # configResourceCreate (actually, resourceCreate) will croak if unsuccessful, so
  # no error checking is really necessary.
  my $file = (configResourceCreate('action', @path, 'action.xml'))[0];
  $instance->write($file); # Now let the instance do its thing
  $file->close; # ...close the filehandle and...done
  
  return;
}

# Return a VASM::Catalog::Message object describing all media types usable by
# VASM::Catalog::Action, both for MIME and URI data:
sub mediaDirectory {
  my ($category) = @_;
  
  croak 'Media category must be given' unless defined $category;
  my $directory = messageCatalogFind(qw/action directory/, $category);

  # The benefits of encapsulation
  return $directory;
}

sub templateInstall {
  my ($templateFile) = @_;
  
  # Check the MIME type of the file
  my $mimeType = magic($templateFile);

  # Return false unless the MIME type is a (compressed) tar
  unless ($mimeType =~ m!application/x-(?:(?:compressed-)?tar|gzip)$!) {
    return false;
  }

  # Find the resource path and create it if necessary
  my $templateDir = configResourcePathCreate('action');
  # Save the /old/ current directory and umask
  my $oldCwd = getcwd; my $old_umask = umask;
  # Change directory and extract the template archive with correct permissions
  # (-rw-------)
  umask 0077; chdir $templateDir; Archive::Tar->extract_archive($templateFile);
  # Reinstate $oldCwd and $old_umask
  chdir $oldCwd; umask $old_umask;

  # Report success
  return true;
}

1;

__END__

=head1 NAME

VASM::Resource::Catalog::Action - load action catalogs with the VASM resources
framework

=head1 SYNOPSIS

    use VASM::Resource::Catalog::Action;

=head1 DESCRIPTION

=head1 FUNCTIONS

=over

=item actionCatalogFind(@path)

=item actionCatalogFindByMedium($path)

=item actionCatalogWrite($catalog, @path)

=item mediaDirectory

=back

=head1 AUTHORS

hanumizzle L<hanumizzle@gmail.com> wrote VASM::Resource::Catalog::Action.
Further considerations by cintyram.

=cut
