# -*- project-name: VASM -*-
package VASM::Catalog::XorgConfig;

use strict;
use warnings;
#use base qw/VASM::Catalog::XorgConfig::Translation/;
use VASM::Tree;
use XML::Parser;
use XML::Writer;
use IO::File;
use Carp;

# Should hardware detection go in separate package? Yes. System interaction
# should be decoupled from data manipulation. This class is just for frobbing.

our $VERSION = '1.02';

sub new {
  my ($self, $handle) = @_; my $instance = {};

  $instance->{catalog} = VASM::Tree->new;
  bless $instance, $self;

  # If a file handle was given to parse:
  defined $handle and $instance->parse($handle);

  return $instance;
}

sub _addValue {
  # $key represents a particular aspect of X configuration, such as 'hsync'.
  # and $value contains the values associated with it
  my ($self, $key, $value, $addFunc) = @_;
  
  # Can't have any of that...
  croak 'Key and value must be given' unless defined $key and defined $value;

  # Store a blank arrayref at the key unless there is something there already
  unless (defined $self->{catalog}->retrieve($key)) {
    $self->{catalog}->store($key, []);
  }

  # Now add the value onto the list
  if ($addFunc eq 'push') {
    push @{ $self->{catalog}->retrieve($key) }, $value;
  } elsif ($addFunc eq 'unshift') {
    unshift @{ $self->{catalog}->retrieve($key) }, $value;
  } else {
    croak 'Unknown add function';
  }

  return;
}

sub Push {
  my ($self) = shift;
  
  $self->_addValue(@_, 'push');
}
 
sub unshift {
  my ($self) = shift;
  
  $self->_addValue(@_, 'unshift');
}

sub store {
  # $key represents a particular aspect of X configuration, such as 'hsync'.
  # @values contains the values associated with it
  my ($self, $key, @values) = @_;
  
  croak 'Key must be given' unless defined $key;
  # It is now very simple to store the list in the catalog itself
  $self->{catalog}->store($key, [ @values ]);

  return;
}

sub clear {
  my ($self, $key) = @_;

  # Stores empty arrayref
  $self->store($key);

  return;
}

sub retrieve {
  my ($self, $key) = @_;

  # Check for key
  croak 'Key must be given' unless defined $key;

  # If the key is defined, return a copy of its array
  my $values = $self->{catalog}->retrieve($key);

  return @{ $self->{catalog}->retrieve($key) }
    if defined $values;
  return;
}

sub list {
  my ($self) = @_;
  
  return $self->{catalog}->children;
}

sub _parseValueList {
  my ($self, $key, $tags) = @_; my @values;
 
  for my $index (grep { $_ % 2 } (0..$#{ $tags })) {
    next if $tags->[$index] eq '0'; # Skip text elements

    if ($tags->[$index] eq 'value') {
      # The textual content of the tag must have a length greater than 0
      croak 'Invalid or blank content in <value> tag'
        unless $tags->[$index + 1]->[1] eq '0'
          and length $tags->[$index + 1]->[2];
      # Add the value to @values in order
      push @values, $tags->[$index + 1]->[2];
    } else {
      croak 'Invalid tag in Xorg config catalog';
    }
  }

  # Intern the digested values
  $self->store($key, @values);
  
  return;
}

sub parse {
  my ($self, $handle) = @_;
  # If this fails, XML::Parser will emit an error message and end the program.
  # Unfortunately, this message is always English. :( There is at least an
  # error context, which can be universally understood by the programmer.
  my $parser = XML::Parser->new(Style => 'Tree', ErrorContext => 3);
  my $tree = $parser->parse($handle); my $tags = $tree->[1];

  # Main loop: iterate through tags under <catalog>
  for my $index (grep { $_ % 2 } (0..$#{ $tags })) {
    next if $tags->[$index] eq '0'; # Skip text elements

    # Accept any tag: pass its name as the first argument to parseValueList,
    # and the branch of the parsed XML structure corresponding to its values
    # as the second argument
    $self->_parseValueList($tags->[$index], $tags->[$index + 1]);
  }
  
  return;
}

sub write {
  my ($self, $handle) = @_;
  
  croak 'Handle argument must be given' unless defined $handle;
  
  # Now create and XML::Writer instance and turn that puppy out
  my $xmlHandle = XML::Writer->new(
    OUTPUT => $handle, ENCODING => 'utf-8',
    DATA_MODE => 1, DATA_INDENT => 2
  );

  # Open up the document
  $xmlHandle->xmlDecl;
  $xmlHandle->startTag('xorgconfig');
  $xmlHandle->comment('Generated by VASM');
  
  # Emit the top-level key tags and their values
  for my $key ($self->list) {
    # Open the tag up
    $xmlHandle->startTag($key);
    # Print out each value
    for my $value ($self->retrieve($key)) {
      $xmlHandle->dataElement(value => $value);
    }
    # Close the tag
    $xmlHandle->endTag($key);
  }

  $xmlHandle->endTag('xorgconfig'); $xmlHandle->end; # End the document...
}

1;

__END__

=head1 NAME

VASM::Catalog::XorgConfig

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

=head1 FORMAT

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com>: principal author
cintyram, YaP, vanger: design suggestions, technical assistance

=cut
