# -*- project-name: VASM -*-
package VASM::Catalog::Menu::Translation;

use strict;
use warnings;
use VASM::Resource;
use VASM::Resource::Catalog::Message;
use File::Spec::Functions qw/splitpath/;
use Image::Magick;
use XML::Writer;
use IO::File;
use Carp;

our $VERSION = '1.10';

# TODO:
#
# KDE (MergeFile)

my $menuMsg = messageCatalogFind(qw/menu translation/);
my $baseIndent = '  '; # Two space indent

sub generateMenuIcon {
  my %args = @_;

  # Check arguments hash
  croak 'Arguments newPath and oldPath must be given'
    unless (grep { defined $args{$_} } qw/newPath oldPath/) == 2;
  my @newPath = (qw/menu translation icon/, @{ $args{newPath} });

  # First, check if the icon already exists
  my $iconName = configResourceFind(@newPath);
  return $iconName if defined $iconName;
  
  # If it doesn't exist, first get the old icon
  my $oldIconHandle = IO::File->new($args{oldPath});
  croak $args{oldPath}, ": $!" unless $oldIconHandle;
  my $transitionIcon = Image::Magick->new;
  # Read the old icon into the Image::Magick object $oldIcon
  $transitionIcon->Read(file => $oldIconHandle);

  # Now call the callback subref given in $args{proc}, if any, with the
  # Image::Magick object as its argument
  $args{proc}->($transitionIcon) if defined $args{proc};

  # Then create a $newIconHandle to which to write out the new icon (from
  # $transitionIcon)
  my $newIconName = configResourceCreate(@newPath);

  # Write out $transitionIcon to $newIconHandle
  $transitionIcon->Write($newIconName);
  return $newIconName; # Then return it (the filename)
}

sub translate_fluxbox {
  my ($self, $menuFH) = @_;
  
  croak 'Handle argument must be given' unless defined $menuFH;

  # Begin the menu
  $menuFH->print("# Generated by VASM\n\n");
  $menuFH->printf("[submenu] (%s)\n", $menuMsg->render('menu title'));
  # Recursive method processes the menu itself
  $self->_translate_fluxbox($menuFH);
  # End the menu
  $menuFH->print("[end]\n");

  return;
}

sub _translate_fluxbox {
  my ($self, $menuFH, @path) = @_;
  # Indent comes in increments of two spaces (tabs are evil)
  my $indent = $baseIndent x (@path + 1);

  for my $child ($self->children(@path)) {
    my $attrs = $self->retrieve(@path, $child);

    # If an item
    if (defined $attrs->{path}) {
      # Generate the icon; the name of the new icon will end in .xpm
      my $iconName = (splitpath($attrs->{icon}))[2] . '.xpm';
      my $iconFile = generateMenuIcon(
        newPath => [ qw/fluxbox/, $iconName ],
        oldPath => $attrs->{icon},
        proc => sub { $_[0]->Resize(width => 16, height => 16);
                      $_[0]->SetAttribute(magick => 'xpm') });
      # Emit attributes
      $menuFH->print("${indent}[exec] " .
                       "($child) " .
                       "{$attrs->{path}} " .
                       "<$iconFile>\n");
    }
    # If a folder
    else {
      # Emit submenu opening
      $menuFH->print("${indent}[submenu] ($child)\n");
      # Recurse
      $self->_translate_fluxbox($menuFH, @path, $child);
      # Emit submenu end
      $menuFH->print("${indent}[end]\n");
    }
  }
  
  return;
}

sub translate_icewm {
  my ($self, $menuFH) = @_;

  croak 'Handle argument must be given' unless defined $menuFH;
  # Begin the menu
  $menuFH->print("# Generated by VASM\n\n");
  # Recursive method processes the menu itself
  $self->_translate_icewm($menuFH);

  return;
}

# This assumes IceWM was compiled with Imlib, which is hopefully a reasonable
# assumption
sub _translate_icewm {
  my ($self, $menuFH, @path) = @_;
  # Indent comes in increments of two spaces (tabs are evil)
  my $indent = @path ? $baseIndent x @path : '';

  for my $child ($self->children(@path)) {
    my $attrs = $self->retrieve(@path, $child);

    # Generate the icon
    my $iconName = (splitpath($attrs->{icon}))[2];
    # A no-op...just writes the icon to the resource directory
    my $iconFile = generateMenuIcon(
      newPath => [ qw/icewm/, $iconName ],
      oldPath => $attrs->{icon}
    );

    # If an item
    if (defined $attrs->{path}) {
      # Emit attributes -- config hierarchy must be in IconPath...
      $menuFH->print(
        "${indent}prog " .
        qq!"$child" ! .
        qq!"$iconFile" ! .
        "$attrs->{path}\n"
      );
    }
    # If a folder
    else {
      # Emit submenu opening
      $menuFH->print(qq!${indent}menu "$child" "$iconName" {\n!);
      # Recurse
      $self->_translate_icewm($menuFH, @path, $child);
      # Emit submenu end
      $menuFH->print("${indent}}\n");
    }
  }

  return;
}

sub translate_windowmaker {
  my ($self, $menuFH) = @_;

  croak 'Handle argument must be given' unless defined $menuFH;
  
  # Open up the menu
  # Escape double quotes in the menu title
  (my $menuTitle = $menuMsg->render('menu title')) =~ s!"!\"!g;
  $menuFH->print("// Generated by VASM\n\n");
  $menuFH->print(qq!#include "wmmacros"\n\n!);
  $menuFH->printf(qq!"$menuTitle" MENU\n!);

  # Recursive method processes the menu itself
  $self->_translate_windowmaker($menuFH);

  # Close the menu
  $menuFH->print(qq!"$menuTitle" END\n!);

  return;
}

sub _translate_windowmaker {
  my ($self, $menuFH, @path) = @_;

  # Indent comes in increments of two spaces (tabs are evil)
  my $indent = $baseIndent x (@path + 1);

  for my $child ($self->children(@path)) {
    my $attrs = $self->retrieve(@path, $child);

    # If an item
    if (defined $attrs->{path}) {
      (my $escapedChild = $child) =~ s!"!\"!g;
      (my $escapedPath = $attrs->{path}) =~ s!"!\"!g;
      $menuFH->print($indent . qq!"$escapedChild" EXEC $escapedPath\n!);
    }
    # If a folder
    else {
      # Emit submenu opening
      (my $escapedChild = $child) =~ s!"!\"!g;
      $menuFH->print($indent . qq!"$escapedChild" MENU\n!);
      $self->_translate_windowmaker($menuFH, @path, $child); # Recurse
      # Emit submenu end
      $menuFH->printf($indent . qq!"$escapedChild" END\n!);
    }
  }
  
  return;
}

sub translate_xfce4 {
  my ($self, $menuFH) = @_;

  croak 'Handle argument must be given' unless defined $menuFH;

  my $menuXML = XML::Writer->new(
    OUTPUT => $menuFH, ENCODING => 'utf-8',
    DATA_MODE => 1, DATA_INDENT => length($baseIndent)
  );
  
  # Open up the menu
  $menuXML->xmlDecl; $menuXML->doctype('xfdesktop-menu');
  $menuXML->comment('Generated by VASM');
  $menuXML->startTag('xfdesktop-menu');

  # Open submenu and add title
  $menuXML->startTag(
    'menu', 
    name => $menuMsg->render('menu title'),
    icon => 'xfce4-backdrop'
  );
  $menuXML->emptyTag(
    'title',
    name => $menuMsg->render('menu title'),
    icon => 'xfce4-backdrop'
  );
  $menuXML->emptyTag('separator');

  # Recursive method processes the menu itself
  $self->_translate_xfce4($menuFH, $menuXML);

  # Close the menu
  $menuXML->endTag('menu');
  $menuXML->endTag('xfdesktop-menu');
  $menuXML->end;
  
  return;
}

sub _translate_xfce4 {
  my ($self, $menuFH, $menuXML, @path) = @_;

  for my $child ($self->children(@path)) {
    my $attrs = $self->retrieve(@path, $child);

    # Generate the icon
    my $iconName = (splitpath($attrs->{icon}))[2];
    # A no-op...just writes the icon to the resource directory
    my $iconFile = generateMenuIcon(
      newPath => [ qw/xfce4/, $iconName ],
      oldPath => $attrs->{icon}
    );
    # If an item
    if (defined $attrs->{path}) {
      # Add the tag
      $menuXML->emptyTag(
        'app', 
        name => $child, 
        cmd => $attrs->{path}, 
        # Maybe a way to include this in icon search path
        # instead?
        icon => $iconFile
      );
    }
    # If a folder
    else {
      # Emit submenu opening
      $menuXML->startTag('menu', name => $child, icon => $iconFile);
      # Recurse
      $self->_translate_xfce4($menuFH, $menuXML, @path, $child);
      # Emit submenu end
      $menuXML->endTag('menu');
    }
  }
  
  return;
}

1;

__END__

=head1 NAME

VASM::Catalog::Menu::Translation - translator methods for menu catalogs

=head1 SYNOPSIS

use VASM::Catalog::Menu;

# Parse menu catalog here
# Write out translated catalog

=head1 DESCRIPTION

VASM::Catalog::Menu::Translation is a superclass for VASM::Catalog::Menu, and
contains a set of methods for the rendition of VASM menu catalogs into useful
window manager-specific files. As a matter of hackneyed duck-typing
convention, each of these methods accepts a single IO::Handle argument and
has a name that begins with 'Translate'. In cases where the destination menu
format supports icons, VASM::Catalog::Menu::Translation furnishes a general
utility function, generateMenuIcon, which copies icons to a central repository
(under the path qw/menu translation icon/ in the user-specific VASM
configuration resource hierarchy) and allows diverse manipulations with the
Image::Magick Perl interface to conform to the restrictions imposed by any
given window manager; Fluxbox, for example, allows only 16x16 XPM files.

=head1 METHODS

=over

=item *

translate_fluxbox

=item *

translate_icewm

=item *

translate_windowmaker

=item *

translate_xfce4

=back

=head1 FUNCTIONS

=over

=item generateMenuIcon

Give a good description of generateMenuIcon here!

=back

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote
VASM::Catalog::Menu::Translators. Further considerations by cintyram.

=cut
